import { TranslateService } from '@ngx-translate/core';
import { BehaviorSubject, Observable } from 'rxjs';
import { FilterExpressionToTranslate, FilterOperationByTypeToTranslate } from '../../models/query-table-filter.model';
import { Query } from '../../models/query.model';
/**
 * AbstractQueriesService class that defines an
 * <br />abstract queries service to
 * <br />be extended by the mock and real
 * <br />queries services
 */
export declare abstract class AbstractQueriesService {
    private translationService;
    /**
     * Current Queries list
     */
    protected queriesChangedSubject: BehaviorSubject<Query[]>;
    /**
     * Constructor
     * @param {TranslateService} translationService
     * @protected
     */
    protected constructor(translationService: TranslateService);
    /**
     * Get all Queries
     * @returns a Promise type Query[]
     */
    abstract getAllQueriesAsync(): Promise<Query[]>;
    /**
     * Function responsible to update a query
     * @param {string} queryId
     * @param {Query} queryUpdated
     * @returns {Promise<Query>}
     */
    abstract updateQueryAsync(queryId: string, queryUpdated: Query): Promise<Query>;
    /**
     * Getter for queriesChangedSubject
     * @returns {Observable<Query[]>} the queriesChangedSubject as an observable
     */
    getQueriesChangedObservable(): Observable<Query[]>;
    /**
     * Getter for queriesChangedSubject
     * @returns {Query[]} the queriesChangedSubject current value
     */
    getQueriesChanged(): Query[];
    /**
     * A setter queriesChangedSubject
     * @param {Query[]} queries
     * @protected
     */
    protected setQueriesChanged(queries: Query[]): void;
    /**
     * Function responsible to store a new query
     * @param {Query} newQuery
     * @returns {Promise<Query>}
     */
    abstract createQueryAsync(newQuery: Query): Promise<Query>;
    /**
     * Function responsible to translate Filters Expressions
     * @param {string[]} filterExpression
     * @returns {Promise<FilterExpressionToTranslate[]>}
     */
    getFilterExpressionTranslationsAsync(filterExpression: string[]): Promise<FilterExpressionToTranslate[]>;
    /**
     * Function responsible to translate Filters Operators
     * @param {string[]} filterOperators
     * @returns {Promise<FilterOperationByTypeToTranslate[]>}
     */
    getFilterOperationsTranslationsAsync(filterOperators: string[]): Promise<FilterOperationByTypeToTranslate[]>;
    /**
     * Get query by Id
     * @param {string} queryId
     * @returns {Promise<Query>}
     */
    abstract getQueryByIdAsync(queryId: string): Promise<Query>;
}
