import { BehaviorSubject, Observable } from 'rxjs';
import { AuthStorage } from '../../models/prg-auth-config';
import { User } from '../../models/user.model';
/**
 * AbstractAuthService class that defines an
 * <br />abstract authentication service to
 * <br />be extended by the mock and real
 * <br />authentication services
 */
export declare abstract class AbstractAuthService {
    private authStorage;
    /**
     * Currently logged user
     */
    protected loggedUserSubject: BehaviorSubject<User>;
    /**
     * Currently stored username to be used
     * <br />on the login form
     * // CR: devemos explicar melhor para que serve esta propriedade, eu não estava a perceber até olhar para o código do login,
     *        talvez mudar também o nome para algo mais explicito
     */
    protected storedUserNameSubject: BehaviorSubject<string>;
    /**
     * constructor
     * @param authStorage
     * @protected
     */
    protected constructor(authStorage: AuthStorage);
    /**
     * User sign up (Async)
     *
     * @param {string} email The user e-mail
     * @returns Boolean promise, true when successful
     */
    abstract signupAsync(email: string): Promise<boolean>;
    /**
     * User login (Async)
     *
     * @param {string} email The user e-mail
     * @param {string} password The user password
     * @param {boolean} storeUserName Store username
     * @returns Boolean promise, true when successful
     */
    abstract loginAsync(email: string, // CR: se pode ser email ou username, mudar parametro para emailOrUsername
    password: string, storeUserName: boolean): Promise<boolean>;
    /**
     * User logout (Async)
     *
     * @returns Boolean promise, true when successful
     */
    abstract logoutAsync(): Promise<boolean>;
    /**
     * Request to reset the password (Async)
     *
     * @param {string} email The user e-mail
     * @returns Boolean promise, true when successful
     */
    abstract requestResetPasswordAsync(email: string): Promise<boolean>;
    /**
     * Getter for token reset password (Async)
     *
     * @param {string} tokenReset The token reset password
     * @returns Boolean promise, true when successful
     */
    abstract getTokenResetPasswordAsync(tokenReset: string): Promise<boolean>;
    /**
     * Method for reset password (Async)
     *
     * @param {string} newPassword The new password
     * @param {string} tokenReset The token reset password
     * @returns Boolean promise, true when successful
     */
    abstract resetPasswordAsync(newPassword: string, tokenReset: string): Promise<boolean>;
    /**
     * Setter for loggedUserSubject
     *
     * @param {User} userLogged The User object
     */
    protected setLoggedUser(userLogged: User): void;
    /**
     * Setter for storedUserNameSubject and saving username on local storage
     *
     * @param {string} value The storedUserName string
     */
    protected setStoredUserName(value: string): void;
    /**
     * Remove username from local storage
     *
     */
    protected removeStoredUserName(): void;
    /**
     * Getter for loggedUserSubject
     *
     * @returns The loggedUserSubject as an observable
     */
    getLoggedUserObservable(): Observable<User>;
    /**
     * Getter for loggedUserSubject
     *
     * @returns The loggedUserSubject current value
     */
    getLoggedUser(): User;
    /**
     * Getter for storedUserNameSubject
     *
     * @returns The storedUserNameSubject as an observable
     */
    getStoredUserNameObservable(): Observable<string>;
    /**
     * Getter for storedUserNameSubject
     *
     * @returns The storedUserNameSubject current value
     */
    getStoredUserName(): string;
    /**
     * Get username from local storage and if it exists,
     *  the value is emitted on storedUserNameSubject
     */
    private getUserNameFromStorage;
    /**
     * Get the user logged from local storage
     * @private
     */
    private getUserLoggedFromStorage;
    /**
     * User logout Base Method (Async)
     *
     * @returns Boolean promise, true when successful
     */
    protected baseLogoutAsync(): Promise<boolean>;
}
