import { BehaviorSubject } from 'rxjs';
/**
 * AbstractQueriesService class that defines an
 * <br />abstract queries service to
 * <br />be extended by the mock and real
 * <br />queries services
 */
export class AbstractQueriesService {
    /**
     * Constructor
     * @param {TranslateService} translationService
     * @protected
     */
    constructor(translationService) {
        this.translationService = translationService;
        /**
         * Current Queries list
         */
        this.queriesChangedSubject = new BehaviorSubject(null);
    }
    /**
     * Getter for queriesChangedSubject
     * @returns {Observable<Query[]>} the queriesChangedSubject as an observable
     */
    getQueriesChangedObservable() {
        return this.queriesChangedSubject.asObservable();
    }
    /**
     * Getter for queriesChangedSubject
     * @returns {Query[]} the queriesChangedSubject current value
     */
    getQueriesChanged() {
        return this.queriesChangedSubject.getValue();
    }
    /**
     * A setter queriesChangedSubject
     * @param {Query[]} queries
     * @protected
     */
    setQueriesChanged(queries) {
        this.queriesChangedSubject.next(queries);
    }
    /**
     * Function responsible to translate Filters Expressions
     * @param {string[]} filterExpression
     * @returns {Promise<FilterExpressionToTranslate[]>}
     */
    async getFilterExpressionTranslationsAsync(filterExpression) {
        return new Promise(async (resolve, reject) => {
            let filterExpressionToTranslate = [];
            filterExpression.map(async (filter) => {
                filterExpressionToTranslate.push({
                    expression: await filter,
                    label: await this.translationService
                        .get('filters.filter-expression.' + filter + '.label')
                        .toPromise(),
                });
            });
            resolve(filterExpressionToTranslate);
        });
    }
    /**
     * Function responsible to translate Filters Operators
     * @param {string[]} filterOperators
     * @returns {Promise<FilterOperationByTypeToTranslate[]>}
     */
    async getFilterOperationsTranslationsAsync(filterOperators) {
        return new Promise(async (resolve, reject) => {
            let filterExpressionToTranslate = [];
            filterOperators.map(async (filter) => {
                filterExpressionToTranslate.push({
                    operators: await filter,
                    label: await this.translationService
                        .get('filters.filter-operators.' + filter + '.label')
                        .toPromise(),
                });
            });
            resolve(filterExpressionToTranslate);
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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