import { Component } from '@angular/core';
import { FormArray, FormGroup, Validators } from '@angular/forms';
import { BaseActionKey, } from '../../../core/components/dynamic-form/models/base-action';
import { ViewMode } from '../../../core/models/constants/view-mode.enum';
import { PrgError } from '../../../core/models/error.model';
import { QueriesFilterColumns, } from '../../models/query-table-filter.model';
import { Query, QueryFilterExpression, QueryFilterOperationByType, } from '../../models/query.model';
import { QUERIES_FILTER_DYNAMIC_FORM } from '../../queries-dynamic-form-structure/queries-filter-dynamic-form';
import { QUERIES_RAW_DYNAMIC_FORM } from '../../queries-dynamic-form-structure/queries-raw-dynamic-form';
import * as i0 from "@angular/core";
import * as i1 from "primeng/dynamicdialog";
import * as i2 from "../../services/queries/abstract-queries.service";
import * as i3 from "../../../core/services/utility/objects-utility.service";
import * as i4 from "../../../core/services/utility/array-utility.service";
import * as i5 from "../../../core/services/form-group/form-group.service";
import * as i6 from "../../../core/services/notifications/notifications.service";
import * as i7 from "primeng/api";
import * as i8 from "../../../entity-types/services/entity-types/abstract-entity-type.service";
import * as i9 from "../../../lookup-table/services/lookup-table/abstract-lookup-table.service";
import * as i10 from "primeng/inputtext";
import * as i11 from "@angular/common";
import * as i12 from "primeng/table";
import * as i13 from "primeng/button";
import * as i14 from "primeng/dropdown";
import * as i15 from "primeng/checkbox";
import * as i16 from "primeng/ripple";
import * as i17 from "@angular/forms";
import * as i18 from "primeng/togglebutton";
import * as i19 from "../../../core/components/dynamic-form/prg-dynamic-form.component";
import * as i20 from "@ngx-translate/core";
/**
 * Query Modal Component
 */
export class PrgQueryModalComponent {
    /**
     * Constructor
     * @param {DynamicDialogRef} querySelectedDialogRef querySelectedDialogRef
     * @param {DynamicDialogConfig} querySelectedDialogConfig querySelectedDialogConfig
     * @param {AbstractQueriesService} queriesService
     * @param {ObjectsUtilityService} objectsUtilityService
     * @param {ArrayUtilityService} arrayUtilityService
     * @param {FormGroupService} formGroupService
     * @param {NotificationsService} notificationsService
     * @param {PrimeNGConfig} primengConfig
     * @param {AbstractEntityTypeService} entityTypeService
     * @param {AbstractLookupTableService} lookupTableService
     */
    constructor(querySelectedDialogRef, querySelectedDialogConfig, queriesService, objectsUtilityService, arrayUtilityService, formGroupService, notificationsService, primengConfig, entityTypeService, lookupTableService) {
        this.querySelectedDialogRef = querySelectedDialogRef;
        this.querySelectedDialogConfig = querySelectedDialogConfig;
        this.queriesService = queriesService;
        this.objectsUtilityService = objectsUtilityService;
        this.arrayUtilityService = arrayUtilityService;
        this.formGroupService = formGroupService;
        this.notificationsService = notificationsService;
        this.primengConfig = primengConfig;
        this.entityTypeService = entityTypeService;
        this.lookupTableService = lookupTableService;
        /**
         * Map of Operators and datatype associated to field
         * @type {Map<string, OperatorsAndDataType>}
         */
        this.mappedOperatorsAndDataTypeByField = new Map();
        /**
         * A copy of mappedOperatorsAndDataTypeByField
         * @type {Map<string, OperatorsAndDataType>}
         */
        this.mappedOperatorsAndDataTypeByFieldCopy = new Map();
        /**
         * Display column "Value2" when a query has a filter with two inputs values.
         *
         * Default false, hidden column.
         */
        this.queryHasFilterWithTwoValues = false;
        /**
         * The filter expressions ex:"AND" and its translations
         * @type {FilterExpressionToTranslate[]}
         */
        this.filterExpressions = [];
        /**
         * Create an object type QueryFilterOperationByType to
         * access his method numberOfFields on template.
         */
        this.operatorType = new QueryFilterOperationByType();
        /**
         * A form array that belongs to the formQueryFilter
         */
        this.queryFilterArrayForm = new FormArray([]);
        /**
         * Property  with type boolean which bind to the checkbox in template to change query type.
         *
         * Default is true (raw query)
         */
        this.checkBoxQueryRaw = true;
        /**
         * The properties and attributes available for selection by entity type
         */
        this.optionsQueryFields = [];
        /**
         * A copy of OptionsQueryFields
         */
        this.optionsQueryFieldsCopy = [];
        /**
         * The dynamic form fields to build dynamic form filter query in template
         */
        this.queriesFilterDynamicFormFields = this.arrayUtilityService.clone(QUERIES_FILTER_DYNAMIC_FORM.fields);
        /**
         * The dynamic form actions to build dynamic form filter query in template
         */
        this.queriesFilterDynamicFormActions = this.arrayUtilityService.clone(QUERIES_FILTER_DYNAMIC_FORM.actions);
        /**
         * The dynamic form fields to build dynamic form raw query in template
         */
        this.queriesRawDynamicFormFields = this.arrayUtilityService.clone(QUERIES_RAW_DYNAMIC_FORM.fields);
        /**
         * The dynamic form actions to build dynamic form raw query in template
         */
        this.queriesRawDynamicFormActions = this.arrayUtilityService.clone(QUERIES_RAW_DYNAMIC_FORM.actions);
        /**
         * Query table columns to build the formQueryFilter
         */
        this.queriesFilterColumns = QueriesFilterColumns;
        /**
         * This property define if component is loading data from an existing query.
         *
         * Default is true.
         * @type {boolean}
         */
        this.isLoading = true;
        /**
         * The view mode type that comes from parent component. Default is "read"
         */
        this.queriesViewMode = ViewMode.Read;
        /**
         * Property that defines if you are editing a query or creating a new one.
         *
         * Default is false (new query)
         */
        this.editMode = false;
    }
    /**
     * ngOnInit
     *
     * Set up all configurations require to initiate properly the component
     */
    async ngOnInit() {
        let dataQueryFromTable = this.querySelectedDialogConfig.data.query;
        this.queriesViewMode = this.querySelectedDialogConfig.data.viewMode;
        this.checkBoxQueryRaw = dataQueryFromTable.isRaw;
        if (dataQueryFromTable.id) {
            this.query = this.objectsUtilityService.cloneObject(await this.queriesService.getQueryByIdAsync(dataQueryFromTable.id));
            this.editMode = true;
            this.currentEntityTypeIdSelected = this.query.entityTypeId;
        }
        else {
            this.query = this.objectsUtilityService.cloneObject(dataQueryFromTable);
            this.editMode = false;
        }
        this.originalQuery =
            this.objectsUtilityService.cloneObject(dataQueryFromTable);
        this.primengConfig.ripple = true;
        this.initFormQueryFilter();
        if (this.queriesViewMode === ViewMode.Read) {
            this.queryFilterArrayForm.disable();
        }
        this.onCheckNumberOfFields();
        await this.getInitialData();
        this.filterExpressions =
            await this.queriesService.getFilterExpressionTranslationsAsync(Object.keys(QueryFilterExpression));
        this.isLoading = false;
    }
    /**
     * Get initial data (async)
     *
     * Function responsible for getting all data from api/mock required to initialize form query filter
     * @private
     */
    async getInitialData() {
        this.entityTypeList = this.arrayUtilityService.clone(await this.entityTypeService.getEntityTypeListAsync());
        if (this.currentEntityTypeIdSelected && !this.checkBoxQueryRaw) {
            await this.getFieldsByEntityTypeSelect();
            await this.mapOperatorsAndDataTypeByField();
        }
    }
    /**
     * Get fields by entity type selected(async)
     *
     * Function responsible for getting all fields (properties and attributes) available
     * from api/mock by entity type selected
     * @private
     */
    async getFieldsByEntityTypeSelect() {
        /*Check if entity selected has properties and attributes*/
        try {
            if (!this.checkBoxQueryRaw && this.currentEntityTypeIdSelected) {
                let entityTypeNameSelected = this.entityTypeList.find((entity) => entity.id === this.currentEntityTypeIdSelected).name;
                let entityAllData = this.objectsUtilityService.cloneObject(await this.entityTypeService.getAllEntityTypeDataByName(entityTypeNameSelected));
                this.optionsQueryFields = this.arrayUtilityService.clone([
                    ...entityAllData.properties,
                    ...entityAllData.attributes,
                ]);
                if (this.optionsQueryFieldsCopy == null ||
                    this.optionsQueryFieldsCopy.length < 1) {
                    this.optionsQueryFieldsCopy = this.arrayUtilityService.clone(this.optionsQueryFields);
                }
            }
        }
        catch (e) {
            this.optionsQueryFields = [];
        }
    }
    /**
     * Query raw action output (async)
     *
     * This function is responsible to handle actions from the dynamic raw form
     * @param {DynamicFormActionOutput} event the output from the dynamic raw form
     */
    async onActionQueryRaw(event) {
        switch (event.action) {
            case BaseActionKey.Save:
                await this.queryRawSave(event);
                break;
            case BaseActionKey.Cancel:
                if (event.formEntity.id) {
                    this.query = this.objectsUtilityService.cloneObject(this.originalQuery);
                }
                else {
                    this.querySelectedDialogRef.close();
                }
                this.queriesViewMode = ViewMode.Read;
                break;
            case BaseActionKey.Edit:
                this.queriesViewMode = ViewMode.Edit;
                break;
            default:
                break;
        }
    }
    /**
     * Save raw query (async)
     *
     * This function is responsible to save a new raw query or update an existing one
     * @param {DynamicFormActionOutput} event the output from the dynamic raw form
     * @private
     */
    async queryRawSave(event) {
        this.queriesViewMode = ViewMode.Read;
        // @ts-ignore
        const { group, ...query } = event.formEntity;
        query.isRaw = true;
        let queryResolve;
        if (query.id == null || query.id.length === 0) {
            queryResolve = await this.queriesService.createQueryAsync(query);
        }
        else {
            delete query.filters;
            delete query.entityTypeId;
            queryResolve = await this.queriesService.updateQueryAsync(query.id, query);
        }
        this.querySelectedDialogRef.close(queryResolve);
    }
    /**
     * Initialize Form Array of Filter Query
     * @param {boolean} editMode
     * @private
     */
    initFormArray(editMode = false) {
        if (this.query['filters'] && !editMode) {
            const aux = this.formGroupService.toFormGroupMultipleObjects(this.query.filters, this.queriesFilterColumns);
            this.query.filters.forEach((value, index) => {
                this.queryFilterArrayForm.push(aux[index]);
            });
        }
        else {
            const newQueryFilter = PrgQueryModalComponent.newQueryFilterObject();
            this.queryFilterArrayForm.controls = [];
            const aux2 = this.formGroupService.toFormGroupOneObject(newQueryFilter, this.queriesFilterColumns);
            this.queryFilterArrayForm.push(aux2);
        }
    }
    /**
     * Initialize Query Filter Form
     * @private
     */
    initFormQueryFilter() {
        this.initFormArray();
        this.formQueryFilter = new FormGroup({
            filters: this.queryFilterArrayForm,
        });
    }
    /**
     * A getter for the form array controls
     * @returns {any}
     */
    get filters() {
        return this.formQueryFilter.get('filters').controls;
    }
    /**
     * Delete a row of the filter query table  (async)
     * @param {number} i The table row
     *
     */
    onDeleteRow(i) {
        if (this.queryFilterArrayForm.length > 1) {
            this.formQueryFilter.get('filters').removeAt(i);
        }
        else {
            this.formQueryFilter.get('filters').reset();
        }
    }
    /**
     * Changing field (async)
     *
     * Function is call whenever is a change of field value
     * @param {number} i The table row
     * @param {string}value the name of the field
     */
    async onChangeField(i, value) {
        this.formQueryFilter.get('filters').controls.forEach((arrayRow, index) => {
            if (index === i) {
                arrayRow.controls.filterOperation.reset();
            }
        });
    }
    /**
     * Add a new element to query filter array (async)
     * @param {number} i The table row
     */
    async onAddRow(i) {
        const newQueryFilter = PrgQueryModalComponent.newQueryFilterObject();
        this.queryFilterArrayForm.insert(i + 1, this.formGroupService.toFormGroupOneObject(newQueryFilter, this.queriesFilterColumns));
    }
    /**
     * Function is call whenever entity type value changes
     * @param {string} entityID
     *
     */
    async onChangeEntity(entityID) {
        if (entityID == null) {
            return;
        }
        else {
            this.currentEntityTypeIdSelected = entityID;
            await this.getFieldsByEntityTypeSelect();
            await this.mapOperatorsAndDataTypeByField();
            this.initFormArray(this.editMode);
        }
    }
    /**
     * Changing Query type
     *
     * Function is call whenever toggle button changes
     */
    async onChangeQueryType() {
        if (!this.checkBoxQueryRaw && !this.query.id) {
            await this.getFieldsByEntityTypeSelect();
            await this.mapOperatorsAndDataTypeByField();
            this.initFormArray();
        }
    }
    /**
     * Auxiliary function to create a new instance of class QueryFilter
     * with all its properties
     * @returns {QueryFilter} new Object QueryFilter
     * @private
     */
    static newQueryFilterObject() {
        return {
            id: null,
            universalStateId: null,
            transactionId: null,
            operationId: null,
            name: null,
            workspaceId: null,
            filterExpression: QueryFilterExpression.And,
            filterOperation: null,
            queryId: null,
            propertyName: null,
            startGroup: true,
            value: null,
            value2: null,
            createdBy: null,
            createdOn: null,
            modifiedBy: null,
            modifiedOn: null,
        };
    }
    /**
     * Function is call whenever operator input changes in each row
     *
     * Function is responsible to find out if any of filter rows has an
     * operator that requires two inputs values an if so makes field value2 visible and apply required validator
     */
    onCheckNumberOfFields() {
        if (this.queryFilterArrayForm != null &&
            this.queryFilterArrayForm.length > 0) {
            const allOperations = this.queryFilterArrayForm.controls.map((a) => {
                return a.value.filterOperation;
            });
            let checkMultipleFields = false;
            for (let operation of allOperations) {
                let index = allOperations.indexOf(operation);
                let operatorType = new QueryFilterOperationByType();
                if (operatorType.numberOfFields(operation) === 2) {
                    checkMultipleFields = true;
                    this.queryFilterArrayForm.controls[index]
                        .get('value2')
                        .setValidators(Validators.required);
                    break;
                }
            }
            this.queryHasFilterWithTwoValues = checkMultipleFields;
        }
    }
    /**
     * Query filter action output (async)
     *
     * This function is responsible to handle actions from the dynamic filter form
     * @param {DynamicFormActionOutput} event the output from the dynamic filter form
     */
    async onActionQueryFilter(event) {
        switch (event.action) {
            case BaseActionKey.Save:
                if (this.queryFilterArrayForm.status === 'INVALID') {
                    this.notificationsService.errorNotification(new PrgError({
                        titleKey: 'Error Submitting Query ',
                        detailKey: 'Filling missing fields',
                    }));
                    return;
                }
                await this.queryFilterSave(event);
                break;
            case BaseActionKey.Cancel:
                await this.queryFilterCancel(event);
                break;
            case BaseActionKey.Edit:
                this.queriesViewMode = ViewMode.Edit;
                this.queryFilterArrayForm.enable();
                break;
            default:
                break;
        }
    }
    /**
     * Save filter query (async)
     *
     * This function is responsible to save a new filter query or update an existing one
     * @param {DynamicFormActionOutput} event the output from the dynamic filter form
     * @private
     */
    async queryFilterSave(event) {
        this.queriesViewMode = ViewMode.Read;
        const newQuery = new Query({
            isRaw: false,
            id: this.query.id ? this.query.id : null,
            createdBy: this.query.createdBy ? this.query.createdBy : null,
            name: event.formEntity.name,
            description: event.formEntity.description,
            entityTypeId: event.formEntity.entityTypeId,
            filters: this.formQueryFilter.value.filters,
        });
        let queryResolve;
        if (newQuery.id == null || newQuery.id.length === 0) {
            queryResolve = await this.queriesService.createQueryAsync(newQuery);
        }
        else {
            queryResolve = await this.queriesService.updateQueryAsync(newQuery.id, newQuery);
        }
        this.querySelectedDialogRef.close(queryResolve);
    }
    /**
     * Cancel filter query (async)
     *
     * This function is responsible to handle operations whenever cancel button is pressed
     *
     * If editing an existing query it will reset all values of the form
     *
     * If is a new one will close dialog
     * @param {DynamicFormActionOutput} event the output from the dynamic filter form
     * @private
     */
    async queryFilterCancel(event) {
        if (event.formEntity.id) {
            this.formQueryFilter.reset();
            this.isLoading = true;
            this.currentEntityTypeIdSelected = this.query.entityTypeId;
            this.optionsQueryFields = this.arrayUtilityService.clone(this.optionsQueryFieldsCopy);
            this.mappedOperatorsAndDataTypeByField.clear();
            this.mappedOperatorsAndDataTypeByField =
                this.objectsUtilityService.cloneObject(this.mappedOperatorsAndDataTypeByFieldCopy);
            this.queryFilterArrayForm.controls = [];
            this.initFormArray();
            this.query = this.objectsUtilityService.cloneObject(this.originalQuery);
            this.isLoading = false;
        }
        else {
            this.querySelectedDialogRef.close();
        }
        this.queriesViewMode = ViewMode.Read;
        this.queryFilterArrayForm.disable();
    }
    /**
     * This Function is responsible to map datatype for the fields values and operators
     *
     * associated to the chosen property or attribute for each filter row
     * @returns {Promise<void>}
     */
    async mapOperatorsAndDataTypeByField() {
        if (this.optionsQueryFields != null && this.optionsQueryFields.length > 0) {
            this.mappedOperatorsAndDataTypeByField.clear();
            let initMappedCopy = false;
            if (this.mappedOperatorsAndDataTypeByFieldCopy == null ||
                this.mappedOperatorsAndDataTypeByFieldCopy.size < 1) {
                initMappedCopy = true;
            }
            for (const fields of this.optionsQueryFields) {
                let dataTypeAux = this.lookupTableService
                    .getLookUpTableItemByIdAsync(fields.dataTypeId)
                    .then((value) => {
                    return value.name;
                });
                let auxOperation = QueryFilterOperationByType[await dataTypeAux];
                let queryFilterOperationAux = await this.queriesService.getFilterOperationsTranslationsAsync(auxOperation);
                Promise.all([dataTypeAux, queryFilterOperationAux]).then((values) => {
                    this.mappedOperatorsAndDataTypeByField.set(fields.name, {
                        queryFilterOperationByType: values[1],
                        dataType: values[0],
                    });
                    if (initMappedCopy) {
                        this.mappedOperatorsAndDataTypeByFieldCopy.set(fields.name, {
                            queryFilterOperationByType: values[1],
                            dataType: values[0],
                        });
                    }
                });
            }
        }
    }
}
PrgQueryModalComponent.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "14.2.0", ngImport: i0, type: PrgQueryModalComponent, deps: [{ token: i1.DynamicDialogRef }, { token: i1.DynamicDialogConfig }, { token: i2.AbstractQueriesService }, { token: i3.ObjectsUtilityService }, { token: i4.ArrayUtilityService }, { token: i5.FormGroupService }, { token: i6.NotificationsService }, { token: i7.PrimeNGConfig }, { token: i8.AbstractEntityTypeService }, { token: i9.AbstractLookupTableService }], target: i0.ɵɵFactoryTarget.Component });
PrgQueryModalComponent.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "14.0.0", version: "14.2.0", type: PrgQueryModalComponent, selector: "prg-select-query", ngImport: i0, template: "<div *ngIf=\"!isLoading\">\r\n<div class=\"my-4\">\r\n  <p-toggleButton (onChange)=\"onChangeQueryType()\" onLabel=\"{{'queries.query-modal.actions.query-type.filter-query.label' |translate}}\" offLabel=\"{{'queries.query-modal.actions.query-type.raw-query.label' |translate}}\"\r\n                  onIcon=\"pi pi-filter-fill\" offIcon=\"pi pi-file\" iconPos=\"right\"\r\n                  [(ngModel)]=\"checkBoxQueryRaw\"></p-toggleButton>\r\n\r\n</div>\r\n<ng-template [ngIf]=\"checkBoxQueryRaw\" [ngIfElse]=\"noRawQuery\">\r\n\r\n\r\n  <div class=\"mt-3\">\r\n    <prg-dynamic-form [viewMode]=\"queriesViewMode\" [actions]=\"queriesRawDynamicFormActions\"\r\n                      [fields]=\"queriesRawDynamicFormFields\"\r\n                      [entity]=\"query\" (actionOutput)=\"onActionQueryRaw($event)\">\r\n    </prg-dynamic-form>\r\n  </div>\r\n\r\n</ng-template>\r\n\r\n\r\n<ng-template #noRawQuery >\r\n\r\n  <prg-dynamic-form (fieldChangeOutput)=\"onChangeEntity($event.value)\" #PrgDynamicFormComponent\r\n                    (actionOutput)=\"onActionQueryFilter($event)\" [viewMode]=\"queriesViewMode\"\r\n                    [fields]=\"queriesFilterDynamicFormFields\" [actions]=\"queriesFilterDynamicFormActions\"\r\n                    [entity]=\"query\">\r\n\r\n  </prg-dynamic-form >\r\n\r\n\r\n\r\n\r\n  <form [formGroup]=\"formQueryFilter\" *ngIf=\"!isLoading\">\r\n\r\n    <div class=\"mt-2\">\r\n      <ng-template [ngIf]=\"PrgDynamicFormComponent.form.value['entityTypeId']\">\r\n        <p-table styleClass=\"p-datatable-gridlines\" [value]=\"filters\" dataKey=\"value\" formArrayName=\"filters\">\r\n          <ng-template pTemplate=\"header\">\r\n            <tr >\r\n              <ng-container *ngFor=\"let col of queriesFilterColumns\">\r\n              <th style=\"width: 10rem;text-align: center\" *ngIf=\"col.key==='startGroup'||col.key==='filterExpression'; else otherFields \" [hidden]=\"queryFilterArrayForm.length<=1\">{{col.basePathTranslation+col.key+'.label'|translate}}</th>\r\n              <ng-template #otherFields>\r\n                <th style=\"width: 10rem;text-align: center\" *ngIf=\"col.key!=='value2'&&col.key!=='query-filter-row-actions'\">{{col.basePathTranslation+col.key+'.label'|translate}}</th>\r\n                <th style=\"width: 10rem;text-align: center\" *ngIf=\"col.key==='query-filter-row-actions'\">{{col.basePathTranslation+col.key+'.label'|translate}}</th>\r\n                <th style=\"width: 10rem;text-align: center\" *ngIf=\"col.key==='value2'&&queryHasFilterWithTwoValues\">{{col.basePathTranslation+col.key+'.label'|translate}}</th>\r\n              </ng-template>\r\n              </ng-container>\r\n            </tr>\r\n          </ng-template>\r\n\r\n          <ng-template pTemplate=\"body\" let-i=\"rowIndex\">\r\n\r\n            <tr [formGroupName]=\"i\">\r\n\r\n              <td style=\"width: 5rem;text-align: center\" [hidden]=\"queryFilterArrayForm.length<=1\">\r\n\r\n                <p-checkbox formControlName=\"startGroup\" [binary]=\"true\"></p-checkbox>\r\n              </td>\r\n\r\n              <td style=\"width: 10rem;text-align: center\">\r\n\r\n\r\n                <p-dropdown [hidden]=\"PrgDynamicFormComponent.form.value['entityTypeId']==null\"\r\n                            (onChange)=\"onChangeField(i,$event.value)\" formControlName=\"propertyName\"\r\n                            [options]=\"optionsQueryFields\" optionValue=\"name\" optionLabel=\"label\" [filter]=\"true\"\r\n                            [showClear]=\"true\" placeholder=\"{{'queries.fields.query-filter.fields.propertyName.placeholder'|translate}}\" #fieldInput>\r\n\r\n                  <!--optionsQueryFields-->\r\n\r\n                </p-dropdown>\r\n              </td>\r\n              <td style=\"width: 10rem;text-align: center\">\r\n\r\n                <p-dropdown [hidden]=\"fieldInput.value==null\" formControlName=\"filterOperation\" optionLabel=\"label\" optionValue=\"operators\"\r\n                            (onChange)=\"onCheckNumberOfFields()\"\r\n                            [options]=\"mappedOperatorsAndDataTypeByField?.get(fieldInput.value)?.queryFilterOperationByType\" [filter]=\"true\"\r\n                            [showClear]=\"true\" placeholder=\"{{'queries.fields.query-filter.fields.filterOperation.placeholder'|translate}}\" #operatorInput>\r\n\r\n\r\n                </p-dropdown>\r\n\r\n              </td>\r\n\r\n\r\n              <td style=\"width: 10rem;text-align: center\"><input *ngIf=\"operatorInput.value!=null\"\r\n                                                                 formControlName=\"value\" id=\"value\"\r\n                                                                 [type]=\"mappedOperatorsAndDataTypeByField?.get(fieldInput.value)?.dataType==='number'?'number':mappedOperatorsAndDataTypeByField?.get(fieldInput.value)?.dataType==='date'?'date':'text'\"\r\n                                                                 placeholder=\"{{'queries.fields.query-filter.fields.value.placeholder'|translate}}\" pInputText\r\n\r\n\r\n              >\r\n              </td>\r\n\r\n              <td *ngIf=\"queryHasFilterWithTwoValues\" style=\"width: 10rem;text-align: center\"><input\r\n                *ngIf=\"operatorType.numberOfFields(operatorInput.value)==2\"\r\n                formControlName=\"value2\" id=\"value2\"\r\n                [type]=\"mappedOperatorsAndDataTypeByField?.get(fieldInput.value)?.dataType==='number'?'number':mappedOperatorsAndDataTypeByField?.get(fieldInput.value)?.dataType==='date'?'date':'text'\"\r\n                placeholder=\"{{'queries.fields.query-filter.fields.value2.placeholder'|translate}}\"\r\n                pInputText>\r\n              </td>\r\n\r\n              <td style=\"width: 10rem;text-align: center\">\r\n\r\n                <button [disabled]=\"this.queryFilterArrayForm.disabled\" pButton pRipple type=\"button\"  icon=\"pi pi-plus-circle\"\r\n                        (click)=\"onAddRow(i)\"\r\n                  class=\"p-button-rounded p-button-text p-button-success p-mr-2\"\r\n                ></button>\r\n\r\n                <button [disabled]=\"this.queryFilterArrayForm.disabled\" (click)=\"onDeleteRow(i)\" pButton pRipple type=\"button\"  icon=\"pi pi-times-circle\" class=\"p-button-rounded p-button-text p-button-danger\"\r\n                ></button>\r\n              </td>\r\n              <td style=\"width: 3rem;text-align: center\" [hidden]=\"i===(queryFilterArrayForm.length-1)\">\r\n                <p-dropdown  formControlName=\"filterExpression\" optionLabel=\"label\" optionValue=\"expression\"\r\n                            [options]=\"filterExpressions\" [filter]=\"true\"\r\n                            [showClear]=\"true\" placeholder=\"{{'queries.fields.query-filter.fields.filterExpression.placeholder'|translate}}\">\r\n\r\n\r\n                </p-dropdown>\r\n              </td>\r\n\r\n\r\n            </tr>\r\n\r\n          </ng-template>\r\n\r\n\r\n        </p-table>\r\n\r\n      </ng-template>\r\n    </div>\r\n  </form>\r\n\r\n</ng-template>\r\n</div>\r\n", styles: [""], dependencies: [{ kind: "directive", type: i10.InputText, selector: "[pInputText]" }, { kind: "directive", type: i7.PrimeTemplate, selector: "[pTemplate]", inputs: ["type", "pTemplate"] }, { kind: "directive", type: i11.NgForOf, selector: "[ngFor][ngForOf]", inputs: ["ngForOf", "ngForTrackBy", "ngForTemplate"] }, { kind: "directive", type: i11.NgIf, selector: "[ngIf]", inputs: ["ngIf", "ngIfThen", "ngIfElse"] }, { kind: "component", type: i12.Table, selector: "p-table", inputs: ["frozenColumns", "frozenValue", "style", "styleClass", "tableStyle", "tableStyleClass", "paginator", "pageLinks", "rowsPerPageOptions", "alwaysShowPaginator", "paginatorPosition", "paginatorDropdownAppendTo", "paginatorDropdownScrollHeight", "currentPageReportTemplate", "showCurrentPageReport", "showJumpToPageDropdown", "showJumpToPageInput", "showFirstLastIcon", "showPageLinks", "defaultSortOrder", "sortMode", "resetPageOnSort", "selectionMode", "selectionPageOnly", "contextMenuSelection", "contextMenuSelectionMode", "dataKey", "metaKeySelection", "rowSelectable", "rowTrackBy", "lazy", "lazyLoadOnInit", "compareSelectionBy", "csvSeparator", "exportFilename", "filters", "globalFilterFields", "filterDelay", "filterLocale", "expandedRowKeys", "editingRowKeys", "rowExpandMode", "scrollable", "scrollDirection", "rowGroupMode", "scrollHeight", "virtualScroll", "virtualScrollItemSize", "virtualScrollOptions", "virtualScrollDelay", "frozenWidth", "responsive", "contextMenu", "resizableColumns", "columnResizeMode", "reorderableColumns", "loading", "loadingIcon", "showLoader", "rowHover", "customSort", "showInitialSortBadge", "autoLayout", "exportFunction", "exportHeader", "stateKey", "stateStorage", "editMode", "groupRowsBy", "groupRowsByOrder", "responsiveLayout", "breakpoint", "virtualRowHeight", "value", "columns", "first", "rows", "totalRecords", "sortField", "sortOrder", "multiSortMeta", "selection", "selectAll"], outputs: ["selectAllChange", "selectionChange", "contextMenuSelectionChange", "onRowSelect", "onRowUnselect", "onPage", "onSort", "onFilter", "onLazyLoad", "onRowExpand", "onRowCollapse", "onContextMenuSelect", "onColResize", "onColReorder", "onRowReorder", "onEditInit", "onEditComplete", "onEditCancel", "onHeaderCheckboxToggle", "sortFunction", "firstChange", "rowsChange", "onStateSave", "onStateRestore"] }, { kind: "directive", type: i13.ButtonDirective, selector: "[pButton]", inputs: ["iconPos", "loadingIcon", "label", "icon", "loading"] }, { kind: "component", type: i14.Dropdown, selector: "p-dropdown", inputs: ["scrollHeight", "filter", "name", "style", "panelStyle", "styleClass", "panelStyleClass", "readonly", "required", "editable", "appendTo", "tabindex", "placeholder", "filterPlaceholder", "filterLocale", "inputId", "selectId", "dataKey", "filterBy", "autofocus", "resetFilterOnHide", "dropdownIcon", "optionLabel", "optionValue", "optionDisabled", "optionGroupLabel", "optionGroupChildren", "autoDisplayFirst", "group", "showClear", "emptyFilterMessage", "emptyMessage", "lazy", "virtualScroll", "virtualScrollItemSize", "virtualScrollOptions", "autoZIndex", "baseZIndex", "showTransitionOptions", "hideTransitionOptions", "ariaFilterLabel", "ariaLabel", "ariaLabelledBy", "filterMatchMode", "maxlength", "tooltip", "tooltipPosition", "tooltipPositionStyle", "tooltipStyleClass", "autofocusFilter", "disabled", "itemSize", "options", "filterValue"], outputs: ["onChange", "onFilter", "onFocus", "onBlur", "onClick", "onShow", "onHide", "onClear", "onLazyLoad"] }, { kind: "component", type: i15.Checkbox, selector: "p-checkbox", inputs: ["value", "name", "disabled", "binary", "label", "ariaLabelledBy", "ariaLabel", "tabindex", "inputId", "style", "styleClass", "labelStyleClass", "formControl", "checkboxIcon", "readonly", "required", "trueValue", "falseValue"], outputs: ["onChange"] }, { kind: "directive", type: i16.Ripple, selector: "[pRipple]" }, { kind: "directive", type: i17.ɵNgNoValidate, selector: "form:not([ngNoForm]):not([ngNativeValidate])" }, { kind: "directive", type: i17.DefaultValueAccessor, selector: "input:not([type=checkbox])[formControlName],textarea[formControlName],input:not([type=checkbox])[formControl],textarea[formControl],input:not([type=checkbox])[ngModel],textarea[ngModel],[ngDefaultControl]" }, { kind: "directive", type: i17.NgControlStatus, selector: "[formControlName],[ngModel],[formControl]" }, { kind: "directive", type: i17.NgControlStatusGroup, selector: "[formGroupName],[formArrayName],[ngModelGroup],[formGroup],form:not([ngNoForm]),[ngForm]" }, { kind: "directive", type: i17.NgModel, selector: "[ngModel]:not([formControlName]):not([formControl])", inputs: ["name", "disabled", "ngModel", "ngModelOptions"], outputs: ["ngModelChange"], exportAs: ["ngModel"] }, { kind: "directive", type: i17.FormGroupDirective, selector: "[formGroup]", inputs: ["formGroup"], outputs: ["ngSubmit"], exportAs: ["ngForm"] }, { kind: "directive", type: i17.FormControlName, selector: "[formControlName]", inputs: ["formControlName", "disabled", "ngModel"], outputs: ["ngModelChange"] }, { kind: "directive", type: i17.FormGroupName, selector: "[formGroupName]", inputs: ["formGroupName"] }, { kind: "directive", type: i17.FormArrayName, selector: "[formArrayName]", inputs: ["formArrayName"] }, { kind: "component", type: i18.ToggleButton, selector: "p-toggleButton", inputs: ["onLabel", "offLabel", "onIcon", "offIcon", "ariaLabelledBy", "disabled", "style", "styleClass", "inputId", "tabindex", "iconPos"], outputs: ["onChange"] }, { kind: "component", type: i19.PrgDynamicFormComponent, selector: "prg-dynamic-form", inputs: ["context", "fields", "actions", "entity", "viewMode", "formGroupType"], outputs: ["actionOutput", "fieldChangeOutput"] }, { kind: "pipe", type: i20.TranslatePipe, name: "translate" }] });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "14.2.0", ngImport: i0, type: PrgQueryModalComponent, decorators: [{
            type: Component,
            args: [{ selector: 'prg-select-query', template: "<div *ngIf=\"!isLoading\">\r\n<div class=\"my-4\">\r\n  <p-toggleButton (onChange)=\"onChangeQueryType()\" onLabel=\"{{'queries.query-modal.actions.query-type.filter-query.label' |translate}}\" offLabel=\"{{'queries.query-modal.actions.query-type.raw-query.label' |translate}}\"\r\n                  onIcon=\"pi pi-filter-fill\" offIcon=\"pi pi-file\" iconPos=\"right\"\r\n                  [(ngModel)]=\"checkBoxQueryRaw\"></p-toggleButton>\r\n\r\n</div>\r\n<ng-template [ngIf]=\"checkBoxQueryRaw\" [ngIfElse]=\"noRawQuery\">\r\n\r\n\r\n  <div class=\"mt-3\">\r\n    <prg-dynamic-form [viewMode]=\"queriesViewMode\" [actions]=\"queriesRawDynamicFormActions\"\r\n                      [fields]=\"queriesRawDynamicFormFields\"\r\n                      [entity]=\"query\" (actionOutput)=\"onActionQueryRaw($event)\">\r\n    </prg-dynamic-form>\r\n  </div>\r\n\r\n</ng-template>\r\n\r\n\r\n<ng-template #noRawQuery >\r\n\r\n  <prg-dynamic-form (fieldChangeOutput)=\"onChangeEntity($event.value)\" #PrgDynamicFormComponent\r\n                    (actionOutput)=\"onActionQueryFilter($event)\" [viewMode]=\"queriesViewMode\"\r\n                    [fields]=\"queriesFilterDynamicFormFields\" [actions]=\"queriesFilterDynamicFormActions\"\r\n                    [entity]=\"query\">\r\n\r\n  </prg-dynamic-form >\r\n\r\n\r\n\r\n\r\n  <form [formGroup]=\"formQueryFilter\" *ngIf=\"!isLoading\">\r\n\r\n    <div class=\"mt-2\">\r\n      <ng-template [ngIf]=\"PrgDynamicFormComponent.form.value['entityTypeId']\">\r\n        <p-table styleClass=\"p-datatable-gridlines\" [value]=\"filters\" dataKey=\"value\" formArrayName=\"filters\">\r\n          <ng-template pTemplate=\"header\">\r\n            <tr >\r\n              <ng-container *ngFor=\"let col of queriesFilterColumns\">\r\n              <th style=\"width: 10rem;text-align: center\" *ngIf=\"col.key==='startGroup'||col.key==='filterExpression'; else otherFields \" [hidden]=\"queryFilterArrayForm.length<=1\">{{col.basePathTranslation+col.key+'.label'|translate}}</th>\r\n              <ng-template #otherFields>\r\n                <th style=\"width: 10rem;text-align: center\" *ngIf=\"col.key!=='value2'&&col.key!=='query-filter-row-actions'\">{{col.basePathTranslation+col.key+'.label'|translate}}</th>\r\n                <th style=\"width: 10rem;text-align: center\" *ngIf=\"col.key==='query-filter-row-actions'\">{{col.basePathTranslation+col.key+'.label'|translate}}</th>\r\n                <th style=\"width: 10rem;text-align: center\" *ngIf=\"col.key==='value2'&&queryHasFilterWithTwoValues\">{{col.basePathTranslation+col.key+'.label'|translate}}</th>\r\n              </ng-template>\r\n              </ng-container>\r\n            </tr>\r\n          </ng-template>\r\n\r\n          <ng-template pTemplate=\"body\" let-i=\"rowIndex\">\r\n\r\n            <tr [formGroupName]=\"i\">\r\n\r\n              <td style=\"width: 5rem;text-align: center\" [hidden]=\"queryFilterArrayForm.length<=1\">\r\n\r\n                <p-checkbox formControlName=\"startGroup\" [binary]=\"true\"></p-checkbox>\r\n              </td>\r\n\r\n              <td style=\"width: 10rem;text-align: center\">\r\n\r\n\r\n                <p-dropdown [hidden]=\"PrgDynamicFormComponent.form.value['entityTypeId']==null\"\r\n                            (onChange)=\"onChangeField(i,$event.value)\" formControlName=\"propertyName\"\r\n                            [options]=\"optionsQueryFields\" optionValue=\"name\" optionLabel=\"label\" [filter]=\"true\"\r\n                            [showClear]=\"true\" placeholder=\"{{'queries.fields.query-filter.fields.propertyName.placeholder'|translate}}\" #fieldInput>\r\n\r\n                  <!--optionsQueryFields-->\r\n\r\n                </p-dropdown>\r\n              </td>\r\n              <td style=\"width: 10rem;text-align: center\">\r\n\r\n                <p-dropdown [hidden]=\"fieldInput.value==null\" formControlName=\"filterOperation\" optionLabel=\"label\" optionValue=\"operators\"\r\n                            (onChange)=\"onCheckNumberOfFields()\"\r\n                            [options]=\"mappedOperatorsAndDataTypeByField?.get(fieldInput.value)?.queryFilterOperationByType\" [filter]=\"true\"\r\n                            [showClear]=\"true\" placeholder=\"{{'queries.fields.query-filter.fields.filterOperation.placeholder'|translate}}\" #operatorInput>\r\n\r\n\r\n                </p-dropdown>\r\n\r\n              </td>\r\n\r\n\r\n              <td style=\"width: 10rem;text-align: center\"><input *ngIf=\"operatorInput.value!=null\"\r\n                                                                 formControlName=\"value\" id=\"value\"\r\n                                                                 [type]=\"mappedOperatorsAndDataTypeByField?.get(fieldInput.value)?.dataType==='number'?'number':mappedOperatorsAndDataTypeByField?.get(fieldInput.value)?.dataType==='date'?'date':'text'\"\r\n                                                                 placeholder=\"{{'queries.fields.query-filter.fields.value.placeholder'|translate}}\" pInputText\r\n\r\n\r\n              >\r\n              </td>\r\n\r\n              <td *ngIf=\"queryHasFilterWithTwoValues\" style=\"width: 10rem;text-align: center\"><input\r\n                *ngIf=\"operatorType.numberOfFields(operatorInput.value)==2\"\r\n                formControlName=\"value2\" id=\"value2\"\r\n                [type]=\"mappedOperatorsAndDataTypeByField?.get(fieldInput.value)?.dataType==='number'?'number':mappedOperatorsAndDataTypeByField?.get(fieldInput.value)?.dataType==='date'?'date':'text'\"\r\n                placeholder=\"{{'queries.fields.query-filter.fields.value2.placeholder'|translate}}\"\r\n                pInputText>\r\n              </td>\r\n\r\n              <td style=\"width: 10rem;text-align: center\">\r\n\r\n                <button [disabled]=\"this.queryFilterArrayForm.disabled\" pButton pRipple type=\"button\"  icon=\"pi pi-plus-circle\"\r\n                        (click)=\"onAddRow(i)\"\r\n                  class=\"p-button-rounded p-button-text p-button-success p-mr-2\"\r\n                ></button>\r\n\r\n                <button [disabled]=\"this.queryFilterArrayForm.disabled\" (click)=\"onDeleteRow(i)\" pButton pRipple type=\"button\"  icon=\"pi pi-times-circle\" class=\"p-button-rounded p-button-text p-button-danger\"\r\n                ></button>\r\n              </td>\r\n              <td style=\"width: 3rem;text-align: center\" [hidden]=\"i===(queryFilterArrayForm.length-1)\">\r\n                <p-dropdown  formControlName=\"filterExpression\" optionLabel=\"label\" optionValue=\"expression\"\r\n                            [options]=\"filterExpressions\" [filter]=\"true\"\r\n                            [showClear]=\"true\" placeholder=\"{{'queries.fields.query-filter.fields.filterExpression.placeholder'|translate}}\">\r\n\r\n\r\n                </p-dropdown>\r\n              </td>\r\n\r\n\r\n            </tr>\r\n\r\n          </ng-template>\r\n\r\n\r\n        </p-table>\r\n\r\n      </ng-template>\r\n    </div>\r\n  </form>\r\n\r\n</ng-template>\r\n</div>\r\n" }]
        }], ctorParameters: function () { return [{ type: i1.DynamicDialogRef }, { type: i1.DynamicDialogConfig }, { type: i2.AbstractQueriesService }, { type: i3.ObjectsUtilityService }, { type: i4.ArrayUtilityService }, { type: i5.FormGroupService }, { type: i6.NotificationsService }, { type: i7.PrimeNGConfig }, { type: i8.AbstractEntityTypeService }, { type: i9.AbstractLookupTableService }]; } });
//# sourceMappingURL=data:application/json;base64,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