import { Component, Inject } from '@angular/core';
import { ViewMode } from '../../../core/models/constants/view-mode.enum';
import { Query } from '../../models/query.model';
import { PRG_QUERIES_CONFIG } from '../../services/prg-queries-configuration/prg-queries-configuration.service';
import { PrgQueryModalComponent } from '../query-modal/prg-query-modal.component';
import * as i0 from "@angular/core";
import * as i1 from "../../services/queries/abstract-queries.service";
import * as i2 from "../../../core/services/utility/array-utility.service";
import * as i3 from "../../../core/services/utility/objects-utility.service";
import * as i4 from "../../../auth/services/auth/abstract-auth.service";
import * as i5 from "primeng/dynamicdialog";
import * as i6 from "primeng/tooltip";
import * as i7 from "primeng/api";
import * as i8 from "@angular/common";
import * as i9 from "primeng/button";
import * as i10 from "primeng/treetable";
import * as i11 from "@ngx-translate/core";
/**
 * Queries Table Component
 */
export class PrgQueriesTableComponent {
    /**
     * Constructor
     * @param {AbstractQueriesService} queriesService
     * @param {ArrayUtilityService} arrayUtilityService
     * @param {ObjectsUtilityService} objectsUtilityService
     * @param {AbstractAuthService} userService
     * @param {PrgQueriesConfig} prgQueriesConfig
     * @param {DialogService} dialogService dialogService
     */
    constructor(queriesService, arrayUtilityService, objectsUtilityService, userService, prgQueriesConfig, dialogService) {
        this.queriesService = queriesService;
        this.arrayUtilityService = arrayUtilityService;
        this.objectsUtilityService = objectsUtilityService;
        this.userService = userService;
        this.prgQueriesConfig = prgQueriesConfig;
        this.dialogService = dialogService;
        /**
         * Queries Table columns
         */
        this.queriesTableColumns = this.prgQueriesConfig.queryTableColumns;
        /**
         * A class property used to unsubscribe observables on ngOnDestroy
         */
        this.subscription = [];
    }
    /**
     * ngOnInit
     */
    async ngOnInit() {
        await this.getAllQueries();
    }
    /**
     * ngOnDestroy
     *
     * Unsubscribe all subscriptions
     */
    ngOnDestroy() {
        this.subscription.forEach((subscription) => {
            subscription.unsubscribe();
        });
        this.subscription = [];
    }
    /**
     * Get queries list (async)
     *
     * Function responsible to fetch all queries from api to populate table
     * @private
     */
    async getAllQueries() {
        this.queries = await this.queriesService.getAllQueriesAsync();
        this.createTreeNode();
    }
    /**
     * Creation of Tree Node Object
     *
     * This function is responsible to group queries by creator  (logged user and others)
     * to organize query data for the table tree
     * @private
     */
    createTreeNode() {
        this.queries.forEach((query) => {
            if (this.userService.getLoggedUser()) {
                this.loggedUserId = this.userService.getLoggedUser().id;
                if (query.createdBy === this.loggedUserId) {
                    query['group'] = 'myQueries';
                }
                else {
                    query['group'] = 'otherQueries';
                }
            }
        });
        this.treeData = this.arrayUtilityService.groupByField(this.queries, 'group');
    }
    /**
     * Function responsible to open dialog when a query is clicked.
     *
     * It also manages the data that comes from dialog when it is closed.
     *
     * @param {Query} query the selected query
     */
    openQueryDialog(query) {
        let queryAux;
        if (query) {
            queryAux = this.objectsUtilityService.cloneObject(query);
        }
        else {
            queryAux = new Query();
        }
        this.querySelectedDialogRef = this.dialogService.open(PrgQueryModalComponent, {
            header: queryAux ? queryAux.name : 'New Query',
            width: '80%',
            height: '90vh',
            baseZIndex: 10000,
            data: {
                query: queryAux,
                viewMode: queryAux.id ? ViewMode.Read : ViewMode.Edit,
            },
        });
        this.subscription.push(this.querySelectedDialogRef.onClose.subscribe((newData) => {
            this.subscription.push(this.queriesService
                .getQueriesChangedObservable()
                .subscribe((queries) => {
                if (newData && queries) {
                    this.queries = this.arrayUtilityService.clone(queries);
                    this.createTreeNode();
                }
                this.querySelectedDialogRef.destroy();
            }));
        }));
    }
}
PrgQueriesTableComponent.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "14.2.0", ngImport: i0, type: PrgQueriesTableComponent, deps: [{ token: i1.AbstractQueriesService }, { token: i2.ArrayUtilityService }, { token: i3.ObjectsUtilityService }, { token: i4.AbstractAuthService }, { token: PRG_QUERIES_CONFIG }, { token: i5.DialogService }], target: i0.ɵɵFactoryTarget.Component });
PrgQueriesTableComponent.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "14.0.0", version: "14.2.0", type: PrgQueriesTableComponent, selector: "prg-queries-table", ngImport: i0, template: "\r\n<div>\r\n  <p-treeTable [value]=\"treeData\" >\r\n    <ng-template pTemplate=\"header\">\r\n      <tr>\r\n\r\n        <th style=\"width: 10rem;text-align: center\" *ngFor=\"let col of queriesTableColumns\" [ttSortableColumn]=\"col.key\" pTooltip=\"{{col.basePathTranslation+'.'+col.key+'.tooltip'|translate }}\" tooltipPosition=\"top\">\r\n\r\n            {{ col.basePathTranslation+\".\"+col.key+\".label\"|translate }}\r\n\r\n          <p-treeTableSortIcon [field]=\"col.key\"></p-treeTableSortIcon>\r\n\r\n        </th>\r\n        <th style=\"width: 3rem;text-align: center\">\r\n          <p-button pTooltip=\"{{'queries.queries-list.actions.add.label'|translate}}\" tooltipPosition=\"bottom\" (click)=\" openQueryDialog(null)\" type=\"button\" icon=\"pi pi-plus-circle\"></p-button>\r\n        </th>\r\n      </tr>\r\n    </ng-template>\r\n\r\n    <ng-template pTemplate=\"body\" let-rowNode let-rowData=\"rowData\">\r\n      <tr\r\n        [ttRow]=\"rowNode\"\r\n        (click)=\"rowNode.level !== 0 && openQueryDialog(rowData)\"\r\n        [ngClass]=\"{ clickableRow: rowNode.level !== 0 }\"\r\n      >\r\n        <td *ngFor=\"let col of queriesTableColumns; let i = index ;let last = last \" [colSpan]=\"last?2:1\" [ngStyle]=\"{'text-align':(rowNode.level!==0)? 'center':''}\" >\r\n          <p-treeTableToggler\r\n            *ngIf=\"i == 0\"\r\n            [rowNode]=\"rowNode\"\r\n\r\n          ></p-treeTableToggler>\r\n\r\n          <ng-template [ngIf]=\"col.key=='isRaw' && rowNode.level!==0\">\r\n            {{rowData[col.key] ? 'Yes': 'No'}}\r\n          </ng-template>\r\n          <ng-template  [ngIf]=\"col.key!='isRaw' && rowNode.level!==0\" >\r\n            {{rowData[col.key]}}\r\n          </ng-template>\r\n\r\n          <ng-template  [ngIf]=\"i==0&&rowNode.level==0\">\r\n            {{'queries.queries-list.queries-group.'+rowData[col.key]+'.label'|translate }}\r\n\r\n          </ng-template>\r\n\r\n\r\n\r\n        </td>\r\n      </tr>\r\n    </ng-template>\r\n  </p-treeTable>\r\n\r\n</div>\r\n", styles: [".clickableRow{cursor:pointer}.clickableRow:hover{background-color:#cbf6f6}\n"], dependencies: [{ kind: "directive", type: i6.Tooltip, selector: "[pTooltip]", inputs: ["tooltipPosition", "tooltipEvent", "appendTo", "positionStyle", "tooltipStyleClass", "tooltipZIndex", "escape", "showDelay", "hideDelay", "life", "positionTop", "positionLeft", "fitContent", "pTooltip", "tooltipDisabled", "tooltipOptions"] }, { kind: "directive", type: i7.PrimeTemplate, selector: "[pTemplate]", inputs: ["type", "pTemplate"] }, { kind: "directive", type: i8.NgClass, selector: "[ngClass]", inputs: ["class", "ngClass"] }, { kind: "directive", type: i8.NgForOf, selector: "[ngFor][ngForOf]", inputs: ["ngForOf", "ngForTrackBy", "ngForTemplate"] }, { kind: "directive", type: i8.NgIf, selector: "[ngIf]", inputs: ["ngIf", "ngIfThen", "ngIfElse"] }, { kind: "directive", type: i8.NgStyle, selector: "[ngStyle]", inputs: ["ngStyle"] }, { kind: "component", type: i9.Button, selector: "p-button", inputs: ["type", "iconPos", "icon", "badge", "label", "disabled", "loading", "loadingIcon", "style", "styleClass", "badgeClass", "ariaLabel"], outputs: ["onClick", "onFocus", "onBlur"] }, { kind: "component", type: i10.TreeTable, selector: "p-treeTable", inputs: ["columns", "style", "styleClass", "tableStyle", "tableStyleClass", "autoLayout", "lazy", "lazyLoadOnInit", "paginator", "rows", "first", "pageLinks", "rowsPerPageOptions", "alwaysShowPaginator", "paginatorPosition", "paginatorDropdownAppendTo", "currentPageReportTemplate", "showCurrentPageReport", "showJumpToPageDropdown", "showFirstLastIcon", "showPageLinks", "defaultSortOrder", "sortMode", "resetPageOnSort", "customSort", "selectionMode", "contextMenuSelection", "contextMenuSelectionMode", "dataKey", "metaKeySelection", "compareSelectionBy", "rowHover", "loading", "loadingIcon", "showLoader", "scrollable", "scrollHeight", "virtualScroll", "virtualScrollItemSize", "virtualScrollOptions", "virtualScrollDelay", "frozenWidth", "frozenColumns", "resizableColumns", "columnResizeMode", "reorderableColumns", "contextMenu", "rowTrackBy", "filters", "globalFilterFields", "filterDelay", "filterMode", "filterLocale", "virtualRowHeight", "value", "totalRecords", "sortField", "sortOrder", "multiSortMeta", "selection"], outputs: ["selectionChange", "contextMenuSelectionChange", "onFilter", "onNodeExpand", "onNodeCollapse", "onPage", "onSort", "onLazyLoad", "sortFunction", "onColResize", "onColReorder", "onNodeSelect", "onNodeUnselect", "onContextMenuSelect", "onHeaderCheckboxToggle", "onEditInit", "onEditComplete", "onEditCancel"] }, { kind: "component", type: i10.TreeTableToggler, selector: "p-treeTableToggler", inputs: ["rowNode"] }, { kind: "directive", type: i10.TTSortableColumn, selector: "[ttSortableColumn]", inputs: ["ttSortableColumn", "ttSortableColumnDisabled"] }, { kind: "component", type: i10.TTSortIcon, selector: "p-treeTableSortIcon", inputs: ["field", "ariaLabelDesc", "ariaLabelAsc"] }, { kind: "directive", type: i10.TTRow, selector: "[ttRow]", inputs: ["ttRow"] }, { kind: "pipe", type: i11.TranslatePipe, name: "translate" }] });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "14.2.0", ngImport: i0, type: PrgQueriesTableComponent, decorators: [{
            type: Component,
            args: [{ selector: 'prg-queries-table', template: "\r\n<div>\r\n  <p-treeTable [value]=\"treeData\" >\r\n    <ng-template pTemplate=\"header\">\r\n      <tr>\r\n\r\n        <th style=\"width: 10rem;text-align: center\" *ngFor=\"let col of queriesTableColumns\" [ttSortableColumn]=\"col.key\" pTooltip=\"{{col.basePathTranslation+'.'+col.key+'.tooltip'|translate }}\" tooltipPosition=\"top\">\r\n\r\n            {{ col.basePathTranslation+\".\"+col.key+\".label\"|translate }}\r\n\r\n          <p-treeTableSortIcon [field]=\"col.key\"></p-treeTableSortIcon>\r\n\r\n        </th>\r\n        <th style=\"width: 3rem;text-align: center\">\r\n          <p-button pTooltip=\"{{'queries.queries-list.actions.add.label'|translate}}\" tooltipPosition=\"bottom\" (click)=\" openQueryDialog(null)\" type=\"button\" icon=\"pi pi-plus-circle\"></p-button>\r\n        </th>\r\n      </tr>\r\n    </ng-template>\r\n\r\n    <ng-template pTemplate=\"body\" let-rowNode let-rowData=\"rowData\">\r\n      <tr\r\n        [ttRow]=\"rowNode\"\r\n        (click)=\"rowNode.level !== 0 && openQueryDialog(rowData)\"\r\n        [ngClass]=\"{ clickableRow: rowNode.level !== 0 }\"\r\n      >\r\n        <td *ngFor=\"let col of queriesTableColumns; let i = index ;let last = last \" [colSpan]=\"last?2:1\" [ngStyle]=\"{'text-align':(rowNode.level!==0)? 'center':''}\" >\r\n          <p-treeTableToggler\r\n            *ngIf=\"i == 0\"\r\n            [rowNode]=\"rowNode\"\r\n\r\n          ></p-treeTableToggler>\r\n\r\n          <ng-template [ngIf]=\"col.key=='isRaw' && rowNode.level!==0\">\r\n            {{rowData[col.key] ? 'Yes': 'No'}}\r\n          </ng-template>\r\n          <ng-template  [ngIf]=\"col.key!='isRaw' && rowNode.level!==0\" >\r\n            {{rowData[col.key]}}\r\n          </ng-template>\r\n\r\n          <ng-template  [ngIf]=\"i==0&&rowNode.level==0\">\r\n            {{'queries.queries-list.queries-group.'+rowData[col.key]+'.label'|translate }}\r\n\r\n          </ng-template>\r\n\r\n\r\n\r\n        </td>\r\n      </tr>\r\n    </ng-template>\r\n  </p-treeTable>\r\n\r\n</div>\r\n", styles: [".clickableRow{cursor:pointer}.clickableRow:hover{background-color:#cbf6f6}\n"] }]
        }], ctorParameters: function () { return [{ type: i1.AbstractQueriesService }, { type: i2.ArrayUtilityService }, { type: i3.ObjectsUtilityService }, { type: i4.AbstractAuthService }, { type: undefined, decorators: [{
                    type: Inject,
                    args: [PRG_QUERIES_CONFIG]
                }] }, { type: i5.DialogService }]; } });
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicHJnLXF1ZXJpZXMtdGFibGUuY29tcG9uZW50LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vLi4vLi4vLi4vLi4vLi4vcHJvamVjdHMvUHJnRnJhbWV3b3JrLkZFLkNvcmVMaWIvc3JjL2xpYi9tb2R1bGVzL3F1ZXJpZXMvY29tcG9uZW50cy9xdWVyaWVzLXRhYmxlL3ByZy1xdWVyaWVzLXRhYmxlLmNvbXBvbmVudC50cyIsIi4uLy4uLy4uLy4uLy4uLy4uLy4uLy4uL3Byb2plY3RzL1ByZ0ZyYW1ld29yay5GRS5Db3JlTGliL3NyYy9saWIvbW9kdWxlcy9xdWVyaWVzL2NvbXBvbmVudHMvcXVlcmllcy10YWJsZS9wcmctcXVlcmllcy10YWJsZS5jb21wb25lbnQuaHRtbCJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQSxPQUFPLEVBQUUsU0FBUyxFQUFFLE1BQU0sRUFBcUIsTUFBTSxlQUFlLENBQUM7QUFNckUsT0FBTyxFQUFFLFFBQVEsRUFBRSxNQUFNLCtDQUErQyxDQUFDO0FBSXpFLE9BQU8sRUFBRSxLQUFLLEVBQUUsTUFBTSwwQkFBMEIsQ0FBQztBQUNqRCxPQUFPLEVBQUUsa0JBQWtCLEVBQUUsTUFBTSw0RUFBNEUsQ0FBQztBQUVoSCxPQUFPLEVBQUUsc0JBQXNCLEVBQUUsTUFBTSwwQ0FBMEMsQ0FBQzs7Ozs7Ozs7Ozs7OztBQUVsRjs7R0FFRztBQU1ILE1BQU0sT0FBTyx3QkFBd0I7SUE2Qm5DOzs7Ozs7OztPQVFHO0lBQ0gsWUFDVSxjQUFzQyxFQUN0QyxtQkFBd0MsRUFDeEMscUJBQTRDLEVBQzVDLFdBQWdDLEVBQ0osZ0JBQWtDLEVBQy9ELGFBQTRCO1FBTDNCLG1CQUFjLEdBQWQsY0FBYyxDQUF3QjtRQUN0Qyx3QkFBbUIsR0FBbkIsbUJBQW1CLENBQXFCO1FBQ3hDLDBCQUFxQixHQUFyQixxQkFBcUIsQ0FBdUI7UUFDNUMsZ0JBQVcsR0FBWCxXQUFXLENBQXFCO1FBQ0oscUJBQWdCLEdBQWhCLGdCQUFnQixDQUFrQjtRQUMvRCxrQkFBYSxHQUFiLGFBQWEsQ0FBZTtRQXZDckM7O1dBRUc7UUFDSSx3QkFBbUIsR0FDeEIsSUFBSSxDQUFDLGdCQUFnQixDQUFDLGlCQUFpQixDQUFDO1FBQzFDOztXQUVHO1FBQ0ksaUJBQVksR0FBbUIsRUFBRSxDQUFDO0lBZ0N0QyxDQUFDO0lBRUo7O09BRUc7SUFDSSxLQUFLLENBQUMsUUFBUTtRQUNuQixNQUFNLElBQUksQ0FBQyxhQUFhLEVBQUUsQ0FBQztJQUM3QixDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLFdBQVc7UUFDaEIsSUFBSSxDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUMsQ0FBQyxZQUFZLEVBQUUsRUFBRTtZQUN6QyxZQUFZLENBQUMsV0FBVyxFQUFFLENBQUM7UUFDN0IsQ0FBQyxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsWUFBWSxHQUFHLEVBQUUsQ0FBQztJQUN6QixDQUFDO0lBRUQ7Ozs7O09BS0c7SUFFSyxLQUFLLENBQUMsYUFBYTtRQUN6QixJQUFJLENBQUMsT0FBTyxHQUFHLE1BQU0sSUFBSSxDQUFDLGNBQWMsQ0FBQyxrQkFBa0IsRUFBRSxDQUFDO1FBQzlELElBQUksQ0FBQyxjQUFjLEVBQUUsQ0FBQztJQUN4QixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ssY0FBYztRQUNwQixJQUFJLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxDQUFDLEtBQVksRUFBRSxFQUFFO1lBQ3BDLElBQUksSUFBSSxDQUFDLFdBQVcsQ0FBQyxhQUFhLEVBQUUsRUFBRTtnQkFDcEMsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLGFBQWEsRUFBRSxDQUFDLEVBQUUsQ0FBQztnQkFFeEQsSUFBSSxLQUFLLENBQUMsU0FBUyxLQUFLLElBQUksQ0FBQyxZQUFZLEVBQUU7b0JBQ3pDLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxXQUFXLENBQUM7aUJBQzlCO3FCQUFNO29CQUNMLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxjQUFjLENBQUM7aUJBQ2pDO2FBQ0Y7UUFDSCxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxRQUFRLEdBQUcsSUFBSSxDQUFDLG1CQUFtQixDQUFDLFlBQVksQ0FDbkQsSUFBSSxDQUFDLE9BQU8sRUFDWixPQUFPLENBQ1IsQ0FBQztJQUNKLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxlQUFlLENBQUMsS0FBWTtRQUNqQyxJQUFJLFFBQWUsQ0FBQztRQUVwQixJQUFJLEtBQUssRUFBRTtZQUNULFFBQVEsR0FBRyxJQUFJLENBQUMscUJBQXFCLENBQUMsV0FBVyxDQUFDLEtBQUssQ0FBQyxDQUFDO1NBQzFEO2FBQU07WUFDTCxRQUFRLEdBQUcsSUFBSSxLQUFLLEVBQUUsQ0FBQztTQUN4QjtRQUVELElBQUksQ0FBQyxzQkFBc0IsR0FBRyxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FDbkQsc0JBQXNCLEVBQ3RCO1lBQ0UsTUFBTSxFQUFFLFFBQVEsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsV0FBVztZQUM5QyxLQUFLLEVBQUUsS0FBSztZQUNaLE1BQU0sRUFBRSxNQUFNO1lBQ2QsVUFBVSxFQUFFLEtBQUs7WUFDakIsSUFBSSxFQUFFO2dCQUNKLEtBQUssRUFBRSxRQUFRO2dCQUNmLFFBQVEsRUFBRSxRQUFRLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxRQUFRLENBQUMsSUFBSTthQUN0RDtTQUNGLENBQ0YsQ0FBQztRQUNGLElBQUksQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUNwQixJQUFJLENBQUMsc0JBQXNCLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxDQUFDLE9BQU8sRUFBRSxFQUFFO1lBQ3hELElBQUksQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUNwQixJQUFJLENBQUMsY0FBYztpQkFDaEIsMkJBQTJCLEVBQUU7aUJBQzdCLFNBQVMsQ0FBQyxDQUFDLE9BQU8sRUFBRSxFQUFFO2dCQUNyQixJQUFJLE9BQU8sSUFBSSxPQUFPLEVBQUU7b0JBQ3RCLElBQUksQ0FBQyxPQUFPLEdBQUcsSUFBSSxDQUFDLG1CQUFtQixDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQztvQkFDdkQsSUFBSSxDQUFDLGNBQWMsRUFBRSxDQUFDO2lCQUN2QjtnQkFDRCxJQUFJLENBQUMsc0JBQXNCLENBQUMsT0FBTyxFQUFFLENBQUM7WUFDeEMsQ0FBQyxDQUFDLENBQ0wsQ0FBQztRQUNKLENBQUMsQ0FBQyxDQUNILENBQUM7SUFDSixDQUFDOztxSEFwSlUsd0JBQXdCLG1LQTJDekIsa0JBQWtCO3lHQTNDakIsd0JBQXdCLHlEQ3ZCckMsKytEQW9EQTsyRkQ3QmEsd0JBQXdCO2tCQUxwQyxTQUFTOytCQUNFLG1CQUFtQjs7MEJBK0MxQixNQUFNOzJCQUFDLGtCQUFrQiIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IENvbXBvbmVudCwgSW5qZWN0LCBPbkRlc3Ryb3ksIE9uSW5pdCB9IGZyb20gJ0Bhbmd1bGFyL2NvcmUnO1xyXG5pbXBvcnQgeyBUcmVlTm9kZSB9IGZyb20gJ3ByaW1lbmcvYXBpJztcclxuaW1wb3J0IHsgRGlhbG9nU2VydmljZSwgRHluYW1pY0RpYWxvZ1JlZiB9IGZyb20gJ3ByaW1lbmcvZHluYW1pY2RpYWxvZyc7XHJcbmltcG9ydCB7IFN1YnNjcmlwdGlvbiB9IGZyb20gJ3J4anMnO1xyXG5pbXBvcnQgeyBBYnN0cmFjdEF1dGhTZXJ2aWNlIH0gZnJvbSAnLi4vLi4vLi4vYXV0aC9zZXJ2aWNlcy9hdXRoL2Fic3RyYWN0LWF1dGguc2VydmljZSc7XHJcbmltcG9ydCB7IEJhc2VGaWVsZCB9IGZyb20gJy4uLy4uLy4uL2NvcmUvY29tcG9uZW50cy9keW5hbWljLWZvcm0vbW9kZWxzL2Jhc2UtZmllbGQnO1xyXG5pbXBvcnQgeyBWaWV3TW9kZSB9IGZyb20gJy4uLy4uLy4uL2NvcmUvbW9kZWxzL2NvbnN0YW50cy92aWV3LW1vZGUuZW51bSc7XHJcbmltcG9ydCB7IEFycmF5VXRpbGl0eVNlcnZpY2UgfSBmcm9tICcuLi8uLi8uLi9jb3JlL3NlcnZpY2VzL3V0aWxpdHkvYXJyYXktdXRpbGl0eS5zZXJ2aWNlJztcclxuaW1wb3J0IHsgT2JqZWN0c1V0aWxpdHlTZXJ2aWNlIH0gZnJvbSAnLi4vLi4vLi4vY29yZS9zZXJ2aWNlcy91dGlsaXR5L29iamVjdHMtdXRpbGl0eS5zZXJ2aWNlJztcclxuaW1wb3J0IHsgUHJnUXVlcmllc0NvbmZpZyB9IGZyb20gJy4uLy4uL21vZGVscy9wcmctcXVlcmllcy1jb25maWcnO1xyXG5pbXBvcnQgeyBRdWVyeSB9IGZyb20gJy4uLy4uL21vZGVscy9xdWVyeS5tb2RlbCc7XHJcbmltcG9ydCB7IFBSR19RVUVSSUVTX0NPTkZJRyB9IGZyb20gJy4uLy4uL3NlcnZpY2VzL3ByZy1xdWVyaWVzLWNvbmZpZ3VyYXRpb24vcHJnLXF1ZXJpZXMtY29uZmlndXJhdGlvbi5zZXJ2aWNlJztcclxuaW1wb3J0IHsgQWJzdHJhY3RRdWVyaWVzU2VydmljZSB9IGZyb20gJy4uLy4uL3NlcnZpY2VzL3F1ZXJpZXMvYWJzdHJhY3QtcXVlcmllcy5zZXJ2aWNlJztcclxuaW1wb3J0IHsgUHJnUXVlcnlNb2RhbENvbXBvbmVudCB9IGZyb20gJy4uL3F1ZXJ5LW1vZGFsL3ByZy1xdWVyeS1tb2RhbC5jb21wb25lbnQnO1xyXG5cclxuLyoqXHJcbiAqIFF1ZXJpZXMgVGFibGUgQ29tcG9uZW50XHJcbiAqL1xyXG5AQ29tcG9uZW50KHtcclxuICBzZWxlY3RvcjogJ3ByZy1xdWVyaWVzLXRhYmxlJyxcclxuICB0ZW1wbGF0ZVVybDogJy4vcHJnLXF1ZXJpZXMtdGFibGUuY29tcG9uZW50Lmh0bWwnLFxyXG4gIHN0eWxlVXJsczogWycuL3ByZy1xdWVyaWVzLXRhYmxlLmNvbXBvbmVudC5zY3NzJ10sXHJcbn0pXHJcbmV4cG9ydCBjbGFzcyBQcmdRdWVyaWVzVGFibGVDb21wb25lbnQgaW1wbGVtZW50cyBPbkluaXQsIE9uRGVzdHJveSB7XHJcbiAgLyoqXHJcbiAgICogdHJlZURhdGFcclxuICAgKi9cclxuICBwdWJsaWMgdHJlZURhdGE6IFRyZWVOb2RlW107XHJcbiAgLyoqXHJcbiAgICogUXVlcmllcyBUYWJsZSBjb2x1bW5zXHJcbiAgICovXHJcbiAgcHVibGljIHF1ZXJpZXNUYWJsZUNvbHVtbnM6IEJhc2VGaWVsZFtdID1cclxuICAgIHRoaXMucHJnUXVlcmllc0NvbmZpZy5xdWVyeVRhYmxlQ29sdW1ucztcclxuICAvKipcclxuICAgKiBBIGNsYXNzIHByb3BlcnR5IHVzZWQgdG8gdW5zdWJzY3JpYmUgb2JzZXJ2YWJsZXMgb24gbmdPbkRlc3Ryb3lcclxuICAgKi9cclxuICBwdWJsaWMgc3Vic2NyaXB0aW9uOiBTdWJzY3JpcHRpb25bXSA9IFtdO1xyXG5cclxuICAvKipcclxuICAgKiBQcm9wZXJ0eSBvZiB0eXBlIER5bmFtaWNEaWFsb2dSZWYgdG8gY29udHJvbCB0aGUgZGlhbG9nXHJcbiAgICovXHJcbiAgcHJpdmF0ZSBxdWVyeVNlbGVjdGVkRGlhbG9nUmVmITogRHluYW1pY0RpYWxvZ1JlZjtcclxuICAvKipcclxuICAgKiBDdXJyZW50bHkgbG9nZ2VkIHVzZXJcclxuICAgKi9cclxuICBwcml2YXRlIGxvZ2dlZFVzZXJJZDogc3RyaW5nO1xyXG5cclxuICAvKipcclxuICAgKiBRdWVyaWVzIExpc3RcclxuICAgKi9cclxuICBwcml2YXRlIHF1ZXJpZXM6IFF1ZXJ5W107XHJcblxyXG4gIC8qKlxyXG4gICAqIENvbnN0cnVjdG9yXHJcbiAgICogQHBhcmFtIHtBYnN0cmFjdFF1ZXJpZXNTZXJ2aWNlfSBxdWVyaWVzU2VydmljZVxyXG4gICAqIEBwYXJhbSB7QXJyYXlVdGlsaXR5U2VydmljZX0gYXJyYXlVdGlsaXR5U2VydmljZVxyXG4gICAqIEBwYXJhbSB7T2JqZWN0c1V0aWxpdHlTZXJ2aWNlfSBvYmplY3RzVXRpbGl0eVNlcnZpY2VcclxuICAgKiBAcGFyYW0ge0Fic3RyYWN0QXV0aFNlcnZpY2V9IHVzZXJTZXJ2aWNlXHJcbiAgICogQHBhcmFtIHtQcmdRdWVyaWVzQ29uZmlnfSBwcmdRdWVyaWVzQ29uZmlnXHJcbiAgICogQHBhcmFtIHtEaWFsb2dTZXJ2aWNlfSBkaWFsb2dTZXJ2aWNlIGRpYWxvZ1NlcnZpY2VcclxuICAgKi9cclxuICBjb25zdHJ1Y3RvcihcclxuICAgIHByaXZhdGUgcXVlcmllc1NlcnZpY2U6IEFic3RyYWN0UXVlcmllc1NlcnZpY2UsXHJcbiAgICBwcml2YXRlIGFycmF5VXRpbGl0eVNlcnZpY2U6IEFycmF5VXRpbGl0eVNlcnZpY2UsXHJcbiAgICBwcml2YXRlIG9iamVjdHNVdGlsaXR5U2VydmljZTogT2JqZWN0c1V0aWxpdHlTZXJ2aWNlLFxyXG4gICAgcHJpdmF0ZSB1c2VyU2VydmljZTogQWJzdHJhY3RBdXRoU2VydmljZSxcclxuICAgIEBJbmplY3QoUFJHX1FVRVJJRVNfQ09ORklHKSBwcml2YXRlIHByZ1F1ZXJpZXNDb25maWc6IFByZ1F1ZXJpZXNDb25maWcsXHJcbiAgICBwdWJsaWMgZGlhbG9nU2VydmljZTogRGlhbG9nU2VydmljZVxyXG4gICkge31cclxuXHJcbiAgLyoqXHJcbiAgICogbmdPbkluaXRcclxuICAgKi9cclxuICBwdWJsaWMgYXN5bmMgbmdPbkluaXQoKTogUHJvbWlzZTx2b2lkPiB7XHJcbiAgICBhd2FpdCB0aGlzLmdldEFsbFF1ZXJpZXMoKTtcclxuICB9XHJcblxyXG4gIC8qKlxyXG4gICAqIG5nT25EZXN0cm95XHJcbiAgICpcclxuICAgKiBVbnN1YnNjcmliZSBhbGwgc3Vic2NyaXB0aW9uc1xyXG4gICAqL1xyXG4gIHB1YmxpYyBuZ09uRGVzdHJveSgpOiB2b2lkIHtcclxuICAgIHRoaXMuc3Vic2NyaXB0aW9uLmZvckVhY2goKHN1YnNjcmlwdGlvbikgPT4ge1xyXG4gICAgICBzdWJzY3JpcHRpb24udW5zdWJzY3JpYmUoKTtcclxuICAgIH0pO1xyXG4gICAgdGhpcy5zdWJzY3JpcHRpb24gPSBbXTtcclxuICB9XHJcblxyXG4gIC8qKlxyXG4gICAqIEdldCBxdWVyaWVzIGxpc3QgKGFzeW5jKVxyXG4gICAqXHJcbiAgICogRnVuY3Rpb24gcmVzcG9uc2libGUgdG8gZmV0Y2ggYWxsIHF1ZXJpZXMgZnJvbSBhcGkgdG8gcG9wdWxhdGUgdGFibGVcclxuICAgKiBAcHJpdmF0ZVxyXG4gICAqL1xyXG5cclxuICBwcml2YXRlIGFzeW5jIGdldEFsbFF1ZXJpZXMoKTogUHJvbWlzZTx2b2lkPiB7XHJcbiAgICB0aGlzLnF1ZXJpZXMgPSBhd2FpdCB0aGlzLnF1ZXJpZXNTZXJ2aWNlLmdldEFsbFF1ZXJpZXNBc3luYygpO1xyXG4gICAgdGhpcy5jcmVhdGVUcmVlTm9kZSgpO1xyXG4gIH1cclxuXHJcbiAgLyoqXHJcbiAgICogQ3JlYXRpb24gb2YgVHJlZSBOb2RlIE9iamVjdFxyXG4gICAqXHJcbiAgICogVGhpcyBmdW5jdGlvbiBpcyByZXNwb25zaWJsZSB0byBncm91cCBxdWVyaWVzIGJ5IGNyZWF0b3IgIChsb2dnZWQgdXNlciBhbmQgb3RoZXJzKVxyXG4gICAqIHRvIG9yZ2FuaXplIHF1ZXJ5IGRhdGEgZm9yIHRoZSB0YWJsZSB0cmVlXHJcbiAgICogQHByaXZhdGVcclxuICAgKi9cclxuICBwcml2YXRlIGNyZWF0ZVRyZWVOb2RlKCk6IHZvaWQge1xyXG4gICAgdGhpcy5xdWVyaWVzLmZvckVhY2goKHF1ZXJ5OiBRdWVyeSkgPT4ge1xyXG4gICAgICBpZiAodGhpcy51c2VyU2VydmljZS5nZXRMb2dnZWRVc2VyKCkpIHtcclxuICAgICAgICB0aGlzLmxvZ2dlZFVzZXJJZCA9IHRoaXMudXNlclNlcnZpY2UuZ2V0TG9nZ2VkVXNlcigpLmlkO1xyXG5cclxuICAgICAgICBpZiAocXVlcnkuY3JlYXRlZEJ5ID09PSB0aGlzLmxvZ2dlZFVzZXJJZCkge1xyXG4gICAgICAgICAgcXVlcnlbJ2dyb3VwJ10gPSAnbXlRdWVyaWVzJztcclxuICAgICAgICB9IGVsc2Uge1xyXG4gICAgICAgICAgcXVlcnlbJ2dyb3VwJ10gPSAnb3RoZXJRdWVyaWVzJztcclxuICAgICAgICB9XHJcbiAgICAgIH1cclxuICAgIH0pO1xyXG5cclxuICAgIHRoaXMudHJlZURhdGEgPSB0aGlzLmFycmF5VXRpbGl0eVNlcnZpY2UuZ3JvdXBCeUZpZWxkKFxyXG4gICAgICB0aGlzLnF1ZXJpZXMsXHJcbiAgICAgICdncm91cCdcclxuICAgICk7XHJcbiAgfVxyXG5cclxuICAvKipcclxuICAgKiBGdW5jdGlvbiByZXNwb25zaWJsZSB0byBvcGVuIGRpYWxvZyB3aGVuIGEgcXVlcnkgaXMgY2xpY2tlZC5cclxuICAgKlxyXG4gICAqIEl0IGFsc28gbWFuYWdlcyB0aGUgZGF0YSB0aGF0IGNvbWVzIGZyb20gZGlhbG9nIHdoZW4gaXQgaXMgY2xvc2VkLlxyXG4gICAqXHJcbiAgICogQHBhcmFtIHtRdWVyeX0gcXVlcnkgdGhlIHNlbGVjdGVkIHF1ZXJ5XHJcbiAgICovXHJcbiAgcHVibGljIG9wZW5RdWVyeURpYWxvZyhxdWVyeTogUXVlcnkpOiB2b2lkIHtcclxuICAgIGxldCBxdWVyeUF1eDogUXVlcnk7XHJcblxyXG4gICAgaWYgKHF1ZXJ5KSB7XHJcbiAgICAgIHF1ZXJ5QXV4ID0gdGhpcy5vYmplY3RzVXRpbGl0eVNlcnZpY2UuY2xvbmVPYmplY3QocXVlcnkpO1xyXG4gICAgfSBlbHNlIHtcclxuICAgICAgcXVlcnlBdXggPSBuZXcgUXVlcnkoKTtcclxuICAgIH1cclxuXHJcbiAgICB0aGlzLnF1ZXJ5U2VsZWN0ZWREaWFsb2dSZWYgPSB0aGlzLmRpYWxvZ1NlcnZpY2Uub3BlbihcclxuICAgICAgUHJnUXVlcnlNb2RhbENvbXBvbmVudCxcclxuICAgICAge1xyXG4gICAgICAgIGhlYWRlcjogcXVlcnlBdXggPyBxdWVyeUF1eC5uYW1lIDogJ05ldyBRdWVyeScsXHJcbiAgICAgICAgd2lkdGg6ICc4MCUnLFxyXG4gICAgICAgIGhlaWdodDogJzkwdmgnLFxyXG4gICAgICAgIGJhc2VaSW5kZXg6IDEwMDAwLFxyXG4gICAgICAgIGRhdGE6IHtcclxuICAgICAgICAgIHF1ZXJ5OiBxdWVyeUF1eCxcclxuICAgICAgICAgIHZpZXdNb2RlOiBxdWVyeUF1eC5pZCA/IFZpZXdNb2RlLlJlYWQgOiBWaWV3TW9kZS5FZGl0LFxyXG4gICAgICAgIH0sXHJcbiAgICAgIH1cclxuICAgICk7XHJcbiAgICB0aGlzLnN1YnNjcmlwdGlvbi5wdXNoKFxyXG4gICAgICB0aGlzLnF1ZXJ5U2VsZWN0ZWREaWFsb2dSZWYub25DbG9zZS5zdWJzY3JpYmUoKG5ld0RhdGEpID0+IHtcclxuICAgICAgICB0aGlzLnN1YnNjcmlwdGlvbi5wdXNoKFxyXG4gICAgICAgICAgdGhpcy5xdWVyaWVzU2VydmljZVxyXG4gICAgICAgICAgICAuZ2V0UXVlcmllc0NoYW5nZWRPYnNlcnZhYmxlKClcclxuICAgICAgICAgICAgLnN1YnNjcmliZSgocXVlcmllcykgPT4ge1xyXG4gICAgICAgICAgICAgIGlmIChuZXdEYXRhICYmIHF1ZXJpZXMpIHtcclxuICAgICAgICAgICAgICAgIHRoaXMucXVlcmllcyA9IHRoaXMuYXJyYXlVdGlsaXR5U2VydmljZS5jbG9uZShxdWVyaWVzKTtcclxuICAgICAgICAgICAgICAgIHRoaXMuY3JlYXRlVHJlZU5vZGUoKTtcclxuICAgICAgICAgICAgICB9XHJcbiAgICAgICAgICAgICAgdGhpcy5xdWVyeVNlbGVjdGVkRGlhbG9nUmVmLmRlc3Ryb3koKTtcclxuICAgICAgICAgICAgfSlcclxuICAgICAgICApO1xyXG4gICAgICB9KVxyXG4gICAgKTtcclxuICB9XHJcbn1cclxuIiwiXHJcbjxkaXY+XHJcbiAgPHAtdHJlZVRhYmxlIFt2YWx1ZV09XCJ0cmVlRGF0YVwiID5cclxuICAgIDxuZy10ZW1wbGF0ZSBwVGVtcGxhdGU9XCJoZWFkZXJcIj5cclxuICAgICAgPHRyPlxyXG5cclxuICAgICAgICA8dGggc3R5bGU9XCJ3aWR0aDogMTByZW07dGV4dC1hbGlnbjogY2VudGVyXCIgKm5nRm9yPVwibGV0IGNvbCBvZiBxdWVyaWVzVGFibGVDb2x1bW5zXCIgW3R0U29ydGFibGVDb2x1bW5dPVwiY29sLmtleVwiIHBUb29sdGlwPVwie3tjb2wuYmFzZVBhdGhUcmFuc2xhdGlvbisnLicrY29sLmtleSsnLnRvb2x0aXAnfHRyYW5zbGF0ZSB9fVwiIHRvb2x0aXBQb3NpdGlvbj1cInRvcFwiPlxyXG5cclxuICAgICAgICAgICAge3sgY29sLmJhc2VQYXRoVHJhbnNsYXRpb24rXCIuXCIrY29sLmtleStcIi5sYWJlbFwifHRyYW5zbGF0ZSB9fVxyXG5cclxuICAgICAgICAgIDxwLXRyZWVUYWJsZVNvcnRJY29uIFtmaWVsZF09XCJjb2wua2V5XCI+PC9wLXRyZWVUYWJsZVNvcnRJY29uPlxyXG5cclxuICAgICAgICA8L3RoPlxyXG4gICAgICAgIDx0aCBzdHlsZT1cIndpZHRoOiAzcmVtO3RleHQtYWxpZ246IGNlbnRlclwiPlxyXG4gICAgICAgICAgPHAtYnV0dG9uIHBUb29sdGlwPVwie3sncXVlcmllcy5xdWVyaWVzLWxpc3QuYWN0aW9ucy5hZGQubGFiZWwnfHRyYW5zbGF0ZX19XCIgdG9vbHRpcFBvc2l0aW9uPVwiYm90dG9tXCIgKGNsaWNrKT1cIiBvcGVuUXVlcnlEaWFsb2cobnVsbClcIiB0eXBlPVwiYnV0dG9uXCIgaWNvbj1cInBpIHBpLXBsdXMtY2lyY2xlXCI+PC9wLWJ1dHRvbj5cclxuICAgICAgICA8L3RoPlxyXG4gICAgICA8L3RyPlxyXG4gICAgPC9uZy10ZW1wbGF0ZT5cclxuXHJcbiAgICA8bmctdGVtcGxhdGUgcFRlbXBsYXRlPVwiYm9keVwiIGxldC1yb3dOb2RlIGxldC1yb3dEYXRhPVwicm93RGF0YVwiPlxyXG4gICAgICA8dHJcclxuICAgICAgICBbdHRSb3ddPVwicm93Tm9kZVwiXHJcbiAgICAgICAgKGNsaWNrKT1cInJvd05vZGUubGV2ZWwgIT09IDAgJiYgb3BlblF1ZXJ5RGlhbG9nKHJvd0RhdGEpXCJcclxuICAgICAgICBbbmdDbGFzc109XCJ7IGNsaWNrYWJsZVJvdzogcm93Tm9kZS5sZXZlbCAhPT0gMCB9XCJcclxuICAgICAgPlxyXG4gICAgICAgIDx0ZCAqbmdGb3I9XCJsZXQgY29sIG9mIHF1ZXJpZXNUYWJsZUNvbHVtbnM7IGxldCBpID0gaW5kZXggO2xldCBsYXN0ID0gbGFzdCBcIiBbY29sU3Bhbl09XCJsYXN0PzI6MVwiIFtuZ1N0eWxlXT1cInsndGV4dC1hbGlnbic6KHJvd05vZGUubGV2ZWwhPT0wKT8gJ2NlbnRlcic6Jyd9XCIgPlxyXG4gICAgICAgICAgPHAtdHJlZVRhYmxlVG9nZ2xlclxyXG4gICAgICAgICAgICAqbmdJZj1cImkgPT0gMFwiXHJcbiAgICAgICAgICAgIFtyb3dOb2RlXT1cInJvd05vZGVcIlxyXG5cclxuICAgICAgICAgID48L3AtdHJlZVRhYmxlVG9nZ2xlcj5cclxuXHJcbiAgICAgICAgICA8bmctdGVtcGxhdGUgW25nSWZdPVwiY29sLmtleT09J2lzUmF3JyAmJiByb3dOb2RlLmxldmVsIT09MFwiPlxyXG4gICAgICAgICAgICB7e3Jvd0RhdGFbY29sLmtleV0gPyAnWWVzJzogJ05vJ319XHJcbiAgICAgICAgICA8L25nLXRlbXBsYXRlPlxyXG4gICAgICAgICAgPG5nLXRlbXBsYXRlICBbbmdJZl09XCJjb2wua2V5IT0naXNSYXcnICYmIHJvd05vZGUubGV2ZWwhPT0wXCIgPlxyXG4gICAgICAgICAgICB7e3Jvd0RhdGFbY29sLmtleV19fVxyXG4gICAgICAgICAgPC9uZy10ZW1wbGF0ZT5cclxuXHJcbiAgICAgICAgICA8bmctdGVtcGxhdGUgIFtuZ0lmXT1cImk9PTAmJnJvd05vZGUubGV2ZWw9PTBcIj5cclxuICAgICAgICAgICAge3sncXVlcmllcy5xdWVyaWVzLWxpc3QucXVlcmllcy1ncm91cC4nK3Jvd0RhdGFbY29sLmtleV0rJy5sYWJlbCd8dHJhbnNsYXRlIH19XHJcblxyXG4gICAgICAgICAgPC9uZy10ZW1wbGF0ZT5cclxuXHJcblxyXG5cclxuICAgICAgICA8L3RkPlxyXG4gICAgICA8L3RyPlxyXG4gICAgPC9uZy10ZW1wbGF0ZT5cclxuICA8L3AtdHJlZVRhYmxlPlxyXG5cclxuPC9kaXY+XHJcbiJdfQ==