import { Inject, Injectable } from '@angular/core';
import { TranslateService } from '@ngx-translate/core';
import { PrgError } from '../../../../core/models/error.model';
import { PrgSucess } from '../../../../core/models/success.model';
import { AbstractLookupTableService } from '../../../services/lookup-table/abstract-lookup-table.service';
import { MockDatabase } from '../../database.mock';
import * as i0 from "@angular/core";
import * as i1 from "../../../../core/services/is-loading-data/isloadingdata.service";
import * as i2 from "../../../../core/services/notifications/notifications.service";
import * as i3 from "../../../../core/services/utility/utility.service";
import * as i4 from "../../../../core/services/utility/objects-utility.service";
import * as i5 from "../../../../core/services/utility/array-utility.service";
import * as i6 from "@ngx-translate/core";
/**
 * Injectable
 */
/**
 * class MockLookupTableService
 */
export class MockLookupTableService extends AbstractLookupTableService {
    /**
     * Construtor
     *
     * @param isLoadingDataService
     * @param notificationsService
     * @param utilityService
     * @param objectsUtilityService
     */
    constructor(isLoadingDataService, notificationsService, utilityService, objectsUtilityService, translateService, arrayUtilityService) {
        super(translateService);
        this.isLoadingDataService = isLoadingDataService;
        this.notificationsService = notificationsService;
        this.utilityService = utilityService;
        this.objectsUtilityService = objectsUtilityService;
        this.arrayUtilityService = arrayUtilityService;
        /**
         * default time in seconds
         */
        this.sleepTimeSec = 0.5;
    }
    /**
     * this method returns all the lookuptables
     *
     * @returns  Promise<LookupTable[]
     */
    getAllLookupTableAsync() {
        return new Promise(async (resolve, reject) => {
            this.isLoadingDataService.show();
            await this.utilityService.sleepSecAsync(this.sleepTimeSec);
            this.isLoadingDataService.hide();
            if (this.isCacheValid()) {
                const lookupTables = this.getElementsList();
                resolve(lookupTables);
            }
            else {
            }
            const lookupTables = this.arrayUtilityService.clone(MockDatabase.LookupTable);
            if (lookupTables !== null && lookupTables.length > 0) {
                this.setToCache(lookupTables, 'name');
                resolve(lookupTables);
            }
            else {
                this.notificationsService.errorNotification(new PrgError({
                    titleKey: 'error',
                    detailKey: 'errorGetLookupTableList',
                }));
                reject(null);
            }
        });
    }
    /**
     * this method is responsible for saving a new lookupTable
     * or update an existing one
     *
     * @param lookupTable
     * @returns LookupTable
     */
    saveLookupTableAsync(lookupTable) {
        return new Promise(async (resolve, reject) => {
            await this.utilityService.sleepSecAsync(this.sleepTimeSec);
            if (lookupTable.id) {
                const lookupTableToUpdate = MockDatabase.LookupTable.find((ltAux) => ltAux.id === lookupTable.id);
                if (lookupTableToUpdate) {
                    lookupTableToUpdate.description = lookupTable.description;
                    lookupTableToUpdate.universalStateId = lookupTable.universalStateId;
                    resolve(this.objectsUtilityService.cloneObject(lookupTableToUpdate));
                    //set catch
                    this.setToCache(this.objectsUtilityService.cloneObject(lookupTableToUpdate), 'name');
                    this.notificationsService.successNotification(new PrgSucess({
                        titleKey: 'success',
                        detailKey: 'lookupTableSuccessAdded',
                    }));
                }
                else {
                    this.notificationsService.errorNotification(new PrgError({
                        titleKey: 'error',
                        detailKey: 'errorAddLookupTable',
                    }));
                    reject(null);
                }
            }
            else {
                const newLookupTable = {
                    id: this.utilityService.newGuid(),
                    universalStateId: lookupTable.universalStateId,
                    transactionId: null,
                    operationId: null,
                    name: lookupTable.name,
                    label: null,
                    createdBy: null,
                    modifiedBy: null,
                    createdOn: null,
                    modifiedOn: null,
                    description: lookupTable.description,
                    workspaceId: null,
                    items: null,
                };
                const lookupTableList = MockDatabase.LookupTable;
                lookupTableList.unshift(newLookupTable);
                this.setToCache(this.objectsUtilityService.cloneObject(newLookupTable), 'name');
                resolve(this.objectsUtilityService.cloneObject(newLookupTable));
                this.notificationsService.successNotification(new PrgSucess({
                    titleKey: 'success',
                    detailKey: 'lookupTableSuccessAdded',
                }));
            }
        });
    }
    /**
     * this method is responsible for adding or
     * updating a set of lookup tables
     *
     * @param lookupTables
     * @returns LookupTable[]
     */
    saveMultipleLookupTableAsync(lookupTables) {
        return new Promise(async (resolve, reject) => {
            await this.utilityService.sleepSecAsync(this.sleepTimeSec);
            let savedLookupTable = [];
            lookupTables.forEach((lt) => {
                if (lt.id) {
                    const lookupTableToUpdate = MockDatabase.LookupTable.find((ltAux) => ltAux.id === lt.id);
                    if (lookupTableToUpdate) {
                        lookupTableToUpdate.description = lt.description;
                        lookupTableToUpdate.universalStateId = lt.universalStateId;
                        savedLookupTable.push(lookupTableToUpdate);
                    }
                }
                else {
                    const newLookupTable = {
                        id: this.utilityService.newGuid(),
                        universalStateId: lt.universalStateId,
                        transactionId: null,
                        operationId: null,
                        name: lt.name,
                        label: null,
                        createdBy: null,
                        modifiedBy: null,
                        createdOn: null,
                        modifiedOn: null,
                        description: lt.description,
                        workspaceId: null,
                        items: null,
                    };
                    const lookupTableList = MockDatabase.LookupTable;
                    lookupTableList.unshift(newLookupTable);
                    savedLookupTable.push(newLookupTable);
                }
            });
            resolve(savedLookupTable);
            this.notificationsService.successNotification(new PrgSucess({
                titleKey: 'success',
                detailKey: 'sucessedsaveMultipleLookupTableAsync',
            }));
        });
    }
    /**
     * this method returns the items of a given lookup table
     *
     * @param id
     * @returns LookupTableItem[]
     */
    getLookupTableItemsByLookupTableIdAsync(id) {
        return new Promise(async (resolve, reject) => {
            this.isLoadingDataService.show();
            await this.utilityService.sleepSecAsync(this.sleepTimeSec);
            const lookupTableItems = MockDatabase.LookupTableItems.filter((item) => item.lookupTableId == id);
            resolve(lookupTableItems);
            this.isLoadingDataService.hide();
        });
    }
    /**
     * this method returns the items of a given lookup table
     *
     * @param id
     * @returns LookupTableItem[]
     */
    getLookupTableItemsByLookupTableNameAsync(name) {
        return new Promise(async (resolve, reject) => {
            this.isLoadingDataService.show();
            const lookupTable = this.getElementByKey(name);
            if (this.isCacheValid() &&
                lookupTable != null &&
                lookupTable.items != null) {
                resolve(lookupTable.items);
                this.isLoadingDataService.hide();
            }
            else {
                await this.utilityService.sleepSecAsync(this.sleepTimeSec);
                const lookupTable = MockDatabase.LookupTable.find((lookupTable) => lookupTable.name === name);
                const lookupTableItems = MockDatabase.LookupTableItems.filter((item) => item.lookupTableId == lookupTable.id);
                lookupTable.items = lookupTableItems;
                const lookupTableTranslated = await this.getTranslations(lookupTable);
                this.setToCache(lookupTableTranslated, 'name');
                resolve(lookupTableTranslated.items);
                this.isLoadingDataService.hide();
            }
        });
    }
    /**
     * this method is responsible for saving a new lookupTable
     * or update an existing one
     *
     * @param lookupTable
     * @returns LookupTable
     */
    saveLookupTableItemAsync(lookupTableItem) {
        return new Promise(async (resolve, reject) => {
            await this.utilityService.sleepSecAsync(this.sleepTimeSec);
            if (lookupTableItem.id) {
                const lookupTableItemToUpdate = MockDatabase.LookupTableItems.find((ltiAux) => ltiAux.id === lookupTableItem.id);
                if (lookupTableItemToUpdate) {
                    lookupTableItemToUpdate.description = lookupTableItem.description;
                    lookupTableItemToUpdate.universalStateId =
                        lookupTableItem.universalStateId;
                    resolve(this.objectsUtilityService.cloneObject(lookupTableItemToUpdate));
                    this.notificationsService.successNotification(new PrgSucess({
                        titleKey: 'success',
                        detailKey: 'lookupTableSuccessAdded',
                    }));
                }
                else {
                    this.notificationsService.errorNotification(new PrgError({
                        titleKey: 'error',
                        detailKey: 'errorAddLookupTable',
                    }));
                    reject(null);
                }
            }
            else {
                const newLookupTableItem = {
                    id: this.utilityService.newGuid(),
                    universalStateId: lookupTableItem.universalStateId,
                    transactionId: null,
                    operationId: null,
                    name: lookupTableItem.name,
                    label: null,
                    createdBy: null,
                    modifiedBy: null,
                    createdOn: null,
                    modifiedOn: null,
                    description: lookupTableItem.description,
                    workspaceId: null,
                    lookupTableId: lookupTableItem.lookupTableId,
                };
                const lookupTableList = MockDatabase.LookupTableItems;
                lookupTableList.unshift(newLookupTableItem);
                resolve(this.objectsUtilityService.cloneObject(newLookupTableItem));
                this.notificationsService.successNotification(new PrgSucess({
                    titleKey: 'success',
                    detailKey: 'lookupTableItemSuccessAdded',
                }));
            }
        });
    }
    /**
     * this method is responsible for adding or
     * updating a set of lookup tables items
     *
     * @param lookupTablesItems
     * @returns LookupTableItems[]
     */
    saveMultipleLookupTableItemsAsync(lookupTables) {
        return new Promise(async (resolve, reject) => {
            await this.utilityService.sleepSecAsync(this.sleepTimeSec);
            let savedLookupTableItems = [];
            lookupTables.forEach((lt) => {
                if (lt.id) {
                    const lookupTableItemToUpdate = MockDatabase.LookupTableItems.find((ltAux) => ltAux.id === lt.id);
                    if (lookupTableItemToUpdate) {
                        lookupTableItemToUpdate.description = lt.description;
                        lookupTableItemToUpdate.universalStateId = lt.universalStateId;
                        savedLookupTableItems.push(lookupTableItemToUpdate);
                    }
                }
                else {
                    const newLookupTableItem = {
                        id: this.utilityService.newGuid(),
                        universalStateId: lt.universalStateId,
                        transactionId: null,
                        operationId: null,
                        name: lt.name,
                        label: null,
                        createdBy: null,
                        modifiedBy: null,
                        createdOn: null,
                        modifiedOn: null,
                        description: lt.description,
                        workspaceId: null,
                        lookupTableId: lt.lookupTableId,
                    };
                    const lookupTableList = MockDatabase.LookupTableItems;
                    lookupTableList.unshift(newLookupTableItem);
                    savedLookupTableItems.push(newLookupTableItem);
                }
            });
            resolve(savedLookupTableItems);
            this.notificationsService.successNotification(new PrgSucess({
                titleKey: 'success',
                detailKey: 'sucessedsaveMultipleLookupTableAsync',
            }));
        });
    }
    /**
     * Get lookup table item by id
     * @param {string} lookUpTableItemId
     * @returns {Promise<LookupTableItem>}
     */
    getLookUpTableItemByIdAsync(lookUpTableItemId) {
        return new Promise(async (resolve, reject) => {
            this.isLoadingDataService.show();
            await this.utilityService.sleepSecAsync(this.sleepTimeSec);
            const lookUpTableItemsList = [...MockDatabase.LookupTableItems];
            this.isLoadingDataService.hide();
            let lookUpTableItem = lookUpTableItemsList.find((lookUpTableItem) => {
                return lookUpTableItem.id == lookUpTableItemId;
            });
            if (lookUpTableItem) {
                resolve(lookUpTableItem);
            }
            else {
                reject('');
            }
        });
    }
    /**
     *
     */
    getMultipleLookupTableByName(lookupTableNames) {
        return new Promise(async (resolve, reject) => {
            const lookupTablesList = [];
            const lookUpTables = this.arrayUtilityService.clone(MockDatabase.LookupTable);
            const lookUpTableItems = this.arrayUtilityService.clone(MockDatabase.LookupTableItems);
            if (this.isCacheValid()) {
                lookupTableNames.forEach((name) => {
                    let lookupTableAux = this.getElementByKey(name);
                    if (!lookupTableAux) {
                        lookupTableAux = lookUpTables.find((lt) => lt.name === name);
                        if (lookupTableAux) {
                            lookupTableAux.items = lookUpTableItems.filter((li) => li.lookupTableId === lookupTableAux.id);
                        }
                        //TODO: error not found lookuptable
                    }
                    else if (!lookupTableAux.items) {
                        lookupTableAux.items = lookUpTableItems.filter((li) => li.lookupTableId === lookupTableAux.id);
                    }
                    lookupTablesList.push(lookupTableAux);
                });
            }
            else {
                lookupTableNames.forEach(async (name) => {
                    const lookupTableAux = lookUpTables.find((lt) => lt.name === name);
                    if (lookupTableAux) {
                        lookupTableAux.items = lookUpTableItems.filter((li) => li.lookupTableId === lookupTableAux.id);
                        const lookupTableAuxTranslated = await this.getTranslations(lookupTableAux);
                        lookupTablesList.push(lookupTableAuxTranslated);
                    }
                });
            }
            resolve(lookupTablesList);
            this.setToCache(lookupTablesList, 'name');
        });
    }
}
MockLookupTableService.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "14.2.0", ngImport: i0, type: MockLookupTableService, deps: [{ token: i1.IsLoadingDataService }, { token: i2.NotificationsService }, { token: i3.UtilityService }, { token: i4.ObjectsUtilityService }, { token: TranslateService }, { token: i5.ArrayUtilityService }], target: i0.ɵɵFactoryTarget.Injectable });
MockLookupTableService.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "14.2.0", ngImport: i0, type: MockLookupTableService, providedIn: 'root' });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "14.2.0", ngImport: i0, type: MockLookupTableService, decorators: [{
            type: Injectable,
            args: [{
                    providedIn: 'root',
                }]
        }], ctorParameters: function () { return [{ type: i1.IsLoadingDataService }, { type: i2.NotificationsService }, { type: i3.UtilityService }, { type: i4.ObjectsUtilityService }, { type: i6.TranslateService, decorators: [{
                    type: Inject,
                    args: [TranslateService]
                }] }, { type: i5.ArrayUtilityService }]; } });
//# sourceMappingURL=data:application/json;base64,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