import { Inject, Injectable } from '@angular/core';
import { TranslateService } from '@ngx-translate/core';
import { PrgError } from '../../../../core/models/error.model';
import { PaginationResponse } from '../../../../core/models/pagination-response';
import { PrgSucess } from '../../../../core/models/success.model';
import { NotificationsService } from '../../../../core/services/notifications/notifications.service';
import { EntityTypeAttribute, EntityTypeOperation, EntityTypeProperty, } from '../../../models/entity-type.model';
import { AbstractEntityTypeService } from '../../../services/entity-types/abstract-entity-type.service';
import { MockDatabase } from '../../database.mock';
import * as i0 from "@angular/core";
import * as i1 from "../../../../core/services/utility/utility.service";
import * as i2 from "../../../../core/services/utility/objects-utility.service";
import * as i3 from "../../../../core/services/utility/array-utility.service";
import * as i4 from "../../../../core/services/is-loading-data/isloadingdata.service";
import * as i5 from "../../../../core/services/notifications/notifications.service";
import * as i6 from "@ngx-translate/core";
/**
 * injectable
 */
/**
 * class MockEntityTypesService
 */
export class MockEntityTypesService extends AbstractEntityTypeService {
    /**
     * constructor
     * @param utilityService
     * @param objectsUtilityService
     * @param notificationsService
     * @param arrayUtilityService
     * @param isLoadingData
     * @param translateService
     */
    constructor(utilityService, objectsUtilityService, notificationsService, arrayUtilityService, isLoadingData, translateService) {
        super(notificationsService, translateService);
        this.utilityService = utilityService;
        this.objectsUtilityService = objectsUtilityService;
        this.arrayUtilityService = arrayUtilityService;
        this.isLoadingData = isLoadingData;
        /**
         * default time in seconds
         */
        this.sleepTimeSec = 0.2;
    }
    /**
     * Get all Entity Types
     *
     * @Returns Entity types[]
     */
    getEntityTypeListAsync() {
        return new Promise(async (resolve, reject) => {
            if (this.isCacheValid()) {
                const values = this.getElementsList();
                resolve(values);
            }
            else {
                this.isLoadingData.show();
                await this.utilityService.sleepSecAsync(this.sleepTimeSec);
                const entityTypeList = this.arrayUtilityService.clone(MockDatabase.EntityTypes);
                this.isLoadingData.hide();
                if (entityTypeList !== null && entityTypeList.length > 0) {
                    const entityTypeTranslation = (await this.getTranslationsAsync(entityTypeList));
                    this.setToCache(entityTypeTranslation, 'name');
                    resolve(entityTypeTranslation);
                }
                else {
                    this.notificationsService.errorNotification(new PrgError({
                        titleKey: 'error',
                        detailKey: 'errorGetEntityTypeList',
                    }));
                    reject('Error');
                }
            }
        });
    }
    /**
     * Get entity type list with operation
     *
     *  @Returns Entity types[]
     */
    getEntityTypeListWithOperationAsync() {
        return new Promise(async (resolve, reject) => {
            await this.utilityService.sleepSecAsync(this.sleepTimeSec);
            const entityTypeList = this.arrayUtilityService.clone(MockDatabase.EntityTypes);
            this.isLoadingData.hide();
            if (entityTypeList !== null && entityTypeList.length > 0) {
                entityTypeList.forEach((entityType) => {
                    entityType.operations = this.arrayUtilityService
                        .clone(MockDatabase.EntityTypesOperations)
                        .filter((operation) => operation.entityTypeId === entityType.id);
                });
                const entityTypeTranslation = (await this.getTranslationsAsync(entityTypeList));
                this.setToCache(entityTypeTranslation, 'name');
                resolve(entityTypeTranslation);
            }
            else {
                this.notificationsService.errorNotification(new PrgError({
                    titleKey: 'error',
                    detailKey: 'errorGetEntityTypeList',
                }));
                reject('Error');
            }
        });
    }
    /**
     * save entity type
     * @param entityType
     * @return Entity type
     */
    saveEntityTypeAsync(entityType) {
        return new Promise(async (resolve, reject) => {
            if (entityType.id) {
                const entityTypeToUpdate = MockDatabase.EntityTypes.find((et) => et.id === entityType.id);
                if (entityTypeToUpdate) {
                    entityTypeToUpdate.guiSettings = entityType.guiSettings;
                    entityTypeToUpdate.group = entityType.group;
                    entityTypeToUpdate.universalStateId = entityType.universalStateId;
                    //set catch
                    this.setToCache(this.objectsUtilityService.cloneObject(entityTypeToUpdate), 'name');
                    resolve(this.objectsUtilityService.cloneObject(entityTypeToUpdate));
                    this.notificationsService.successNotification(new PrgSucess({
                        titleKey: 'success',
                        detailKey: 'entityTypeUpdatedSucess',
                    }));
                }
                else {
                    this.notificationsService.errorNotification(new PrgError({
                        titleKey: 'error',
                        detailKey: 'entityTypeUpdatedError',
                    }));
                    reject(null);
                }
            }
            else {
                const newEntityType = {
                    id: this.utilityService.newGuid(),
                    universalStateId: entityType.universalStateId,
                    transactionId: null,
                    operationId: null,
                    label: null,
                    name: entityType.name,
                    createdBy: null,
                    modifiedBy: null,
                    createdOn: null,
                    modifiedOn: null,
                    stateId: '1',
                    workspaceId: '1',
                    guiSettings: entityType.guiSettings,
                    group: entityType.group,
                    properties: null,
                    attributes: null,
                    operations: null,
                };
                const entityTypeList = MockDatabase.EntityTypes;
                entityTypeList.unshift(newEntityType);
                this.setToCache(this.objectsUtilityService.cloneObject(newEntityType), 'name');
                resolve(this.objectsUtilityService.cloneObject(newEntityType));
                this.notificationsService.successNotification(new PrgSucess({
                    titleKey: 'success',
                    detailKey: 'entityTypeSuccessAdded',
                }));
            }
        });
    }
    /**
     * get all operations of an entity
     * @param entityTypeId
     * @return EntityTypeOperation[]
     */
    getAllOperationsByEntityTypeIdAsync(entityTypeId) {
        return new Promise((resolve, reject) => {
            const entityTypeOperationList = this.arrayUtilityService
                .clone(MockDatabase.EntityTypesOperations)
                .filter((operation) => operation.entityTypeId === entityTypeId);
            if (entityTypeOperationList) {
                resolve(entityTypeOperationList);
            }
            else {
                this.notificationsService.errorNotification(new PrgError({
                    titleKey: 'error',
                    detailKey: 'errorGetEntityTypeOperationsList',
                }));
                reject('Error');
            }
        });
    }
    /**
     * add or update entity type operation
     * @param entityTypeOperation
     * @return EntityTypeOperation
     */
    saveEntityTypeOperationAsync(entityTypeOperation) {
        return new Promise(async (resolve, reject) => {
            if (entityTypeOperation.id) {
                const entityTypeOperationToUpdate = MockDatabase.EntityTypesOperations.find((eto) => eto.id === entityTypeOperation.id);
                if (entityTypeOperationToUpdate) {
                    entityTypeOperationToUpdate.httpVerb = entityTypeOperation.httpVerb;
                    entityTypeOperationToUpdate.universalStateId =
                        entityTypeOperation.universalStateId;
                    entityTypeOperationToUpdate.guiSettings =
                        entityTypeOperation.guiSettings;
                    entityTypeOperationToUpdate.imports = entityTypeOperation.imports;
                    entityTypeOperationToUpdate.override = entityTypeOperation.override;
                    entityTypeOperationToUpdate.universalStateId =
                        entityTypeOperation.universalStateId;
                    resolve(this.objectsUtilityService.cloneObject(entityTypeOperationToUpdate));
                    /*  this.setEntityTypeOperationByEntityTypeId(
                      this.objectsUtilityService.cloneObject(entityTypeOperationToUpdate)
                    );*/
                    this.notificationsService.successNotification(new PrgSucess({
                        titleKey: 'success',
                        detailKey: 'entityTypeOperationUpdatedSucess',
                    }));
                }
                else {
                    this.notificationsService.errorNotification(new PrgError({
                        titleKey: 'error',
                        detailKey: 'entityTypeOperationUpdatedError',
                    }));
                    reject(null);
                }
            }
            else {
                const newEntityTypeOperation = new EntityTypeOperation({
                    id: this.utilityService.newGuid(),
                    universalStateId: entityTypeOperation.universalStateId,
                    transactionId: null,
                    operationId: null,
                    name: entityTypeOperation.name,
                    createdBy: null,
                    modifiedBy: null,
                    createdOn: null,
                    modifiedOn: null,
                    workspaceId: '1',
                    entityTypeId: entityTypeOperation.entityTypeId,
                    guiSettings: entityTypeOperation.guiSettings,
                    override: entityTypeOperation.override,
                    imports: entityTypeOperation.imports,
                    sourceCode: 'source code',
                    httpVerb: entityTypeOperation.httpVerb,
                });
                const entityTypeOperationList = MockDatabase.EntityTypesOperations;
                entityTypeOperationList.unshift(newEntityTypeOperation);
                resolve(this.objectsUtilityService.cloneObject(newEntityTypeOperation));
                /*this.setEntityTypeOperationByEntityTypeId(
                  this.objectsUtilityService.cloneObject(newEntityTypeOperation)
                );*/
                this.notificationsService.successNotification(new PrgSucess({
                    titleKey: 'success',
                    detailKey: 'entityTypeOperationSuccessAdded',
                }));
            }
        });
    }
    /**
     * get all attributes of an entity
     * @param entityTypeId
     * @return EntityTypeAttribute[]
     */
    getAllAttributesByEntityTypeIdAsync(entityTypeId) {
        return new Promise((resolve, reject) => {
            const entityTypeAttributeList = this.arrayUtilityService
                .clone(MockDatabase.EntityTypeAttribute)
                .filter((operation) => operation.entityTypeId === entityTypeId);
            if (entityTypeAttributeList) {
                resolve(entityTypeAttributeList);
            }
            else {
                this.notificationsService.errorNotification(new PrgError({
                    titleKey: 'error',
                    detailKey: 'errorGetEntityTypeAttributeListList',
                }));
                reject('Error');
            }
        });
    }
    /**
     * add or update entity type attribute
     * @param entityTypeAttribute
     * @return EntityTypeAttribute
     */
    saveEntityTypeAttributeAsync(entityTypeAttribute) {
        return new Promise(async (resolve, reject) => {
            if (entityTypeAttribute.id) {
                const entityTypeAttributeToUpdate = MockDatabase.EntityTypeAttribute.find((eta) => eta.id === entityTypeAttribute.id);
                if (entityTypeAttributeToUpdate) {
                    Object.assign(entityTypeAttributeToUpdate, entityTypeAttribute);
                    resolve(this.objectsUtilityService.cloneObject(entityTypeAttributeToUpdate));
                    //TODO: UPDATE CACHE
                    this.notificationsService.successNotification(new PrgSucess({
                        titleKey: 'success',
                        detailKey: 'entityTypeAttributeUpdatedSucess',
                    }));
                }
                else {
                    this.notificationsService.errorNotification(new PrgError({
                        titleKey: 'error',
                        detailKey: 'entityTypeAttributeUpdatedError',
                    }));
                    reject(null);
                }
            }
            else {
                const newEntityTypeAttribute = new EntityTypeAttribute({
                    id: this.utilityService.newGuid(),
                    universalStateId: entityTypeAttribute.universalStateId,
                    transactionId: null,
                    operationId: null,
                    name: entityTypeAttribute.name,
                    createdBy: null,
                    label: null,
                    modifiedBy: null,
                    createdOn: null,
                    modifiedOn: null,
                    workspaceId: '1',
                    entityTypeId: entityTypeAttribute.entityTypeId,
                    guiSettings: entityTypeAttribute.guiSettings,
                    dataTypeId: entityTypeAttribute.dataTypeId,
                    order: entityTypeAttribute.order,
                    reference: entityTypeAttribute.reference,
                    isArray: entityTypeAttribute.isArray,
                });
                const entityTypeAttributeList = MockDatabase.EntityTypeAttribute;
                entityTypeAttributeList.unshift(newEntityTypeAttribute);
                resolve(this.objectsUtilityService.cloneObject(newEntityTypeAttribute));
                //TODO: Update cache
                this.notificationsService.successNotification(new PrgSucess({
                    titleKey: 'success',
                    detailKey: 'entityTypeAttributeSuccessAdded',
                }));
            }
        });
    }
    /**
     * get all properties of an entity
     * @param entityTypeId
     * @return EntityTypeProperty[]
     */
    getAllPropertiesByEntityTypeIdAsync(entityTypeId) {
        return new Promise((resolve, reject) => {
            const entityTypePropertynList = this.arrayUtilityService
                .clone(MockDatabase.EntityTypeProperty)
                .filter((operation) => operation.entityTypeId === entityTypeId);
            if (entityTypePropertynList) {
                resolve(entityTypePropertynList);
            }
            else {
                this.notificationsService.errorNotification(new PrgError({
                    titleKey: 'error',
                    detailKey: 'errorGetEntityTypePropertynListList',
                }));
                reject('Error');
            }
        });
    }
    /**
     * add or update entity type property
     * @param EntityTypeProperty
     * @return EntityTypeProperty
     */
    saveEntityTypePropertyAsync(entityTypeProperty) {
        return new Promise(async (resolve, reject) => {
            if (entityTypeProperty.id) {
                const entityTypePropertyToUpdate = MockDatabase.EntityTypeProperty.find((etp) => etp.id === entityTypeProperty.id);
                if (entityTypePropertyToUpdate) {
                    Object.assign(entityTypePropertyToUpdate, entityTypeProperty);
                    resolve(this.objectsUtilityService.cloneObject(entityTypePropertyToUpdate));
                    //TODO: UPDATE CACHE
                    this.notificationsService.successNotification(new PrgSucess({
                        titleKey: 'success',
                        detailKey: 'entityTypePropertyUpdatedSucess',
                    }));
                }
                else {
                    this.notificationsService.errorNotification(new PrgError({
                        titleKey: 'error',
                        detailKey: 'entityTypePropertyUpdatedError',
                    }));
                    reject(null);
                }
            }
            else {
                const newEntityTypeProperty = new EntityTypeProperty({
                    id: this.utilityService.newGuid(),
                    universalStateId: entityTypeProperty.universalStateId,
                    transactionId: null,
                    operationId: null,
                    name: entityTypeProperty.name,
                    createdBy: null,
                    modifiedBy: null,
                    label: null,
                    createdOn: null,
                    modifiedOn: null,
                    workspaceId: '1',
                    entityTypeId: entityTypeProperty.entityTypeId,
                    guiSettings: entityTypeProperty.guiSettings,
                    dataTypeId: entityTypeProperty.dataTypeId,
                    order: entityTypeProperty.order,
                    reference: entityTypeProperty.reference,
                    isArray: entityTypeProperty.isArray,
                });
                const entityTypePropertyList = MockDatabase.EntityTypeProperty;
                entityTypePropertyList.unshift(newEntityTypeProperty);
                resolve(this.objectsUtilityService.cloneObject(newEntityTypeProperty));
                //TODO: Update cache
                this.notificationsService.successNotification(new PrgSucess({
                    titleKey: 'success',
                    detailKey: 'entityTypePropertySuccessAdded',
                }));
            }
        });
    }
    /**
     * update entity type properties using patch logic
     * @param entityTypesPropertiesToPatch
     * @return EntityTypeProperty[]
     */
    updateEntityTypePropertiesAsync(entityTypesPropertiesToPatch) {
        return new Promise(async (resolve, reject) => {
            const updated = [];
            entityTypesPropertiesToPatch.forEach((fieldsToPatch, key) => {
                const entityTypePropertyToUpdate = MockDatabase.EntityTypeProperty.find((etp) => etp.id === key);
                updated.push(this.objectsUtilityService.cloneObject(Object.assign(entityTypePropertyToUpdate, fieldsToPatch)));
            });
            resolve(updated);
        });
    }
    /**
     * update entity type attributes using patch logic
     * @param entityTypesAttributesToPatch
     * @return EntityTypeAttribute[]
     */
    updateEntityTypeAttributesAsync(entityTypesAttributesToPatch) {
        return new Promise(async (resolve, reject) => {
            entityTypesAttributesToPatch.forEach((fieldsToPatch, key) => {
                const entityTypeAttributesToUpdate = MockDatabase.EntityTypeAttribute.find((eta) => eta.id === key);
                Object.assign(entityTypeAttributesToUpdate, fieldsToPatch);
            });
            resolve(this.arrayUtilityService.clone(MockDatabase.EntityTypeProperty));
        });
    }
    /**
     * this service returns all the data (properties, attributes and operations)
     * of a given entity type (by entity name)
     * @param entityTypeName
     */
    getAllEntityTypeDataByName(entityTypeName) {
        return new Promise(async (resolve, reject) => {
            let entityType = this.arrayUtilityService
                .clone(MockDatabase.EntityTypes)
                .find((entity) => entity.name === entityTypeName);
            if (entityType) {
                entityType.properties = this.arrayUtilityService
                    .clone(MockDatabase.EntityTypeProperty)
                    .filter((property) => property.entityTypeId === entityType.id);
                entityType.attributes = this.arrayUtilityService
                    .clone(MockDatabase.EntityTypeAttribute)
                    .filter((attribute) => attribute.entityTypeId === entityType.id);
                entityType.operations = this.arrayUtilityService
                    .clone(MockDatabase.EntityTypesOperations)
                    .filter((operation) => operation.entityTypeId === entityType.id);
                entityType = await this.getTranslationsAsync(entityType);
                resolve(entityType);
            }
        });
    }
    /**
     * this service returns an array of elements
     * of the entity sent as a parameter,
     *  based on the filter it receives as a parameter.
     * @param entityTypeName
     * @param FilterGroup
     */
    getEntityTypeElements(entityTypeName, filterGroup) {
        return new Promise(async (resolve, reject) => {
            let entityElementsAux = [];
            entityElementsAux = this.arrayUtilityService.clone(MockDatabase[entityTypeName]);
            if ('orderCollection' in filterGroup) {
                entityElementsAux = this.arrayUtilityService.sortByProperty(entityElementsAux, filterGroup.orderCollection[0].propertyName, filterGroup.orderCollection[0].orderType === 0 ? true : false);
            }
            if ('filterCollections' in filterGroup) {
                filterGroup.filterCollections.forEach((filter) => {
                    entityElementsAux = entityElementsAux.filter((element) => element[filter.propertyName].includes(filter.value));
                });
            }
            const qtdElement = entityElementsAux.length;
            entityElementsAux = entityElementsAux.slice(filterGroup.pageIndex * filterGroup.pageSize, (filterGroup.pageIndex + 1) * filterGroup.pageSize);
            resolve(new PaginationResponse({
                items: entityElementsAux,
                totalItems: qtdElement,
                currentPage: filterGroup.pageIndex,
            }));
        });
    }
    /**
     * Get all properties and attributes by entity type id
     * @param {string} entityTypeId
     * @returns {Promise<EntityTypeProperty[]>}
     */
    getAllPropertiesAndAttributesByEntityTypeIdAsync(entityTypeId) {
        return new Promise(async (resolve, reject) => {
            this.isLoadingData.show();
            await this.utilityService.sleepSecAsync(this.sleepTimeSec);
            this.isLoadingData.hide();
            let entityType = MockDatabase.EntityTypes.find((entityType) => entityType.id === entityTypeId);
            const entityTypeProperties = [...MockDatabase.EntityTypeProperty].filter((entityTypeProperty) => entityTypeProperty.entityTypeId === entityTypeId);
            const entityTypeAttributes = [...MockDatabase.EntityTypeAttribute].filter((entityTypeAttributes) => entityTypeAttributes.entityTypeId === entityTypeId);
            entityType.properties = [...entityTypeProperties];
            entityType.attributes = [...entityTypeAttributes];
            const entityTypeToTranslate = (await this.getTranslationsAsync(entityType));
            const entityTypePropertyAndAttributesList = [
                ...entityTypeToTranslate.properties,
                ...entityTypeToTranslate.attributes,
            ];
            this.isLoadingData.hide();
            if (entityTypePropertyAndAttributesList.length > 0) {
                resolve(entityTypePropertyAndAttributesList);
            }
            else {
                this.notificationsService.errorNotification(new PrgError({
                    titleKey: 'error',
                    detailKey: 'errorGetEntityTypePropertynListList',
                }));
                reject('Error');
            }
        });
    }
    /**
     * This service return a element from a entity type list by id
     * @param entityTypeName
     * @param elementId
     */
    getEntityTypeElementById(entityTypeName, elementId) {
        return new Promise(async (resolve, reject) => {
            const element = this.objectsUtilityService.cloneObject(MockDatabase[entityTypeName].find((element) => element.id === elementId));
            resolve(element);
        });
    }
    /**
     * this service excute an operation
     * @param entityName
     * @param operation
     * @param entity
     * @returns
     */
    executeAction(entityName, operation, entity) {
        return new Promise(async (resolve, reject) => {
            switch (operation.name) {
                case 'save':
                    resolve(this.saveOperationProduct(entity));
                    break;
                case 'delete':
                    resolve(this.deleteOperationProduct(entity));
                    break;
                case 'update':
                    resolve(this.updateOperationProduct(entity));
                    break;
                default:
                    this.notificationsService.errorNotification(new PrgError({
                        titleKey: 'error',
                        detailKey: 'operationNotFound',
                    }));
                    break;
            }
        });
    }
    /**
     * operations save product
     * @param element
     * @returns
     */
    saveOperationProduct(element) {
        if (element.id) {
            const productToUpdate = MockDatabase.product.find((product) => product.id === element.id);
            Object.assign(productToUpdate, element);
        }
        else {
            element['id'] = this.utilityService.newGuid();
            MockDatabase.product.unshift(element);
            return this.objectsUtilityService.cloneObject(element);
        }
    }
    /**
     * operation delte product
     * @param element
     * @returns
     */
    deleteOperationProduct(element) {
        const elementToDeleteIndex = MockDatabase.product.findIndex((product) => product.id === element.id);
        MockDatabase.product.splice(elementToDeleteIndex, 1);
        return element;
    }
    /**
     *  operation update product
     * @param element
     * @returns
     */
    updateOperationProduct(element) {
        const elementToUpdate = MockDatabase.product.find((product) => product.id === element.id);
        if (elementToUpdate) {
            Object.assign(elementToUpdate, element);
            return elementToUpdate;
        }
    }
}
MockEntityTypesService.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "14.2.0", ngImport: i0, type: MockEntityTypesService, deps: [{ token: i1.UtilityService }, { token: i2.ObjectsUtilityService }, { token: NotificationsService }, { token: i3.ArrayUtilityService }, { token: i4.IsLoadingDataService }, { token: TranslateService }], target: i0.ɵɵFactoryTarget.Injectable });
MockEntityTypesService.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "14.2.0", ngImport: i0, type: MockEntityTypesService, providedIn: 'root' });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "14.2.0", ngImport: i0, type: MockEntityTypesService, decorators: [{
            type: Injectable,
            args: [{
                    providedIn: 'root',
                }]
        }], ctorParameters: function () { return [{ type: i1.UtilityService }, { type: i2.ObjectsUtilityService }, { type: i5.NotificationsService, decorators: [{
                    type: Inject,
                    args: [NotificationsService]
                }] }, { type: i3.ArrayUtilityService }, { type: i4.IsLoadingDataService }, { type: i6.TranslateService, decorators: [{
                    type: Inject,
                    args: [TranslateService]
                }] }]; } });
//# sourceMappingURL=data:application/json;base64,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