import { Component } from '@angular/core';
import { ACTIONS_BASE_STRUCTURE } from '../../../../../core/components/dynamic-form/actions-default-structures/actions-base-structure';
import { BaseActionKey, } from '../../../../../core/components/dynamic-form/models/base-action';
import { ViewMode } from '../../../../../core/models/constants/view-mode.enum';
import { PrgError } from '../../../../../core/models/error.model';
import { ENTITY_TYPE_ATTRIBUTE_DYNAMIC_FORM } from '../../../../dynamic-form-struct/entity-type-attribute-dynamic-form';
import { ENTITY_TYPE_DYNAMIC_FORM } from '../../../../dynamic-form-struct/entity-type-dynamic-form';
import { ENTITY_TYPE_OPERATION_DYNAMIC_FORM } from '../../../../dynamic-form-struct/entity-type-operation-dynamic-form';
import { ENTITY_TYPE_PROPERTY_DYNAMIC_FORM } from '../../../../dynamic-form-struct/entity-type-property-dynamic-form';
import { EntityTypeAttribute, EntityTypeOperation, EntityTypeProperty, } from '../../../../models/entity-type.model';
import { MenuTabEntityType } from './models/menu-tab.module';
import * as i0 from "@angular/core";
import * as i1 from "../../../../services/entity-types/abstract-entity-type.service";
import * as i2 from "primeng/dynamicdialog";
import * as i3 from "../../../../../core/services/notifications/notifications.service";
import * as i4 from "../../../../../core/services/utility/array-utility.service";
import * as i5 from "../../../../../core/services/utility/objects-utility.service";
import * as i6 from "../../../../../core/services/utility/utility.service";
import * as i7 from "primeng/api";
import * as i8 from "@angular/common";
import * as i9 from "primeng/splitter";
import * as i10 from "primeng/tabmenu";
import * as i11 from "primeng/orderlist";
import * as i12 from "../../../../../core/components/dynamic-form/prg-dynamic-form.component";
import * as i13 from "@ngx-translate/core";
/**
 * entity type configuration component
 */
export class PrgEntityTypeConfigsComponent {
    /**
     * constructor
     * @param entityTypeService
     * @param config
     * @param ref
     * @param notificationsService
     * @param arrayUtilityService
     * @param objectsUtilityService
     * @param utilityService
     */
    constructor(entityTypeService, config, ref, notificationsService, arrayUtilityService, objectsUtilityService, utilityService) {
        this.entityTypeService = entityTypeService;
        this.config = config;
        this.ref = ref;
        this.notificationsService = notificationsService;
        this.arrayUtilityService = arrayUtilityService;
        this.objectsUtilityService = objectsUtilityService;
        this.utilityService = utilityService;
        /**
         * elements list (properties, attributes, operations)
         */
        this.elements = [];
        /**
         * element selected
         */
        this.elementsSelected = [];
        /**
         * entity type form fields
         */
        this.entityTypeDynamicFormFields = this.arrayUtilityService.clone(ENTITY_TYPE_DYNAMIC_FORM.fields);
        /**
         * entity type form actions
         */
        this.entityTypeDynamicFormActions = this.arrayUtilityService.clone(ENTITY_TYPE_DYNAMIC_FORM.actions);
        /**
         * elements form fields (default is fields from properties)
         */
        this.elementDynamicFormFields = this.arrayUtilityService.clone(ENTITY_TYPE_PROPERTY_DYNAMIC_FORM.fields);
        /**
         * elements form actions
         */
        this.elementDynamicFormActions = this.arrayUtilityService.clone(ACTIONS_BASE_STRUCTURE.actions);
        /**
         * entity type view mode
         */
        this.entityTypeViewMode = ViewMode.Read;
        /**
         * elements view mode
         */
        this.elementViewMode = ViewMode.Add;
        /**
         * diabled list
         */
        this.disabledList = false;
        /**
         * debounce time
         */
        this.debounceTime = 200;
        /**
         * this function is responsible for updating
         * the elements in the database when they have their order changed
         */
        this.reorderElements = this.utilityService.debounce(() => {
            let mapElements = new Map();
            for (let i = 0; i < this.elements.length; i++) {
                this.elements[i].order = i + 1;
                mapElements.set(this.elements[i].id, { order: this.elements[i].order });
            }
            switch (this.activeItem.type) {
                case MenuTabEntityType.EntityTypeProperty:
                    this.entityTypeService.updateEntityTypePropertiesAsync(mapElements);
                    break;
                case MenuTabEntityType.EntityTypeAttribute:
                    this.entityTypeService.updateEntityTypeAttributesAsync(mapElements);
                    break;
                default:
                    this.notificationsService.errorNotification(new PrgError({
                        titleKey: 'error',
                        detailKey: 'notUpdated',
                    }));
                    break;
            }
        }, this.debounceTime);
    }
    /**
     * ngOnInit
     */
    ngOnInit() {
        this.menuTabItems = [
            {
                label: 'properties',
                icon: 'pi pi-fw pi-pencil',
                type: MenuTabEntityType.EntityTypeProperty,
                disabled: false,
            },
            {
                label: 'attributes',
                icon: 'pi pi-fw pi-pencil',
                type: MenuTabEntityType.EntityTypeAttribute,
                disabled: false,
            },
            {
                label: 'operations',
                icon: 'pi pi-fw pi-pencil',
                type: MenuTabEntityType.EntityTypeOperation,
                disabled: false,
            },
        ];
        this.activeItem = this.menuTabItems[0];
        this.entityType = this.config.data.entityType;
        this.entityTypeViewMode = this.config.data.viewMode;
        if (!this.entityType.id) {
            this.menuTabsItemsManage(null);
        }
        this.loadElements();
    }
    /**
     * this function receives the event
     * when a new element is selected in the list
     * @param event
     * @returns
     */
    onSelectionChanged(event) {
        if (event.value == null) {
            return;
        }
        this.elementsSelected = [event.value[event.value.length - 1]];
        this.elementViewMode = ViewMode.Read;
    }
    /**
     * this function receives the output referring to the actions of an entity
     * type and in function of the received action performs the corresponding actions
     * @param event
     */
    async onEntityTypeActionOutputted(event) {
        this.menuTabsItemsManage(null);
        switch (event.action) {
            case BaseActionKey.Save:
                this.entityType = await this.entityTypeService.saveEntityTypeAsync(event.formEntity);
                this.menuTabsItemsManage(null, false);
                this.activeItem = this.menuTabItems[0];
                this.entityTypeViewMode = ViewMode.Read;
                break;
            case BaseActionKey.Cancel:
                this.entityTypeViewMode = ViewMode.Read;
                if (event.formEntity.id) {
                    this.entityType = this.objectsUtilityService.cloneObject(this.entityType);
                }
                else {
                    this.ref.close();
                }
                break;
            case BaseActionKey.Edit:
                this.entityTypeViewMode = ViewMode.Edit;
                break;
            default:
                break;
        }
    }
    /**
     * this function is responsible for receiving the changes from the selected tab and loading the respective data
     * @param event
     */
    tabChange(event) {
        this.activeItem = event;
        this.elementsSelected = [];
        this.elementViewMode = ViewMode.Add;
        this.loadElements();
    }
    /**
     * this function manages the status of the
     * tabs according to the parameters
     * it can activate or deactivate all tabs as well
     * as change the status of a single tab
     *
     * @param menuTabItem
     * @param status
     */
    menuTabsItemsManage(menuTabItem, status = true) {
        this.menuTabItems.forEach((item) => {
            item.disabled = status;
        });
        if (menuTabItem) {
            const item = this.menuTabItems.find((item) => (item.type = menuTabItem.type));
            item.disabled = true;
        }
        this.menuTabItems = this.arrayUtilityService.clone(this.menuTabItems);
        this.activeItem = this.menuTabItems.find((item) => item.type === this.activeItem.type);
    }
    /**
     * create empty element, depending on the tab selected
     */
    addNewElement() {
        let newElement = null;
        if (this.entityType && this.activeItem) {
            this.disabledList = true;
            switch (this.activeItem.type) {
                case MenuTabEntityType.EntityTypeProperty:
                    newElement = new EntityTypeProperty({
                        entityTypeId: this.entityType.id,
                        order: this.elements.length + 1,
                    });
                    break;
                case MenuTabEntityType.EntityTypeAttribute:
                    newElement = new EntityTypeAttribute({
                        entityTypeId: this.entityType.id,
                        order: this.elements.length + 1,
                    });
                    break;
                case MenuTabEntityType.EntityTypeOperation:
                    newElement = new EntityTypeOperation({
                        entityTypeId: this.entityType.id,
                    });
                    break;
                default:
                    this.notificationsService.errorNotification(new PrgError({
                        titleKey: 'error',
                        detailKey: 'cantFindEntity',
                    }));
                    break;
            }
        }
        // add element to elements to array
        this.elements.push(newElement);
        //this.elements = this.arrayUtilityService.clone(this.elements);
        this.elementsSelected = [this.elements[this.elements.length - 1]];
    }
    /**
     * this function loads the elements according to the tab selected,
     *  and changes the form's data
     */
    async loadElements() {
        if (this.entityType && this.activeItem) {
            switch (this.activeItem.type) {
                case MenuTabEntityType.EntityTypeProperty:
                    this.elementDynamicFormFields = this.arrayUtilityService.clone(ENTITY_TYPE_PROPERTY_DYNAMIC_FORM.fields);
                    this.elements = this.arrayUtilityService.sortByProperty(await this.entityTypeService.getAllPropertiesByEntityTypeIdAsync(this.entityType.id), 'order');
                    break;
                case MenuTabEntityType.EntityTypeAttribute:
                    this.elementDynamicFormFields = this.arrayUtilityService.clone(ENTITY_TYPE_ATTRIBUTE_DYNAMIC_FORM.fields);
                    this.elements = this.arrayUtilityService.sortByProperty(await this.entityTypeService.getAllAttributesByEntityTypeIdAsync(this.entityType.id), 'order');
                    break;
                case MenuTabEntityType.EntityTypeOperation:
                    this.elementDynamicFormFields = this.arrayUtilityService.clone(ENTITY_TYPE_OPERATION_DYNAMIC_FORM.fields);
                    this.elements =
                        await this.entityTypeService.getAllOperationsByEntityTypeIdAsync(this.entityType.id);
                    break;
                default:
                    this.notificationsService.errorNotification(new PrgError({
                        titleKey: 'error',
                        detailKey: 'cantAddNewElement',
                    }));
                    break;
            }
        }
    }
    /**
     * this function receives the output of an entity,
     * and treats each of the actions with the respective logic to be applied
     * @param event
     */
    onEntityActionOutputted(event) {
        switch (event.action) {
            case BaseActionKey.Save:
                this.saveNewElement(event.formEntity);
                this.menuTabsItemsManage(null, false);
                break;
            case BaseActionKey.Edit:
                this.disabledList = true;
                this.elementViewMode = ViewMode.Edit;
                this.menuTabsItemsManage(null, true);
                break;
            case BaseActionKey.Cancel:
                if (event.formEntity.id) {
                    this.elementsSelected = this.arrayUtilityService.clone(this.elementsSelected);
                    this.elementViewMode = ViewMode.Read;
                }
                else {
                    this.elementViewMode = ViewMode.Add;
                    this.elementsSelected = [];
                    this.elements.splice(this.elements.length - 1, 1);
                }
                this.disabledList = false;
                this.menuTabsItemsManage(null, false);
                break;
            case BaseActionKey.Add:
                this.menuTabsItemsManage(null, true);
                this.elementViewMode = ViewMode.Edit;
                this.addNewElement();
                break;
            default:
                this.notificationsService.errorNotification(new PrgError({
                    titleKey: 'error',
                    detailKey: 'actionNotFound',
                }));
                break;
        }
    }
    /**
     * this function is responsible for saving the new element added,
     * depending on the selected tab it calls the respective method
     * @param newElement
     */
    async saveNewElement(newElement) {
        let addedElement;
        switch (this.activeItem.type) {
            case MenuTabEntityType.EntityTypeProperty:
                addedElement = await this.entityTypeService.saveEntityTypePropertyAsync(newElement);
                break;
            case MenuTabEntityType.EntityTypeAttribute:
                addedElement =
                    await this.entityTypeService.saveEntityTypeAttributeAsync(newElement);
                break;
            case MenuTabEntityType.EntityTypeOperation:
                addedElement =
                    await this.entityTypeService.saveEntityTypeOperationAsync(newElement);
                break;
            default:
                this.notificationsService.errorNotification(new PrgError({
                    titleKey: 'error',
                    detailKey: 'tabNotFound',
                }));
                break;
        }
        if (!newElement.id) {
            this.elements.splice(this.elements.length - 1, 1);
        }
        this.elementViewMode = ViewMode.Read;
        this.elementsSelected = [];
        this.disabledList = false;
        await this.loadElements();
        const elementAux = this.elements.find((element) => element.id === addedElement.id);
        this.elementsSelected = [elementAux];
    }
}
PrgEntityTypeConfigsComponent.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "14.2.0", ngImport: i0, type: PrgEntityTypeConfigsComponent, deps: [{ token: i1.AbstractEntityTypeService }, { token: i2.DynamicDialogConfig }, { token: i2.DynamicDialogRef }, { token: i3.NotificationsService }, { token: i4.ArrayUtilityService }, { token: i5.ObjectsUtilityService }, { token: i6.UtilityService }], target: i0.ɵɵFactoryTarget.Component });
PrgEntityTypeConfigsComponent.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "14.0.0", version: "14.2.0", type: PrgEntityTypeConfigsComponent, selector: "prg-entity-type-configs", ngImport: i0, template: "<div>\r\n  <prg-dynamic-form\r\n    [fields]=\"entityTypeDynamicFormFields\"\r\n    [actions]=\"entityTypeDynamicFormActions\"\r\n    [entity]=\"entityType\"\r\n    [viewMode]=\"entityTypeViewMode\"\r\n    (actionOutput)=\"onEntityTypeActionOutputted($event)\"\r\n  ></prg-dynamic-form>\r\n\r\n  <p-tabMenu\r\n    [model]=\"menuTabItems\"\r\n    [activeItem]=\"activeItem\"\r\n    styleClass=\"p-tabmenu-nav\"\r\n  >\r\n    <ng-template pTemplate=\"item\" let-item let-i=\"index\">\r\n      <div (click)=\"!item.disabled && tabChange(item)\">\r\n        <i [class]=\"item.icon\"></i>\r\n        {{ 'entity-type.tabs.' + item.label + '.label' | translate }}\r\n      </div>\r\n    </ng-template>\r\n  </p-tabMenu>\r\n\r\n  <p-splitter\r\n    *ngIf=\"entityType.id\"\r\n    [style]=\"{ height: '300px' }\"\r\n    [panelSizes]=\"[30, 70]\"\r\n    [minSizes]=\"[20, 40]\"\r\n  >\r\n    <ng-template pTemplate>\r\n      <p-orderList\r\n        *ngIf=\"elements.length > 0\"\r\n        [value]=\"elements\"\r\n        [disabled]=\"disabledList\"\r\n        [selection]=\"elementsSelected\"\r\n        (onSelectionChange)=\"onSelectionChanged($event)\"\r\n        filterBy=\"name\"\r\n        [dragdrop]=\"true\"\r\n        styleClass=\"p-tabmenu-nav\"\r\n        (onReorder)=\"reorderElements($event)\"\r\n      >\r\n        <ng-template let-element pTemplate=\"list\">\r\n          <div>\r\n            <div>\r\n              {{\r\n                element.id != null\r\n                  ? ('entities.' +\r\n                      entityType.name +\r\n                      '.fields.' +\r\n                      element.name +\r\n                      '.label' | translate)\r\n                  : ('entity.new.' + activeItem.type.toLowerCase() | translate)\r\n              }}\r\n            </div>\r\n          </div>\r\n        </ng-template>\r\n      </p-orderList></ng-template\r\n    >\r\n    <ng-template pTemplate=\"dynamicForm\">\r\n      <div>\r\n        <prg-dynamic-form\r\n          [fields]=\"elementDynamicFormFields\"\r\n          [actions]=\"elementDynamicFormActions\"\r\n          [entity]=\"elementsSelected.length ? elementsSelected[0] : null\"\r\n          [viewMode]=\"elementViewMode\"\r\n          (actionOutput)=\"onEntityActionOutputted($event)\"\r\n        ></prg-dynamic-form>\r\n      </div>\r\n    </ng-template>\r\n  </p-splitter>\r\n</div>\r\n", styles: [":host ::ng-deep .p-tabmenu-nav{width:-moz-fit-content!important;width:fit-content!important;margin:auto!important}\n"], dependencies: [{ kind: "directive", type: i7.PrimeTemplate, selector: "[pTemplate]", inputs: ["type", "pTemplate"] }, { kind: "directive", type: i8.NgIf, selector: "[ngIf]", inputs: ["ngIf", "ngIfThen", "ngIfElse"] }, { kind: "component", type: i9.Splitter, selector: "p-splitter", inputs: ["styleClass", "panelStyleClass", "style", "panelStyle", "stateStorage", "stateKey", "layout", "gutterSize", "minSizes", "panelSizes"], outputs: ["onResizeEnd", "onResizeStart"] }, { kind: "component", type: i10.TabMenu, selector: "p-tabMenu", inputs: ["model", "activeItem", "scrollable", "popup", "style", "styleClass"] }, { kind: "component", type: i11.OrderList, selector: "p-orderList", inputs: ["header", "style", "styleClass", "listStyle", "responsive", "filterBy", "filterPlaceholder", "filterLocale", "metaKeySelection", "dragdrop", "controlsPosition", "ariaFilterLabel", "filterMatchMode", "breakpoint", "stripedRows", "disabled", "trackBy", "selection", "value"], outputs: ["selectionChange", "onReorder", "onSelectionChange", "onFilterEvent"] }, { kind: "component", type: i12.PrgDynamicFormComponent, selector: "prg-dynamic-form", inputs: ["context", "fields", "actions", "entity", "viewMode", "formGroupType"], outputs: ["actionOutput", "fieldChangeOutput"] }, { kind: "pipe", type: i13.TranslatePipe, name: "translate" }] });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "14.2.0", ngImport: i0, type: PrgEntityTypeConfigsComponent, decorators: [{
            type: Component,
            args: [{ selector: 'prg-entity-type-configs', template: "<div>\r\n  <prg-dynamic-form\r\n    [fields]=\"entityTypeDynamicFormFields\"\r\n    [actions]=\"entityTypeDynamicFormActions\"\r\n    [entity]=\"entityType\"\r\n    [viewMode]=\"entityTypeViewMode\"\r\n    (actionOutput)=\"onEntityTypeActionOutputted($event)\"\r\n  ></prg-dynamic-form>\r\n\r\n  <p-tabMenu\r\n    [model]=\"menuTabItems\"\r\n    [activeItem]=\"activeItem\"\r\n    styleClass=\"p-tabmenu-nav\"\r\n  >\r\n    <ng-template pTemplate=\"item\" let-item let-i=\"index\">\r\n      <div (click)=\"!item.disabled && tabChange(item)\">\r\n        <i [class]=\"item.icon\"></i>\r\n        {{ 'entity-type.tabs.' + item.label + '.label' | translate }}\r\n      </div>\r\n    </ng-template>\r\n  </p-tabMenu>\r\n\r\n  <p-splitter\r\n    *ngIf=\"entityType.id\"\r\n    [style]=\"{ height: '300px' }\"\r\n    [panelSizes]=\"[30, 70]\"\r\n    [minSizes]=\"[20, 40]\"\r\n  >\r\n    <ng-template pTemplate>\r\n      <p-orderList\r\n        *ngIf=\"elements.length > 0\"\r\n        [value]=\"elements\"\r\n        [disabled]=\"disabledList\"\r\n        [selection]=\"elementsSelected\"\r\n        (onSelectionChange)=\"onSelectionChanged($event)\"\r\n        filterBy=\"name\"\r\n        [dragdrop]=\"true\"\r\n        styleClass=\"p-tabmenu-nav\"\r\n        (onReorder)=\"reorderElements($event)\"\r\n      >\r\n        <ng-template let-element pTemplate=\"list\">\r\n          <div>\r\n            <div>\r\n              {{\r\n                element.id != null\r\n                  ? ('entities.' +\r\n                      entityType.name +\r\n                      '.fields.' +\r\n                      element.name +\r\n                      '.label' | translate)\r\n                  : ('entity.new.' + activeItem.type.toLowerCase() | translate)\r\n              }}\r\n            </div>\r\n          </div>\r\n        </ng-template>\r\n      </p-orderList></ng-template\r\n    >\r\n    <ng-template pTemplate=\"dynamicForm\">\r\n      <div>\r\n        <prg-dynamic-form\r\n          [fields]=\"elementDynamicFormFields\"\r\n          [actions]=\"elementDynamicFormActions\"\r\n          [entity]=\"elementsSelected.length ? elementsSelected[0] : null\"\r\n          [viewMode]=\"elementViewMode\"\r\n          (actionOutput)=\"onEntityActionOutputted($event)\"\r\n        ></prg-dynamic-form>\r\n      </div>\r\n    </ng-template>\r\n  </p-splitter>\r\n</div>\r\n", styles: [":host ::ng-deep .p-tabmenu-nav{width:-moz-fit-content!important;width:fit-content!important;margin:auto!important}\n"] }]
        }], ctorParameters: function () { return [{ type: i1.AbstractEntityTypeService }, { type: i2.DynamicDialogConfig }, { type: i2.DynamicDialogRef }, { type: i3.NotificationsService }, { type: i4.ArrayUtilityService }, { type: i5.ObjectsUtilityService }, { type: i6.UtilityService }]; } });
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicHJnLWVudGl0eS10eXBlLWNvbmZpZ3MuY29tcG9uZW50LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vLi4vLi4vLi4vLi4vLi4vLi4vLi4vcHJvamVjdHMvUHJnRnJhbWV3b3JrLkZFLkNvcmVMaWIvc3JjL2xpYi9tb2R1bGVzL2VudGl0eS10eXBlcy9jb21wb25lbnRzL2VudGl0eS10eXBlL2NvbXBvbmVudHMvZW50aXR5LXR5cGUtY29uZmlncy9wcmctZW50aXR5LXR5cGUtY29uZmlncy5jb21wb25lbnQudHMiLCIuLi8uLi8uLi8uLi8uLi8uLi8uLi8uLi8uLi8uLi9wcm9qZWN0cy9QcmdGcmFtZXdvcmsuRkUuQ29yZUxpYi9zcmMvbGliL21vZHVsZXMvZW50aXR5LXR5cGVzL2NvbXBvbmVudHMvZW50aXR5LXR5cGUvY29tcG9uZW50cy9lbnRpdHktdHlwZS1jb25maWdzL3ByZy1lbnRpdHktdHlwZS1jb25maWdzLmNvbXBvbmVudC5odG1sIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFBLE9BQU8sRUFBRSxTQUFTLEVBQVUsTUFBTSxlQUFlLENBQUM7QUFFbEQsT0FBTyxFQUFFLHNCQUFzQixFQUFFLE1BQU0sK0ZBQStGLENBQUM7QUFDdkksT0FBTyxFQUVMLGFBQWEsR0FFZCxNQUFNLGdFQUFnRSxDQUFDO0FBRXhFLE9BQU8sRUFBRSxRQUFRLEVBQUUsTUFBTSxxREFBcUQsQ0FBQztBQUMvRSxPQUFPLEVBQUUsUUFBUSxFQUFFLE1BQU0sd0NBQXdDLENBQUM7QUFLbEUsT0FBTyxFQUFFLGtDQUFrQyxFQUFFLE1BQU0sb0VBQW9FLENBQUM7QUFDeEgsT0FBTyxFQUFFLHdCQUF3QixFQUFFLE1BQU0sMERBQTBELENBQUM7QUFDcEcsT0FBTyxFQUFFLGtDQUFrQyxFQUFFLE1BQU0sb0VBQW9FLENBQUM7QUFDeEgsT0FBTyxFQUFFLGlDQUFpQyxFQUFFLE1BQU0sbUVBQW1FLENBQUM7QUFDdEgsT0FBTyxFQUVMLG1CQUFtQixFQUNuQixtQkFBbUIsRUFDbkIsa0JBQWtCLEdBQ25CLE1BQU0sc0NBQXNDLENBQUM7QUFFOUMsT0FBTyxFQUFXLGlCQUFpQixFQUFFLE1BQU0sMEJBQTBCLENBQUM7Ozs7Ozs7Ozs7Ozs7OztBQUV0RTs7R0FFRztBQU1ILE1BQU0sT0FBTyw2QkFBNkI7SUFDeEM7Ozs7Ozs7OztPQVNHO0lBQ0gsWUFDVSxpQkFBNEMsRUFDNUMsTUFBMkIsRUFDM0IsR0FBcUIsRUFDckIsb0JBQTBDLEVBQzFDLG1CQUF3QyxFQUN4QyxxQkFBNEMsRUFDNUMsY0FBOEI7UUFOOUIsc0JBQWlCLEdBQWpCLGlCQUFpQixDQUEyQjtRQUM1QyxXQUFNLEdBQU4sTUFBTSxDQUFxQjtRQUMzQixRQUFHLEdBQUgsR0FBRyxDQUFrQjtRQUNyQix5QkFBb0IsR0FBcEIsb0JBQW9CLENBQXNCO1FBQzFDLHdCQUFtQixHQUFuQixtQkFBbUIsQ0FBcUI7UUFDeEMsMEJBQXFCLEdBQXJCLHFCQUFxQixDQUF1QjtRQUM1QyxtQkFBYyxHQUFkLGNBQWMsQ0FBZ0I7UUFpQnhDOztXQUVHO1FBQ0ksYUFBUSxHQUFVLEVBQUUsQ0FBQztRQUU1Qjs7V0FFRztRQUNJLHFCQUFnQixHQUFVLEVBQUUsQ0FBQztRQUVwQzs7V0FFRztRQUNJLGdDQUEyQixHQUNoQyxJQUFJLENBQUMsbUJBQW1CLENBQUMsS0FBSyxDQUFDLHdCQUF3QixDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQ2xFOztXQUVHO1FBQ0ksaUNBQTRCLEdBQ2pDLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLENBQUMsd0JBQXdCLENBQUMsT0FBTyxDQUFDLENBQUM7UUFFbkU7O1dBRUc7UUFDSSw2QkFBd0IsR0FBZ0IsSUFBSSxDQUFDLG1CQUFtQixDQUFDLEtBQUssQ0FDM0UsaUNBQWlDLENBQUMsTUFBTSxDQUN6QyxDQUFDO1FBRUY7O1dBRUc7UUFDSSw4QkFBeUIsR0FDOUIsSUFBSSxDQUFDLG1CQUFtQixDQUFDLEtBQUssQ0FBQyxzQkFBc0IsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUVqRTs7V0FFRztRQUNJLHVCQUFrQixHQUFhLFFBQVEsQ0FBQyxJQUFJLENBQUM7UUFDcEQ7O1dBRUc7UUFDSSxvQkFBZSxHQUFhLFFBQVEsQ0FBQyxHQUFHLENBQUM7UUFFaEQ7O1dBRUc7UUFDSSxpQkFBWSxHQUFZLEtBQUssQ0FBQztRQUVyQzs7V0FFRztRQUNLLGlCQUFZLEdBQVcsR0FBRyxDQUFDO1FBMlRuQzs7O1dBR0c7UUFDSSxvQkFBZSxHQUFHLElBQUksQ0FBQyxjQUFjLENBQUMsUUFBUSxDQUFDLEdBQUcsRUFBRTtZQUN6RCxJQUFJLFdBQVcsR0FBRyxJQUFJLEdBQUcsRUFBZSxDQUFDO1lBRXpDLEtBQUssSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsRUFBRTtnQkFDN0MsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLEdBQUcsQ0FBQyxHQUFHLENBQUMsQ0FBQztnQkFDL0IsV0FBVyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDLEtBQUssRUFBRSxDQUFDLENBQUM7YUFDekU7WUFFRCxRQUFRLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxFQUFFO2dCQUM1QixLQUFLLGlCQUFpQixDQUFDLGtCQUFrQjtvQkFDdkMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLCtCQUErQixDQUFDLFdBQVcsQ0FBQyxDQUFDO29CQUNwRSxNQUFNO2dCQUNSLEtBQUssaUJBQWlCLENBQUMsbUJBQW1CO29CQUN4QyxJQUFJLENBQUMsaUJBQWlCLENBQUMsK0JBQStCLENBQUMsV0FBVyxDQUFDLENBQUM7b0JBQ3BFLE1BQU07Z0JBRVI7b0JBQ0UsSUFBSSxDQUFDLG9CQUFvQixDQUFDLGlCQUFpQixDQUN6QyxJQUFJLFFBQVEsQ0FBQzt3QkFDWCxRQUFRLEVBQUUsT0FBTzt3QkFDakIsU0FBUyxFQUFFLFlBQVk7cUJBQ3hCLENBQUMsQ0FDSCxDQUFDO29CQUNGLE1BQU07YUFDVDtRQUNILENBQUMsRUFBRSxJQUFJLENBQUMsWUFBWSxDQUFDLENBQUM7SUEzWm5CLENBQUM7SUFxRUo7O09BRUc7SUFDSCxRQUFRO1FBQ04sSUFBSSxDQUFDLFlBQVksR0FBRztZQUNsQjtnQkFDRSxLQUFLLEVBQUUsWUFBWTtnQkFDbkIsSUFBSSxFQUFFLG9CQUFvQjtnQkFDMUIsSUFBSSxFQUFFLGlCQUFpQixDQUFDLGtCQUFrQjtnQkFDMUMsUUFBUSxFQUFFLEtBQUs7YUFDaEI7WUFDRDtnQkFDRSxLQUFLLEVBQUUsWUFBWTtnQkFDbkIsSUFBSSxFQUFFLG9CQUFvQjtnQkFDMUIsSUFBSSxFQUFFLGlCQUFpQixDQUFDLG1CQUFtQjtnQkFDM0MsUUFBUSxFQUFFLEtBQUs7YUFDaEI7WUFDRDtnQkFDRSxLQUFLLEVBQUUsWUFBWTtnQkFDbkIsSUFBSSxFQUFFLG9CQUFvQjtnQkFDMUIsSUFBSSxFQUFFLGlCQUFpQixDQUFDLG1CQUFtQjtnQkFDM0MsUUFBUSxFQUFFLEtBQUs7YUFDaEI7U0FDRixDQUFDO1FBRUYsSUFBSSxDQUFDLFVBQVUsR0FBRyxJQUFJLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBRXZDLElBQUksQ0FBQyxVQUFVLEdBQUcsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDO1FBQzlDLElBQUksQ0FBQyxrQkFBa0IsR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUM7UUFDcEQsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsRUFBRSxFQUFFO1lBQ3ZCLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUNoQztRQUNELElBQUksQ0FBQyxZQUFZLEVBQUUsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSSxrQkFBa0IsQ0FBQyxLQUFVO1FBQ2xDLElBQUksS0FBSyxDQUFDLEtBQUssSUFBSSxJQUFJLEVBQUU7WUFDdkIsT0FBTztTQUNSO1FBQ0QsSUFBSSxDQUFDLGdCQUFnQixHQUFHLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQzlELElBQUksQ0FBQyxlQUFlLEdBQUcsUUFBUSxDQUFDLElBQUksQ0FBQztJQUN2QyxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLEtBQUssQ0FBQywyQkFBMkIsQ0FBQyxLQUE4QjtRQUNyRSxJQUFJLENBQUMsbUJBQW1CLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDL0IsUUFBUSxLQUFLLENBQUMsTUFBTSxFQUFFO1lBQ3BCLEtBQUssYUFBYSxDQUFDLElBQUk7Z0JBQ3JCLElBQUksQ0FBQyxVQUFVLEdBQUcsTUFBTSxJQUFJLENBQUMsaUJBQWlCLENBQUMsbUJBQW1CLENBQ2hFLEtBQUssQ0FBQyxVQUFVLENBQ2pCLENBQUM7Z0JBQ0YsSUFBSSxDQUFDLG1CQUFtQixDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsQ0FBQztnQkFDdEMsSUFBSSxDQUFDLFVBQVUsR0FBRyxJQUFJLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUN2QyxJQUFJLENBQUMsa0JBQWtCLEdBQUcsUUFBUSxDQUFDLElBQUksQ0FBQztnQkFFeEMsTUFBTTtZQUNSLEtBQUssYUFBYSxDQUFDLE1BQU07Z0JBQ3ZCLElBQUksQ0FBQyxrQkFBa0IsR0FBRyxRQUFRLENBQUMsSUFBSSxDQUFDO2dCQUN4QyxJQUFJLEtBQUssQ0FBQyxVQUFVLENBQUMsRUFBRSxFQUFFO29CQUN2QixJQUFJLENBQUMsVUFBVSxHQUFHLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxXQUFXLENBQ3RELElBQUksQ0FBQyxVQUFVLENBQ2hCLENBQUM7aUJBQ0g7cUJBQU07b0JBQ0wsSUFBSSxDQUFDLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztpQkFDbEI7Z0JBQ0QsTUFBTTtZQUVSLEtBQUssYUFBYSxDQUFDLElBQUk7Z0JBQ3JCLElBQUksQ0FBQyxrQkFBa0IsR0FBRyxRQUFRLENBQUMsSUFBSSxDQUFDO2dCQUN4QyxNQUFNO1lBQ1I7Z0JBQ0UsTUFBTTtTQUNUO0lBQ0gsQ0FBQztJQUVEOzs7T0FHRztJQUNJLFNBQVMsQ0FBQyxLQUFjO1FBQzdCLElBQUksQ0FBQyxVQUFVLEdBQUcsS0FBSyxDQUFDO1FBQ3hCLElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxFQUFFLENBQUM7UUFDM0IsSUFBSSxDQUFDLGVBQWUsR0FBRyxRQUFRLENBQUMsR0FBRyxDQUFDO1FBQ3BDLElBQUksQ0FBQyxZQUFZLEVBQUUsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7O09BUUc7SUFDSyxtQkFBbUIsQ0FBQyxXQUFvQixFQUFFLFNBQWtCLElBQUk7UUFDdEUsSUFBSSxDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBRTtZQUNqQyxJQUFJLENBQUMsUUFBUSxHQUFHLE1BQU0sQ0FBQztRQUN6QixDQUFDLENBQUMsQ0FBQztRQUNILElBQUksV0FBVyxFQUFFO1lBQ2YsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQ2pDLENBQUMsSUFBSSxFQUFFLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLEdBQUcsV0FBVyxDQUFDLElBQUksQ0FBQyxDQUN6QyxDQUFDO1lBQ0YsSUFBSSxDQUFDLFFBQVEsR0FBRyxJQUFJLENBQUM7U0FDdEI7UUFDRCxJQUFJLENBQUMsWUFBWSxHQUFHLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDO1FBQ3RFLElBQUksQ0FBQyxVQUFVLEdBQUcsSUFBSSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQ3RDLENBQUMsSUFBSSxFQUFFLEVBQUUsQ0FBQyxJQUFJLENBQUMsSUFBSSxLQUFLLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUM3QyxDQUFDO0lBQ0osQ0FBQztJQUVEOztPQUVHO0lBQ0ssYUFBYTtRQUNuQixJQUFJLFVBQVUsR0FHWSxJQUFJLENBQUM7UUFDL0IsSUFBSSxJQUFJLENBQUMsVUFBVSxJQUFJLElBQUksQ0FBQyxVQUFVLEVBQUU7WUFDdEMsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLENBQUM7WUFDekIsUUFBUSxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksRUFBRTtnQkFDNUIsS0FBSyxpQkFBaUIsQ0FBQyxrQkFBa0I7b0JBQ3ZDLFVBQVUsR0FBRyxJQUFJLGtCQUFrQixDQUFDO3dCQUNsQyxZQUFZLEVBQUUsSUFBSSxDQUFDLFVBQVUsQ0FBQyxFQUFFO3dCQUNoQyxLQUFLLEVBQUUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLEdBQUcsQ0FBQztxQkFDaEMsQ0FBQyxDQUFDO29CQUNILE1BQU07Z0JBQ1IsS0FBSyxpQkFBaUIsQ0FBQyxtQkFBbUI7b0JBQ3hDLFVBQVUsR0FBRyxJQUFJLG1CQUFtQixDQUFDO3dCQUNuQyxZQUFZLEVBQUUsSUFBSSxDQUFDLFVBQVUsQ0FBQyxFQUFFO3dCQUNoQyxLQUFLLEVBQUUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLEdBQUcsQ0FBQztxQkFDaEMsQ0FBQyxDQUFDO29CQUNILE1BQU07Z0JBQ1IsS0FBSyxpQkFBaUIsQ0FBQyxtQkFBbUI7b0JBQ3hDLFVBQVUsR0FBRyxJQUFJLG1CQUFtQixDQUFDO3dCQUNuQyxZQUFZLEVBQUUsSUFBSSxDQUFDLFVBQVUsQ0FBQyxFQUFFO3FCQUNqQyxDQUFDLENBQUM7b0JBQ0gsTUFBTTtnQkFFUjtvQkFDRSxJQUFJLENBQUMsb0JBQW9CLENBQUMsaUJBQWlCLENBQ3pDLElBQUksUUFBUSxDQUFDO3dCQUNYLFFBQVEsRUFBRSxPQUFPO3dCQUNqQixTQUFTLEVBQUUsZ0JBQWdCO3FCQUM1QixDQUFDLENBQ0gsQ0FBQztvQkFDRixNQUFNO2FBQ1Q7U0FDRjtRQUNELG1DQUFtQztRQUNuQyxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUMvQixnRUFBZ0U7UUFDaEUsSUFBSSxDQUFDLGdCQUFnQixHQUFHLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQ3BFLENBQUM7SUFFRDs7O09BR0c7SUFDSyxLQUFLLENBQUMsWUFBWTtRQUN4QixJQUFJLElBQUksQ0FBQyxVQUFVLElBQUksSUFBSSxDQUFDLFVBQVUsRUFBRTtZQUN0QyxRQUFRLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxFQUFFO2dCQUM1QixLQUFLLGlCQUFpQixDQUFDLGtCQUFrQjtvQkFDdkMsSUFBSSxDQUFDLHdCQUF3QixHQUFHLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLENBQzVELGlDQUFpQyxDQUFDLE1BQU0sQ0FDekMsQ0FBQztvQkFDRixJQUFJLENBQUMsUUFBUSxHQUFHLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxjQUFjLENBQ3JELE1BQU0sSUFBSSxDQUFDLGlCQUFpQixDQUFDLG1DQUFtQyxDQUM5RCxJQUFJLENBQUMsVUFBVSxDQUFDLEVBQUUsQ0FDbkIsRUFDRCxPQUFPLENBQ1IsQ0FBQztvQkFDRixNQUFNO2dCQUNSLEtBQUssaUJBQWlCLENBQUMsbUJBQW1CO29CQUN4QyxJQUFJLENBQUMsd0JBQXdCLEdBQUcsSUFBSSxDQUFDLG1CQUFtQixDQUFDLEtBQUssQ0FDNUQsa0NBQWtDLENBQUMsTUFBTSxDQUMxQyxDQUFDO29CQUVGLElBQUksQ0FBQyxRQUFRLEdBQUcsSUFBSSxDQUFDLG1CQUFtQixDQUFDLGNBQWMsQ0FDckQsTUFBTSxJQUFJLENBQUMsaUJBQWlCLENBQUMsbUNBQW1DLENBQzlELElBQUksQ0FBQyxVQUFVLENBQUMsRUFBRSxDQUNuQixFQUNELE9BQU8sQ0FDUixDQUFDO29CQUNGLE1BQU07Z0JBQ1IsS0FBSyxpQkFBaUIsQ0FBQyxtQkFBbUI7b0JBQ3hDLElBQUksQ0FBQyx3QkFBd0IsR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQUMsS0FBSyxDQUM1RCxrQ0FBa0MsQ0FBQyxNQUFNLENBQzFDLENBQUM7b0JBQ0YsSUFBSSxDQUFDLFFBQVE7d0JBQ1gsTUFBTSxJQUFJLENBQUMsaUJBQWlCLENBQUMsbUNBQW1DLENBQzlELElBQUksQ0FBQyxVQUFVLENBQUMsRUFBRSxDQUNuQixDQUFDO29CQUNKLE1BQU07Z0JBRVI7b0JBQ0UsSUFBSSxDQUFDLG9CQUFvQixDQUFDLGlCQUFpQixDQUN6QyxJQUFJLFFBQVEsQ0FBQzt3QkFDWCxRQUFRLEVBQUUsT0FBTzt3QkFDakIsU0FBUyxFQUFFLG1CQUFtQjtxQkFDL0IsQ0FBQyxDQUNILENBQUM7b0JBQ0YsTUFBTTthQUNUO1NBQ0Y7SUFDSCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLHVCQUF1QixDQUFDLEtBQThCO1FBQzNELFFBQVEsS0FBSyxDQUFDLE1BQU0sRUFBRTtZQUNwQixLQUFLLGFBQWEsQ0FBQyxJQUFJO2dCQUNyQixJQUFJLENBQUMsY0FBYyxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQztnQkFDdEMsSUFBSSxDQUFDLG1CQUFtQixDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsQ0FBQztnQkFDdEMsTUFBTTtZQUNSLEtBQUssYUFBYSxDQUFDLElBQUk7Z0JBQ3JCLElBQUksQ0FBQyxZQUFZLEdBQUcsSUFBSSxDQUFDO2dCQUN6QixJQUFJLENBQUMsZUFBZSxHQUFHLFFBQVEsQ0FBQyxJQUFJLENBQUM7Z0JBQ3JDLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLENBQUM7Z0JBQ3JDLE1BQU07WUFDUixLQUFLLGFBQWEsQ0FBQyxNQUFNO2dCQUN2QixJQUFJLEtBQUssQ0FBQyxVQUFVLENBQUMsRUFBRSxFQUFFO29CQUN2QixJQUFJLENBQUMsZ0JBQWdCLEdBQUcsSUFBSSxDQUFDLG1CQUFtQixDQUFDLEtBQUssQ0FDcEQsSUFBSSxDQUFDLGdCQUFnQixDQUN0QixDQUFDO29CQUNGLElBQUksQ0FBQyxlQUFlLEdBQUcsUUFBUSxDQUFDLElBQUksQ0FBQztpQkFDdEM7cUJBQU07b0JBQ0wsSUFBSSxDQUFDLGVBQWUsR0FBRyxRQUFRLENBQUMsR0FBRyxDQUFDO29CQUNwQyxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsRUFBRSxDQUFDO29CQUMzQixJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7aUJBQ25EO2dCQUNELElBQUksQ0FBQyxZQUFZLEdBQUcsS0FBSyxDQUFDO2dCQUMxQixJQUFJLENBQUMsbUJBQW1CLENBQUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxDQUFDO2dCQUV0QyxNQUFNO1lBRVIsS0FBSyxhQUFhLENBQUMsR0FBRztnQkFDcEIsSUFBSSxDQUFDLG1CQUFtQixDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsQ0FBQztnQkFDckMsSUFBSSxDQUFDLGVBQWUsR0FBRyxRQUFRLENBQUMsSUFBSSxDQUFDO2dCQUNyQyxJQUFJLENBQUMsYUFBYSxFQUFFLENBQUM7Z0JBQ3JCLE1BQU07WUFFUjtnQkFDRSxJQUFJLENBQUMsb0JBQW9CLENBQUMsaUJBQWlCLENBQ3pDLElBQUksUUFBUSxDQUFDO29CQUNYLFFBQVEsRUFBRSxPQUFPO29CQUNqQixTQUFTLEVBQUUsZ0JBQWdCO2lCQUM1QixDQUFDLENBQ0gsQ0FBQztnQkFDRixNQUFNO1NBQ1Q7SUFDSCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLEtBQUssQ0FBQyxjQUFjLENBQUMsVUFBZTtRQUMxQyxJQUFJLFlBQWlCLENBQUM7UUFDdEIsUUFBUSxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksRUFBRTtZQUM1QixLQUFLLGlCQUFpQixDQUFDLGtCQUFrQjtnQkFDdkMsWUFBWSxHQUFHLE1BQU0sSUFBSSxDQUFDLGlCQUFpQixDQUFDLDJCQUEyQixDQUNyRSxVQUFVLENBQ1gsQ0FBQztnQkFDRixNQUFNO1lBQ1IsS0FBSyxpQkFBaUIsQ0FBQyxtQkFBbUI7Z0JBQ3hDLFlBQVk7b0JBQ1YsTUFBTSxJQUFJLENBQUMsaUJBQWlCLENBQUMsNEJBQTRCLENBQUMsVUFBVSxDQUFDLENBQUM7Z0JBQ3hFLE1BQU07WUFDUixLQUFLLGlCQUFpQixDQUFDLG1CQUFtQjtnQkFDeEMsWUFBWTtvQkFDVixNQUFNLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyw0QkFBNEIsQ0FBQyxVQUFVLENBQUMsQ0FBQztnQkFDeEUsTUFBTTtZQUNSO2dCQUNFLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxpQkFBaUIsQ0FDekMsSUFBSSxRQUFRLENBQUM7b0JBQ1gsUUFBUSxFQUFFLE9BQU87b0JBQ2pCLFNBQVMsRUFBRSxhQUFhO2lCQUN6QixDQUFDLENBQ0gsQ0FBQztnQkFDRixNQUFNO1NBQ1Q7UUFFRCxJQUFJLENBQUMsVUFBVSxDQUFDLEVBQUUsRUFBRTtZQUNsQixJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7U0FDbkQ7UUFDRCxJQUFJLENBQUMsZUFBZSxHQUFHLFFBQVEsQ0FBQyxJQUFJLENBQUM7UUFFckMsSUFBSSxDQUFDLGdCQUFnQixHQUFHLEVBQUUsQ0FBQztRQUMzQixJQUFJLENBQUMsWUFBWSxHQUFHLEtBQUssQ0FBQztRQUMxQixNQUFNLElBQUksQ0FBQyxZQUFZLEVBQUUsQ0FBQztRQUUxQixNQUFNLFVBQVUsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FDbkMsQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDLE9BQU8sQ0FBQyxFQUFFLEtBQUssWUFBWSxDQUFDLEVBQUUsQ0FDNUMsQ0FBQztRQUNGLElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxDQUFDLFVBQVUsQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7OzBIQS9ZVSw2QkFBNkI7OEdBQTdCLDZCQUE2QiwrRENwQzFDLG0wRUFzRUE7MkZEbENhLDZCQUE2QjtrQkFMekMsU0FBUzsrQkFDRSx5QkFBeUIiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBDb21wb25lbnQsIE9uSW5pdCB9IGZyb20gJ0Bhbmd1bGFyL2NvcmUnO1xyXG5pbXBvcnQgeyBEeW5hbWljRGlhbG9nQ29uZmlnLCBEeW5hbWljRGlhbG9nUmVmIH0gZnJvbSAncHJpbWVuZy9keW5hbWljZGlhbG9nJztcclxuaW1wb3J0IHsgQUNUSU9OU19CQVNFX1NUUlVDVFVSRSB9IGZyb20gJy4uLy4uLy4uLy4uLy4uL2NvcmUvY29tcG9uZW50cy9keW5hbWljLWZvcm0vYWN0aW9ucy1kZWZhdWx0LXN0cnVjdHVyZXMvYWN0aW9ucy1iYXNlLXN0cnVjdHVyZSc7XHJcbmltcG9ydCB7XHJcbiAgQmFzZUFjdGlvbixcclxuICBCYXNlQWN0aW9uS2V5LFxyXG4gIER5bmFtaWNGb3JtQWN0aW9uT3V0cHV0LFxyXG59IGZyb20gJy4uLy4uLy4uLy4uLy4uL2NvcmUvY29tcG9uZW50cy9keW5hbWljLWZvcm0vbW9kZWxzL2Jhc2UtYWN0aW9uJztcclxuaW1wb3J0IHsgQmFzZUZpZWxkIH0gZnJvbSAnLi4vLi4vLi4vLi4vLi4vY29yZS9jb21wb25lbnRzL2R5bmFtaWMtZm9ybS9tb2RlbHMvYmFzZS1maWVsZCc7XHJcbmltcG9ydCB7IFZpZXdNb2RlIH0gZnJvbSAnLi4vLi4vLi4vLi4vLi4vY29yZS9tb2RlbHMvY29uc3RhbnRzL3ZpZXctbW9kZS5lbnVtJztcclxuaW1wb3J0IHsgUHJnRXJyb3IgfSBmcm9tICcuLi8uLi8uLi8uLi8uLi9jb3JlL21vZGVscy9lcnJvci5tb2RlbCc7XHJcbmltcG9ydCB7IE5vdGlmaWNhdGlvbnNTZXJ2aWNlIH0gZnJvbSAnLi4vLi4vLi4vLi4vLi4vY29yZS9zZXJ2aWNlcy9ub3RpZmljYXRpb25zL25vdGlmaWNhdGlvbnMuc2VydmljZSc7XHJcbmltcG9ydCB7IEFycmF5VXRpbGl0eVNlcnZpY2UgfSBmcm9tICcuLi8uLi8uLi8uLi8uLi9jb3JlL3NlcnZpY2VzL3V0aWxpdHkvYXJyYXktdXRpbGl0eS5zZXJ2aWNlJztcclxuaW1wb3J0IHsgT2JqZWN0c1V0aWxpdHlTZXJ2aWNlIH0gZnJvbSAnLi4vLi4vLi4vLi4vLi4vY29yZS9zZXJ2aWNlcy91dGlsaXR5L29iamVjdHMtdXRpbGl0eS5zZXJ2aWNlJztcclxuaW1wb3J0IHsgVXRpbGl0eVNlcnZpY2UgfSBmcm9tICcuLi8uLi8uLi8uLi8uLi9jb3JlL3NlcnZpY2VzL3V0aWxpdHkvdXRpbGl0eS5zZXJ2aWNlJztcclxuaW1wb3J0IHsgRU5USVRZX1RZUEVfQVRUUklCVVRFX0RZTkFNSUNfRk9STSB9IGZyb20gJy4uLy4uLy4uLy4uL2R5bmFtaWMtZm9ybS1zdHJ1Y3QvZW50aXR5LXR5cGUtYXR0cmlidXRlLWR5bmFtaWMtZm9ybSc7XHJcbmltcG9ydCB7IEVOVElUWV9UWVBFX0RZTkFNSUNfRk9STSB9IGZyb20gJy4uLy4uLy4uLy4uL2R5bmFtaWMtZm9ybS1zdHJ1Y3QvZW50aXR5LXR5cGUtZHluYW1pYy1mb3JtJztcclxuaW1wb3J0IHsgRU5USVRZX1RZUEVfT1BFUkFUSU9OX0RZTkFNSUNfRk9STSB9IGZyb20gJy4uLy4uLy4uLy4uL2R5bmFtaWMtZm9ybS1zdHJ1Y3QvZW50aXR5LXR5cGUtb3BlcmF0aW9uLWR5bmFtaWMtZm9ybSc7XHJcbmltcG9ydCB7IEVOVElUWV9UWVBFX1BST1BFUlRZX0RZTkFNSUNfRk9STSB9IGZyb20gJy4uLy4uLy4uLy4uL2R5bmFtaWMtZm9ybS1zdHJ1Y3QvZW50aXR5LXR5cGUtcHJvcGVydHktZHluYW1pYy1mb3JtJztcclxuaW1wb3J0IHtcclxuICBFbnRpdHlUeXBlLFxyXG4gIEVudGl0eVR5cGVBdHRyaWJ1dGUsXHJcbiAgRW50aXR5VHlwZU9wZXJhdGlvbixcclxuICBFbnRpdHlUeXBlUHJvcGVydHksXHJcbn0gZnJvbSAnLi4vLi4vLi4vLi4vbW9kZWxzL2VudGl0eS10eXBlLm1vZGVsJztcclxuaW1wb3J0IHsgQWJzdHJhY3RFbnRpdHlUeXBlU2VydmljZSB9IGZyb20gJy4uLy4uLy4uLy4uL3NlcnZpY2VzL2VudGl0eS10eXBlcy9hYnN0cmFjdC1lbnRpdHktdHlwZS5zZXJ2aWNlJztcclxuaW1wb3J0IHsgTWVudVRhYiwgTWVudVRhYkVudGl0eVR5cGUgfSBmcm9tICcuL21vZGVscy9tZW51LXRhYi5tb2R1bGUnO1xyXG5cclxuLyoqXHJcbiAqIGVudGl0eSB0eXBlIGNvbmZpZ3VyYXRpb24gY29tcG9uZW50XHJcbiAqL1xyXG5AQ29tcG9uZW50KHtcclxuICBzZWxlY3RvcjogJ3ByZy1lbnRpdHktdHlwZS1jb25maWdzJyxcclxuICB0ZW1wbGF0ZVVybDogJy4vcHJnLWVudGl0eS10eXBlLWNvbmZpZ3MuY29tcG9uZW50Lmh0bWwnLFxyXG4gIHN0eWxlVXJsczogWycuL3ByZy1lbnRpdHktdHlwZS1jb25maWdzLmNvbXBvbmVudC5zY3NzJ10sXHJcbn0pXHJcbmV4cG9ydCBjbGFzcyBQcmdFbnRpdHlUeXBlQ29uZmlnc0NvbXBvbmVudCBpbXBsZW1lbnRzIE9uSW5pdCB7XHJcbiAgLyoqXHJcbiAgICogY29uc3RydWN0b3JcclxuICAgKiBAcGFyYW0gZW50aXR5VHlwZVNlcnZpY2VcclxuICAgKiBAcGFyYW0gY29uZmlnXHJcbiAgICogQHBhcmFtIHJlZlxyXG4gICAqIEBwYXJhbSBub3RpZmljYXRpb25zU2VydmljZVxyXG4gICAqIEBwYXJhbSBhcnJheVV0aWxpdHlTZXJ2aWNlXHJcbiAgICogQHBhcmFtIG9iamVjdHNVdGlsaXR5U2VydmljZVxyXG4gICAqIEBwYXJhbSB1dGlsaXR5U2VydmljZVxyXG4gICAqL1xyXG4gIGNvbnN0cnVjdG9yKFxyXG4gICAgcHJpdmF0ZSBlbnRpdHlUeXBlU2VydmljZTogQWJzdHJhY3RFbnRpdHlUeXBlU2VydmljZSxcclxuICAgIHByaXZhdGUgY29uZmlnOiBEeW5hbWljRGlhbG9nQ29uZmlnLFxyXG4gICAgcHJpdmF0ZSByZWY6IER5bmFtaWNEaWFsb2dSZWYsXHJcbiAgICBwcml2YXRlIG5vdGlmaWNhdGlvbnNTZXJ2aWNlOiBOb3RpZmljYXRpb25zU2VydmljZSxcclxuICAgIHByaXZhdGUgYXJyYXlVdGlsaXR5U2VydmljZTogQXJyYXlVdGlsaXR5U2VydmljZSxcclxuICAgIHByaXZhdGUgb2JqZWN0c1V0aWxpdHlTZXJ2aWNlOiBPYmplY3RzVXRpbGl0eVNlcnZpY2UsXHJcbiAgICBwcml2YXRlIHV0aWxpdHlTZXJ2aWNlOiBVdGlsaXR5U2VydmljZVxyXG4gICkge31cclxuICAvKipcclxuICAgKiBlbnRpdHkgdHlwZVxyXG4gICAqL1xyXG4gIHB1YmxpYyBlbnRpdHlUeXBlOiBFbnRpdHlUeXBlO1xyXG5cclxuICAvKipcclxuICAgKiBtYW51IHRhYiBpdGVtc1xyXG4gICAqL1xyXG4gIHB1YmxpYyBtZW51VGFiSXRlbXM6IE1lbnVUYWJbXTtcclxuXHJcbiAgLyoqXHJcbiAgICogdGFiIGFjdGl2ZSBpdGVtXHJcbiAgICovXHJcbiAgcHVibGljIGFjdGl2ZUl0ZW06IE1lbnVUYWI7XHJcblxyXG4gIC8qKlxyXG4gICAqIGVsZW1lbnRzIGxpc3QgKHByb3BlcnRpZXMsIGF0dHJpYnV0ZXMsIG9wZXJhdGlvbnMpXHJcbiAgICovXHJcbiAgcHVibGljIGVsZW1lbnRzOiBhbnlbXSA9IFtdO1xyXG5cclxuICAvKipcclxuICAgKiBlbGVtZW50IHNlbGVjdGVkXHJcbiAgICovXHJcbiAgcHVibGljIGVsZW1lbnRzU2VsZWN0ZWQ6IGFueVtdID0gW107XHJcblxyXG4gIC8qKlxyXG4gICAqIGVudGl0eSB0eXBlIGZvcm0gZmllbGRzXHJcbiAgICovXHJcbiAgcHVibGljIGVudGl0eVR5cGVEeW5hbWljRm9ybUZpZWxkczogQmFzZUZpZWxkW10gPVxyXG4gICAgdGhpcy5hcnJheVV0aWxpdHlTZXJ2aWNlLmNsb25lKEVOVElUWV9UWVBFX0RZTkFNSUNfRk9STS5maWVsZHMpO1xyXG4gIC8qKlxyXG4gICAqIGVudGl0eSB0eXBlIGZvcm0gYWN0aW9uc1xyXG4gICAqL1xyXG4gIHB1YmxpYyBlbnRpdHlUeXBlRHluYW1pY0Zvcm1BY3Rpb25zOiBCYXNlQWN0aW9uW10gPVxyXG4gICAgdGhpcy5hcnJheVV0aWxpdHlTZXJ2aWNlLmNsb25lKEVOVElUWV9UWVBFX0RZTkFNSUNfRk9STS5hY3Rpb25zKTtcclxuXHJcbiAgLyoqXHJcbiAgICogZWxlbWVudHMgZm9ybSBmaWVsZHMgKGRlZmF1bHQgaXMgZmllbGRzIGZyb20gcHJvcGVydGllcylcclxuICAgKi9cclxuICBwdWJsaWMgZWxlbWVudER5bmFtaWNGb3JtRmllbGRzOiBCYXNlRmllbGRbXSA9IHRoaXMuYXJyYXlVdGlsaXR5U2VydmljZS5jbG9uZShcclxuICAgIEVOVElUWV9UWVBFX1BST1BFUlRZX0RZTkFNSUNfRk9STS5maWVsZHNcclxuICApO1xyXG5cclxuICAvKipcclxuICAgKiBlbGVtZW50cyBmb3JtIGFjdGlvbnNcclxuICAgKi9cclxuICBwdWJsaWMgZWxlbWVudER5bmFtaWNGb3JtQWN0aW9uczogQmFzZUFjdGlvbltdID1cclxuICAgIHRoaXMuYXJyYXlVdGlsaXR5U2VydmljZS5jbG9uZShBQ1RJT05TX0JBU0VfU1RSVUNUVVJFLmFjdGlvbnMpO1xyXG5cclxuICAvKipcclxuICAgKiBlbnRpdHkgdHlwZSB2aWV3IG1vZGVcclxuICAgKi9cclxuICBwdWJsaWMgZW50aXR5VHlwZVZpZXdNb2RlOiBWaWV3TW9kZSA9IFZpZXdNb2RlLlJlYWQ7XHJcbiAgLyoqXHJcbiAgICogZWxlbWVudHMgdmlldyBtb2RlXHJcbiAgICovXHJcbiAgcHVibGljIGVsZW1lbnRWaWV3TW9kZTogVmlld01vZGUgPSBWaWV3TW9kZS5BZGQ7XHJcblxyXG4gIC8qKlxyXG4gICAqIGRpYWJsZWQgbGlzdFxyXG4gICAqL1xyXG4gIHB1YmxpYyBkaXNhYmxlZExpc3Q6IGJvb2xlYW4gPSBmYWxzZTtcclxuXHJcbiAgLyoqXHJcbiAgICogZGVib3VuY2UgdGltZVxyXG4gICAqL1xyXG4gIHByaXZhdGUgZGVib3VuY2VUaW1lOiBudW1iZXIgPSAyMDA7XHJcblxyXG4gIC8qKlxyXG4gICAqIG5nT25Jbml0XHJcbiAgICovXHJcbiAgbmdPbkluaXQoKSB7XHJcbiAgICB0aGlzLm1lbnVUYWJJdGVtcyA9IFtcclxuICAgICAge1xyXG4gICAgICAgIGxhYmVsOiAncHJvcGVydGllcycsXHJcbiAgICAgICAgaWNvbjogJ3BpIHBpLWZ3IHBpLXBlbmNpbCcsXHJcbiAgICAgICAgdHlwZTogTWVudVRhYkVudGl0eVR5cGUuRW50aXR5VHlwZVByb3BlcnR5LFxyXG4gICAgICAgIGRpc2FibGVkOiBmYWxzZSxcclxuICAgICAgfSxcclxuICAgICAge1xyXG4gICAgICAgIGxhYmVsOiAnYXR0cmlidXRlcycsXHJcbiAgICAgICAgaWNvbjogJ3BpIHBpLWZ3IHBpLXBlbmNpbCcsXHJcbiAgICAgICAgdHlwZTogTWVudVRhYkVudGl0eVR5cGUuRW50aXR5VHlwZUF0dHJpYnV0ZSxcclxuICAgICAgICBkaXNhYmxlZDogZmFsc2UsXHJcbiAgICAgIH0sXHJcbiAgICAgIHtcclxuICAgICAgICBsYWJlbDogJ29wZXJhdGlvbnMnLFxyXG4gICAgICAgIGljb246ICdwaSBwaS1mdyBwaS1wZW5jaWwnLFxyXG4gICAgICAgIHR5cGU6IE1lbnVUYWJFbnRpdHlUeXBlLkVudGl0eVR5cGVPcGVyYXRpb24sXHJcbiAgICAgICAgZGlzYWJsZWQ6IGZhbHNlLFxyXG4gICAgICB9LFxyXG4gICAgXTtcclxuXHJcbiAgICB0aGlzLmFjdGl2ZUl0ZW0gPSB0aGlzLm1lbnVUYWJJdGVtc1swXTtcclxuXHJcbiAgICB0aGlzLmVudGl0eVR5cGUgPSB0aGlzLmNvbmZpZy5kYXRhLmVudGl0eVR5cGU7XHJcbiAgICB0aGlzLmVudGl0eVR5cGVWaWV3TW9kZSA9IHRoaXMuY29uZmlnLmRhdGEudmlld01vZGU7XHJcbiAgICBpZiAoIXRoaXMuZW50aXR5VHlwZS5pZCkge1xyXG4gICAgICB0aGlzLm1lbnVUYWJzSXRlbXNNYW5hZ2UobnVsbCk7XHJcbiAgICB9XHJcbiAgICB0aGlzLmxvYWRFbGVtZW50cygpO1xyXG4gIH1cclxuXHJcbiAgLyoqXHJcbiAgICogdGhpcyBmdW5jdGlvbiByZWNlaXZlcyB0aGUgZXZlbnRcclxuICAgKiB3aGVuIGEgbmV3IGVsZW1lbnQgaXMgc2VsZWN0ZWQgaW4gdGhlIGxpc3RcclxuICAgKiBAcGFyYW0gZXZlbnRcclxuICAgKiBAcmV0dXJuc1xyXG4gICAqL1xyXG4gIHB1YmxpYyBvblNlbGVjdGlvbkNoYW5nZWQoZXZlbnQ6IGFueSk6IHZvaWQge1xyXG4gICAgaWYgKGV2ZW50LnZhbHVlID09IG51bGwpIHtcclxuICAgICAgcmV0dXJuO1xyXG4gICAgfVxyXG4gICAgdGhpcy5lbGVtZW50c1NlbGVjdGVkID0gW2V2ZW50LnZhbHVlW2V2ZW50LnZhbHVlLmxlbmd0aCAtIDFdXTtcclxuICAgIHRoaXMuZWxlbWVudFZpZXdNb2RlID0gVmlld01vZGUuUmVhZDtcclxuICB9XHJcblxyXG4gIC8qKlxyXG4gICAqIHRoaXMgZnVuY3Rpb24gcmVjZWl2ZXMgdGhlIG91dHB1dCByZWZlcnJpbmcgdG8gdGhlIGFjdGlvbnMgb2YgYW4gZW50aXR5XHJcbiAgICogdHlwZSBhbmQgaW4gZnVuY3Rpb24gb2YgdGhlIHJlY2VpdmVkIGFjdGlvbiBwZXJmb3JtcyB0aGUgY29ycmVzcG9uZGluZyBhY3Rpb25zXHJcbiAgICogQHBhcmFtIGV2ZW50XHJcbiAgICovXHJcbiAgcHVibGljIGFzeW5jIG9uRW50aXR5VHlwZUFjdGlvbk91dHB1dHRlZChldmVudDogRHluYW1pY0Zvcm1BY3Rpb25PdXRwdXQpIHtcclxuICAgIHRoaXMubWVudVRhYnNJdGVtc01hbmFnZShudWxsKTtcclxuICAgIHN3aXRjaCAoZXZlbnQuYWN0aW9uKSB7XHJcbiAgICAgIGNhc2UgQmFzZUFjdGlvbktleS5TYXZlOlxyXG4gICAgICAgIHRoaXMuZW50aXR5VHlwZSA9IGF3YWl0IHRoaXMuZW50aXR5VHlwZVNlcnZpY2Uuc2F2ZUVudGl0eVR5cGVBc3luYyhcclxuICAgICAgICAgIGV2ZW50LmZvcm1FbnRpdHlcclxuICAgICAgICApO1xyXG4gICAgICAgIHRoaXMubWVudVRhYnNJdGVtc01hbmFnZShudWxsLCBmYWxzZSk7XHJcbiAgICAgICAgdGhpcy5hY3RpdmVJdGVtID0gdGhpcy5tZW51VGFiSXRlbXNbMF07XHJcbiAgICAgICAgdGhpcy5lbnRpdHlUeXBlVmlld01vZGUgPSBWaWV3TW9kZS5SZWFkO1xyXG5cclxuICAgICAgICBicmVhaztcclxuICAgICAgY2FzZSBCYXNlQWN0aW9uS2V5LkNhbmNlbDpcclxuICAgICAgICB0aGlzLmVudGl0eVR5cGVWaWV3TW9kZSA9IFZpZXdNb2RlLlJlYWQ7XHJcbiAgICAgICAgaWYgKGV2ZW50LmZvcm1FbnRpdHkuaWQpIHtcclxuICAgICAgICAgIHRoaXMuZW50aXR5VHlwZSA9IHRoaXMub2JqZWN0c1V0aWxpdHlTZXJ2aWNlLmNsb25lT2JqZWN0KFxyXG4gICAgICAgICAgICB0aGlzLmVudGl0eVR5cGVcclxuICAgICAgICAgICk7XHJcbiAgICAgICAgfSBlbHNlIHtcclxuICAgICAgICAgIHRoaXMucmVmLmNsb3NlKCk7XHJcbiAgICAgICAgfVxyXG4gICAgICAgIGJyZWFrO1xyXG5cclxuICAgICAgY2FzZSBCYXNlQWN0aW9uS2V5LkVkaXQ6XHJcbiAgICAgICAgdGhpcy5lbnRpdHlUeXBlVmlld01vZGUgPSBWaWV3TW9kZS5FZGl0O1xyXG4gICAgICAgIGJyZWFrO1xyXG4gICAgICBkZWZhdWx0OlxyXG4gICAgICAgIGJyZWFrO1xyXG4gICAgfVxyXG4gIH1cclxuXHJcbiAgLyoqXHJcbiAgICogdGhpcyBmdW5jdGlvbiBpcyByZXNwb25zaWJsZSBmb3IgcmVjZWl2aW5nIHRoZSBjaGFuZ2VzIGZyb20gdGhlIHNlbGVjdGVkIHRhYiBhbmQgbG9hZGluZyB0aGUgcmVzcGVjdGl2ZSBkYXRhXHJcbiAgICogQHBhcmFtIGV2ZW50XHJcbiAgICovXHJcbiAgcHVibGljIHRhYkNoYW5nZShldmVudDogTWVudVRhYik6IHZvaWQge1xyXG4gICAgdGhpcy5hY3RpdmVJdGVtID0gZXZlbnQ7XHJcbiAgICB0aGlzLmVsZW1lbnRzU2VsZWN0ZWQgPSBbXTtcclxuICAgIHRoaXMuZWxlbWVudFZpZXdNb2RlID0gVmlld01vZGUuQWRkO1xyXG4gICAgdGhpcy5sb2FkRWxlbWVudHMoKTtcclxuICB9XHJcblxyXG4gIC8qKlxyXG4gICAqIHRoaXMgZnVuY3Rpb24gbWFuYWdlcyB0aGUgc3RhdHVzIG9mIHRoZVxyXG4gICAqIHRhYnMgYWNjb3JkaW5nIHRvIHRoZSBwYXJhbWV0ZXJzXHJcbiAgICogaXQgY2FuIGFjdGl2YXRlIG9yIGRlYWN0aXZhdGUgYWxsIHRhYnMgYXMgd2VsbFxyXG4gICAqIGFzIGNoYW5nZSB0aGUgc3RhdHVzIG9mIGEgc2luZ2xlIHRhYlxyXG4gICAqXHJcbiAgICogQHBhcmFtIG1lbnVUYWJJdGVtXHJcbiAgICogQHBhcmFtIHN0YXR1c1xyXG4gICAqL1xyXG4gIHByaXZhdGUgbWVudVRhYnNJdGVtc01hbmFnZShtZW51VGFiSXRlbTogTWVudVRhYiwgc3RhdHVzOiBib29sZWFuID0gdHJ1ZSkge1xyXG4gICAgdGhpcy5tZW51VGFiSXRlbXMuZm9yRWFjaCgoaXRlbSkgPT4ge1xyXG4gICAgICBpdGVtLmRpc2FibGVkID0gc3RhdHVzO1xyXG4gICAgfSk7XHJcbiAgICBpZiAobWVudVRhYkl0ZW0pIHtcclxuICAgICAgY29uc3QgaXRlbSA9IHRoaXMubWVudVRhYkl0ZW1zLmZpbmQoXHJcbiAgICAgICAgKGl0ZW0pID0+IChpdGVtLnR5cGUgPSBtZW51VGFiSXRlbS50eXBlKVxyXG4gICAgICApO1xyXG4gICAgICBpdGVtLmRpc2FibGVkID0gdHJ1ZTtcclxuICAgIH1cclxuICAgIHRoaXMubWVudVRhYkl0ZW1zID0gdGhpcy5hcnJheVV0aWxpdHlTZXJ2aWNlLmNsb25lKHRoaXMubWVudVRhYkl0ZW1zKTtcclxuICAgIHRoaXMuYWN0aXZlSXRlbSA9IHRoaXMubWVudVRhYkl0ZW1zLmZpbmQoXHJcbiAgICAgIChpdGVtKSA9PiBpdGVtLnR5cGUgPT09IHRoaXMuYWN0aXZlSXRlbS50eXBlXHJcbiAgICApO1xyXG4gIH1cclxuXHJcbiAgLyoqXHJcbiAgICogY3JlYXRlIGVtcHR5IGVsZW1lbnQsIGRlcGVuZGluZyBvbiB0aGUgdGFiIHNlbGVjdGVkXHJcbiAgICovXHJcbiAgcHJpdmF0ZSBhZGROZXdFbGVtZW50KCk6IHZvaWQge1xyXG4gICAgbGV0IG5ld0VsZW1lbnQ6XHJcbiAgICAgIHwgRW50aXR5VHlwZVByb3BlcnR5XHJcbiAgICAgIHwgRW50aXR5VHlwZUF0dHJpYnV0ZVxyXG4gICAgICB8IEVudGl0eVR5cGVPcGVyYXRpb24gPSBudWxsO1xyXG4gICAgaWYgKHRoaXMuZW50aXR5VHlwZSAmJiB0aGlzLmFjdGl2ZUl0ZW0pIHtcclxuICAgICAgdGhpcy5kaXNhYmxlZExpc3QgPSB0cnVlO1xyXG4gICAgICBzd2l0Y2ggKHRoaXMuYWN0aXZlSXRlbS50eXBlKSB7XHJcbiAgICAgICAgY2FzZSBNZW51VGFiRW50aXR5VHlwZS5FbnRpdHlUeXBlUHJvcGVydHk6XHJcbiAgICAgICAgICBuZXdFbGVtZW50ID0gbmV3IEVudGl0eVR5cGVQcm9wZXJ0eSh7XHJcbiAgICAgICAgICAgIGVudGl0eVR5cGVJZDogdGhpcy5lbnRpdHlUeXBlLmlkLFxyXG4gICAgICAgICAgICBvcmRlcjogdGhpcy5lbGVtZW50cy5sZW5ndGggKyAxLFxyXG4gICAgICAgICAgfSk7XHJcbiAgICAgICAgICBicmVhaztcclxuICAgICAgICBjYXNlIE1lbnVUYWJFbnRpdHlUeXBlLkVudGl0eVR5cGVBdHRyaWJ1dGU6XHJcbiAgICAgICAgICBuZXdFbGVtZW50ID0gbmV3IEVudGl0eVR5cGVBdHRyaWJ1dGUoe1xyXG4gICAgICAgICAgICBlbnRpdHlUeXBlSWQ6IHRoaXMuZW50aXR5VHlwZS5pZCxcclxuICAgICAgICAgICAgb3JkZXI6IHRoaXMuZWxlbWVudHMubGVuZ3RoICsgMSxcclxuICAgICAgICAgIH0pO1xyXG4gICAgICAgICAgYnJlYWs7XHJcbiAgICAgICAgY2FzZSBNZW51VGFiRW50aXR5VHlwZS5FbnRpdHlUeXBlT3BlcmF0aW9uOlxyXG4gICAgICAgICAgbmV3RWxlbWVudCA9IG5ldyBFbnRpdHlUeXBlT3BlcmF0aW9uKHtcclxuICAgICAgICAgICAgZW50aXR5VHlwZUlkOiB0aGlzLmVudGl0eVR5cGUuaWQsXHJcbiAgICAgICAgICB9KTtcclxuICAgICAgICAgIGJyZWFrO1xyXG5cclxuICAgICAgICBkZWZhdWx0OlxyXG4gICAgICAgICAgdGhpcy5ub3RpZmljYXRpb25zU2VydmljZS5lcnJvck5vdGlmaWNhdGlvbihcclxuICAgICAgICAgICAgbmV3IFByZ0Vycm9yKHtcclxuICAgICAgICAgICAgICB0aXRsZUtleTogJ2Vycm9yJyxcclxuICAgICAgICAgICAgICBkZXRhaWxLZXk6ICdjYW50RmluZEVudGl0eScsXHJcbiAgICAgICAgICAgIH0pXHJcbiAgICAgICAgICApO1xyXG4gICAgICAgICAgYnJlYWs7XHJcbiAgICAgIH1cclxuICAgIH1cclxuICAgIC8vIGFkZCBlbGVtZW50IHRvIGVsZW1lbnRzIHRvIGFycmF5XHJcbiAgICB0aGlzLmVsZW1lbnRzLnB1c2gobmV3RWxlbWVudCk7XHJcbiAgICAvL3RoaXMuZWxlbWVudHMgPSB0aGlzLmFycmF5VXRpbGl0eVNlcnZpY2UuY2xvbmUodGhpcy5lbGVtZW50cyk7XHJcbiAgICB0aGlzLmVsZW1lbnRzU2VsZWN0ZWQgPSBbdGhpcy5lbGVtZW50c1t0aGlzLmVsZW1lbnRzLmxlbmd0aCAtIDFdXTtcclxuICB9XHJcblxyXG4gIC8qKlxyXG4gICAqIHRoaXMgZnVuY3Rpb24gbG9hZHMgdGhlIGVsZW1lbnRzIGFjY29yZGluZyB0byB0aGUgdGFiIHNlbGVjdGVkLFxyXG4gICAqICBhbmQgY2hhbmdlcyB0aGUgZm9ybSdzIGRhdGFcclxuICAgKi9cclxuICBwcml2YXRlIGFzeW5jIGxvYWRFbGVtZW50cygpOiBQcm9taXNlPHZvaWQ+IHtcclxuICAgIGlmICh0aGlzLmVudGl0eVR5cGUgJiYgdGhpcy5hY3RpdmVJdGVtKSB7XHJcbiAgICAgIHN3aXRjaCAodGhpcy5hY3RpdmVJdGVtLnR5cGUpIHtcclxuICAgICAgICBjYXNlIE1lbnVUYWJFbnRpdHlUeXBlLkVudGl0eVR5cGVQcm9wZXJ0eTpcclxuICAgICAgICAgIHRoaXMuZWxlbWVudER5bmFtaWNGb3JtRmllbGRzID0gdGhpcy5hcnJheVV0aWxpdHlTZXJ2aWNlLmNsb25lKFxyXG4gICAgICAgICAgICBFTlRJVFlfVFlQRV9QUk9QRVJUWV9EWU5BTUlDX0ZPUk0uZmllbGRzXHJcbiAgICAgICAgICApO1xyXG4gICAgICAgICAgdGhpcy5lbGVtZW50cyA9IHRoaXMuYXJyYXlVdGlsaXR5U2VydmljZS5zb3J0QnlQcm9wZXJ0eShcclxuICAgICAgICAgICAgYXdhaXQgdGhpcy5lbnRpdHlUeXBlU2VydmljZS5nZXRBbGxQcm9wZXJ0aWVzQnlFbnRpdHlUeXBlSWRBc3luYyhcclxuICAgICAgICAgICAgICB0aGlzLmVudGl0eVR5cGUuaWRcclxuICAgICAgICAgICAgKSxcclxuICAgICAgICAgICAgJ29yZGVyJ1xyXG4gICAgICAgICAgKTtcclxuICAgICAgICAgIGJyZWFrO1xyXG4gICAgICAgIGNhc2UgTWVudVRhYkVudGl0eVR5cGUuRW50aXR5VHlwZUF0dHJpYnV0ZTpcclxuICAgICAgICAgIHRoaXMuZWxlbWVudER5bmFtaWNGb3JtRmllbGRzID0gdGhpcy5hcnJheVV0aWxpdHlTZXJ2aWNlLmNsb25lKFxyXG4gICAgICAgICAgICBFTlRJVFlfVFlQRV9BVFRSSUJVVEVfRFlOQU1JQ19GT1JNLmZpZWxkc1xyXG4gICAgICAgICAgKTtcclxuXHJcbiAgICAgICAgICB0aGlzLmVsZW1lbnRzID0gdGhpcy5hcnJheVV0aWxpdHlTZXJ2aWNlLnNvcnRCeVByb3BlcnR5KFxyXG4gICAgICAgICAgICBhd2FpdCB0aGlzLmVudGl0eVR5cGVTZXJ2aWNlLmdldEFsbEF0dHJpYnV0ZXNCeUVudGl0eVR5cGVJZEFzeW5jKFxyXG4gICAgICAgICAgICAgIHRoaXMuZW50aXR5VHlwZS5pZFxyXG4gICAgICAgICAgICApLFxyXG4gICAgICAgICAgICAnb3JkZXInXHJcbiAgICAgICAgICApO1xyXG4gICAgICAgICAgYnJlYWs7XHJcbiAgICAgICAgY2FzZSBNZW51VGFiRW50aXR5VHlwZS5FbnRpdHlUeXBlT3BlcmF0aW9uOlxyXG4gICAgICAgICAgdGhpcy5lbGVtZW50RHluYW1pY0Zvcm1GaWVsZHMgPSB0aGlzLmFycmF5VXRpbGl0eVNlcnZpY2UuY2xvbmUoXHJcbiAgICAgICAgICAgIEVOVElUWV9UWVBFX09QRVJBVElPTl9EWU5BTUlDX0ZPUk0uZmllbGRzXHJcbiAgICAgICAgICApO1xyXG4gICAgICAgICAgdGhpcy5lbGVtZW50cyA9XHJcbiAgICAgICAgICAgIGF3YWl0IHRoaXMuZW50aXR5VHlwZVNlcnZpY2UuZ2V0QWxsT3BlcmF0aW9uc0J5RW50aXR5VHlwZUlkQXN5bmMoXHJcbiAgICAgICAgICAgICAgdGhpcy5lbnRpdHlUeXBlLmlkXHJcbiAgICAgICAgICAgICk7XHJcbiAgICAgICAgICBicmVhaztcclxuXHJcbiAgICAgICAgZGVmYXVsdDpcclxuICAgICAgICAgIHRoaXMubm90aWZpY2F0aW9uc1NlcnZpY2UuZXJyb3JOb3RpZmljYXRpb24oXHJcbiAgICAgICAgICAgIG5ldyBQcmdFcnJvcih7XHJcbiAgICAgICAgICAgICAgdGl0bGVLZXk6ICdlcnJvcicsXHJcbiAgICAgICAgICAgICAgZGV0YWlsS2V5OiAnY2FudEFkZE5ld0VsZW1lbnQnLFxyXG4gICAgICAgICAgICB9KVxyXG4gICAgICAgICAgKTtcclxuICAgICAgICAgIGJyZWFrO1xyXG4gICAgICB9XHJcbiAgICB9XHJcbiAgfVxyXG5cclxuICAvKipcclxuICAgKiB0aGlzIGZ1bmN0aW9uIHJlY2VpdmVzIHRoZSBvdXRwdXQgb2YgYW4gZW50aXR5LFxyXG4gICAqIGFuZCB0cmVhdHMgZWFjaCBvZiB0aGUgYWN0aW9ucyB3aXRoIHRoZSByZXNwZWN0aXZlIGxvZ2ljIHRvIGJlIGFwcGxpZWRcclxuICAgKiBAcGFyYW0gZXZlbnRcclxuICAgKi9cclxuICBwdWJsaWMgb25FbnRpdHlBY3Rpb25PdXRwdXR0ZWQoZXZlbnQ6IER5bmFtaWNGb3JtQWN0aW9uT3V0cHV0KTogdm9pZCB7XHJcbiAgICBzd2l0Y2ggKGV2ZW50LmFjdGlvbikge1xyXG4gICAgICBjYXNlIEJhc2VBY3Rpb25LZXkuU2F2ZTpcclxuICAgICAgICB0aGlzLnNhdmVOZXdFbGVtZW50KGV2ZW50LmZvcm1FbnRpdHkpO1xyXG4gICAgICAgIHRoaXMubWVudVRhYnNJdGVtc01hbmFnZShudWxsLCBmYWxzZSk7XHJcbiAgICAgICAgYnJlYWs7XHJcbiAgICAgIGNhc2UgQmFzZUFjdGlvbktleS5FZGl0OlxyXG4gICAgICAgIHRoaXMuZGlzYWJsZWRMaXN0ID0gdHJ1ZTtcclxuICAgICAgICB0aGlzLmVsZW1lbnRWaWV3TW9kZSA9IFZpZXdNb2RlLkVkaXQ7XHJcbiAgICAgICAgdGhpcy5tZW51VGFic0l0ZW1zTWFuYWdlKG51bGwsIHRydWUpO1xyXG4gICAgICAgIGJyZWFrO1xyXG4gICAgICBjYXNlIEJhc2VBY3Rpb25LZXkuQ2FuY2VsOlxyXG4gICAgICAgIGlmIChldmVudC5mb3JtRW50aXR5LmlkKSB7XHJcbiAgICAgICAgICB0aGlzLmVsZW1lbnRzU2VsZWN0ZWQgPSB0aGlzLmFycmF5VXRpbGl0eVNlcnZpY2UuY2xvbmUoXHJcbiAgICAgICAgICAgIHRoaXMuZWxlbWVudHNTZWxlY3RlZFxyXG4gICAgICAgICAgKTtcclxuICAgICAgICAgIHRoaXMuZWxlbWVudFZpZXdNb2RlID0gVmlld01vZGUuUmVhZDtcclxuICAgICAgICB9IGVsc2Uge1xyXG4gICAgICAgICAgdGhpcy5lbGVtZW50Vmlld01vZGUgPSBWaWV3TW9kZS5BZGQ7XHJcbiAgICAgICAgICB0aGlzLmVsZW1lbnRzU2VsZWN0ZWQgPSBbXTtcclxuICAgICAgICAgIHRoaXMuZWxlbWVudHMuc3BsaWNlKHRoaXMuZWxlbWVudHMubGVuZ3RoIC0gMSwgMSk7XHJcbiAgICAgICAgfVxyXG4gICAgICAgIHRoaXMuZGlzYWJsZWRMaXN0ID0gZmFsc2U7XHJcbiAgICAgICAgdGhpcy5tZW51VGFic0l0ZW1zTWFuYWdlKG51bGwsIGZhbHNlKTtcclxuXHJcbiAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICBjYXNlIEJhc2VBY3Rpb25LZXkuQWRkOlxyXG4gICAgICAgIHRoaXMubWVudVRhYnNJdGVtc01hbmFnZShudWxsLCB0cnVlKTtcclxuICAgICAgICB0aGlzLmVsZW1lbnRWaWV3TW9kZSA9IFZpZXdNb2RlLkVkaXQ7XHJcbiAgICAgICAgdGhpcy5hZGROZXdFbGVtZW50KCk7XHJcbiAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICBkZWZhdWx0OlxyXG4gICAgICAgIHRoaXMubm90aWZpY2F0aW9uc1NlcnZpY2UuZXJyb3JOb3RpZmljYXRpb24oXHJcbiAgICAgICAgICBuZXcgUHJnRXJyb3Ioe1xyXG4gICAgICAgICAgICB0aXRsZUtleTogJ2Vycm9yJyxcclxuICAgICAgICAgICAgZGV0YWlsS2V5OiAnYWN0aW9uTm90Rm91bmQnLFxyXG4gICAgICAgICAgfSlcclxuICAgICAgICApO1xyXG4gICAgICAgIGJyZWFrO1xyXG4gICAgfVxyXG4gIH1cclxuXHJcbiAgLyoqXHJcbiAgICogdGhpcyBmdW5jdGlvbiBpcyByZXNwb25zaWJsZSBmb3Igc2F2aW5nIHRoZSBuZXcgZWxlbWVudCBhZGRlZCxcclxuICAgKiBkZXBlbmRpbmcgb24gdGhlIHNlbGVjdGVkIHRhYiBpdCBjYWxscyB0aGUgcmVzcGVjdGl2ZSBtZXRob2RcclxuICAgKiBAcGFyYW0gbmV3RWxlbWVudFxyXG4gICAqL1xyXG4gIHByaXZhdGUgYXN5bmMgc2F2ZU5ld0VsZW1lbnQobmV3RWxlbWVudDogYW55KTogUHJvbWlzZTx2b2lkPiB7XHJcbiAgICBsZXQgYWRkZWRFbGVtZW50OiBhbnk7XHJcbiAgICBzd2l0Y2ggKHRoaXMuYWN0aXZlSXRlbS50eXBlKSB7XHJcbiAgICAgIGNhc2UgTWVudVRhYkVudGl0eVR5cGUuRW50aXR5VHlwZVByb3BlcnR5OlxyXG4gICAgICAgIGFkZGVkRWxlbWVudCA9IGF3YWl0IHRoaXMuZW50aXR5VHlwZVNlcnZpY2Uuc2F2ZUVudGl0eVR5cGVQcm9wZXJ0eUFzeW5jKFxyXG4gICAgICAgICAgbmV3RWxlbWVudFxyXG4gICAgICAgICk7XHJcbiAgICAgICAgYnJlYWs7XHJcbiAgICAgIGNhc2UgTWVudVRhYkVudGl0eVR5cGUuRW50aXR5VHlwZUF0dHJpYnV0ZTpcclxuICAgICAgICBhZGRlZEVsZW1lbnQgPVxyXG4gICAgICAgICAgYXdhaXQgdGhpcy5lbnRpdHlUeXBlU2VydmljZS5zYXZlRW50aXR5VHlwZUF0dHJpYnV0ZUFzeW5jKG5ld0VsZW1lbnQpO1xyXG4gICAgICAgIGJyZWFrO1xyXG4gICAgICBjYXNlIE1lbnVUYWJFbnRpdHlUeXBlLkVudGl0eVR5cGVPcGVyYXRpb246XHJcbiAgICAgICAgYWRkZWRFbGVtZW50ID1cclxuICAgICAgICAgIGF3YWl0IHRoaXMuZW50aXR5VHlwZVNlcnZpY2Uuc2F2ZUVudGl0eVR5cGVPcGVyYXRpb25Bc3luYyhuZXdFbGVtZW50KTtcclxuICAgICAgICBicmVhaztcclxuICAgICAgZGVmYXVsdDpcclxuICAgICAgICB0aGlzLm5vdGlmaWNhdGlvbnNTZXJ2aWNlLmVycm9yTm90aWZpY2F0aW9uKFxyXG4gICAgICAgICAgbmV3IFByZ0Vycm9yKHtcclxuICAgICAgICAgICAgdGl0bGVLZXk6ICdlcnJvcicsXHJcbiAgICAgICAgICAgIGRldGFpbEtleTogJ3RhYk5vdEZvdW5kJyxcclxuICAgICAgICAgIH0pXHJcbiAgICAgICAgKTtcclxuICAgICAgICBicmVhaztcclxuICAgIH1cclxuXHJcbiAgICBpZiAoIW5ld0VsZW1lbnQuaWQpIHtcclxuICAgICAgdGhpcy5lbGVtZW50cy5zcGxpY2UodGhpcy5lbGVtZW50cy5sZW5ndGggLSAxLCAxKTtcclxuICAgIH1cclxuICAgIHRoaXMuZWxlbWVudFZpZXdNb2RlID0gVmlld01vZGUuUmVhZDtcclxuXHJcbiAgICB0aGlzLmVsZW1lbnRzU2VsZWN0ZWQgPSBbXTtcclxuICAgIHRoaXMuZGlzYWJsZWRMaXN0ID0gZmFsc2U7XHJcbiAgICBhd2FpdCB0aGlzLmxvYWRFbGVtZW50cygpO1xyXG5cclxuICAgIGNvbnN0IGVsZW1lbnRBdXggPSB0aGlzLmVsZW1lbnRzLmZpbmQoXHJcbiAgICAgIChlbGVtZW50KSA9PiBlbGVtZW50LmlkID09PSBhZGRlZEVsZW1lbnQuaWRcclxuICAgICk7XHJcbiAgICB0aGlzLmVsZW1lbnRzU2VsZWN0ZWQgPSBbZWxlbWVudEF1eF07XHJcbiAgfVxyXG5cclxuICAvKipcclxuICAgKiB0aGlzIGZ1bmN0aW9uIGlzIHJlc3BvbnNpYmxlIGZvciB1cGRhdGluZ1xyXG4gICAqIHRoZSBlbGVtZW50cyBpbiB0aGUgZGF0YWJhc2Ugd2hlbiB0aGV5IGhhdmUgdGhlaXIgb3JkZXIgY2hhbmdlZFxyXG4gICAqL1xyXG4gIHB1YmxpYyByZW9yZGVyRWxlbWVudHMgPSB0aGlzLnV0aWxpdHlTZXJ2aWNlLmRlYm91bmNlKCgpID0+IHtcclxuICAgIGxldCBtYXBFbGVtZW50cyA9IG5ldyBNYXA8c3RyaW5nLCBhbnk+KCk7XHJcblxyXG4gICAgZm9yIChsZXQgaSA9IDA7IGkgPCB0aGlzLmVsZW1lbnRzLmxlbmd0aDsgaSsrKSB7XHJcbiAgICAgIHRoaXMuZWxlbWVudHNbaV0ub3JkZXIgPSBpICsgMTtcclxuICAgICAgbWFwRWxlbWVudHMuc2V0KHRoaXMuZWxlbWVudHNbaV0uaWQsIHsgb3JkZXI6IHRoaXMuZWxlbWVudHNbaV0ub3JkZXIgfSk7XHJcbiAgICB9XHJcblxyXG4gICAgc3dpdGNoICh0aGlzLmFjdGl2ZUl0ZW0udHlwZSkge1xyXG4gICAgICBjYXNlIE1lbnVUYWJFbnRpdHlUeXBlLkVudGl0eVR5cGVQcm9wZXJ0eTpcclxuICAgICAgICB0aGlzLmVudGl0eVR5cGVTZXJ2aWNlLnVwZGF0ZUVudGl0eVR5cGVQcm9wZXJ0aWVzQXN5bmMobWFwRWxlbWVudHMpO1xyXG4gICAgICAgIGJyZWFrO1xyXG4gICAgICBjYXNlIE1lbnVUYWJFbnRpdHlUeXBlLkVudGl0eVR5cGVBdHRyaWJ1dGU6XHJcbiAgICAgICAgdGhpcy5lbnRpdHlUeXBlU2VydmljZS51cGRhdGVFbnRpdHlUeXBlQXR0cmlidXRlc0FzeW5jKG1hcEVsZW1lbnRzKTtcclxuICAgICAgICBicmVhaztcclxuXHJcbiAgICAgIGRlZmF1bHQ6XHJcbiAgICAgICAgdGhpcy5ub3RpZmljYXRpb25zU2VydmljZS5lcnJvck5vdGlmaWNhdGlvbihcclxuICAgICAgICAgIG5ldyBQcmdFcnJvcih7XHJcbiAgICAgICAgICAgIHRpdGxlS2V5OiAnZXJyb3InLFxyXG4gICAgICAgICAgICBkZXRhaWxLZXk6ICdub3RVcGRhdGVkJyxcclxuICAgICAgICAgIH0pXHJcbiAgICAgICAgKTtcclxuICAgICAgICBicmVhaztcclxuICAgIH1cclxuICB9LCB0aGlzLmRlYm91bmNlVGltZSk7XHJcbn1cclxuIiwiPGRpdj5cclxuICA8cHJnLWR5bmFtaWMtZm9ybVxyXG4gICAgW2ZpZWxkc109XCJlbnRpdHlUeXBlRHluYW1pY0Zvcm1GaWVsZHNcIlxyXG4gICAgW2FjdGlvbnNdPVwiZW50aXR5VHlwZUR5bmFtaWNGb3JtQWN0aW9uc1wiXHJcbiAgICBbZW50aXR5XT1cImVudGl0eVR5cGVcIlxyXG4gICAgW3ZpZXdNb2RlXT1cImVudGl0eVR5cGVWaWV3TW9kZVwiXHJcbiAgICAoYWN0aW9uT3V0cHV0KT1cIm9uRW50aXR5VHlwZUFjdGlvbk91dHB1dHRlZCgkZXZlbnQpXCJcclxuICA+PC9wcmctZHluYW1pYy1mb3JtPlxyXG5cclxuICA8cC10YWJNZW51XHJcbiAgICBbbW9kZWxdPVwibWVudVRhYkl0ZW1zXCJcclxuICAgIFthY3RpdmVJdGVtXT1cImFjdGl2ZUl0ZW1cIlxyXG4gICAgc3R5bGVDbGFzcz1cInAtdGFibWVudS1uYXZcIlxyXG4gID5cclxuICAgIDxuZy10ZW1wbGF0ZSBwVGVtcGxhdGU9XCJpdGVtXCIgbGV0LWl0ZW0gbGV0LWk9XCJpbmRleFwiPlxyXG4gICAgICA8ZGl2IChjbGljayk9XCIhaXRlbS5kaXNhYmxlZCAmJiB0YWJDaGFuZ2UoaXRlbSlcIj5cclxuICAgICAgICA8aSBbY2xhc3NdPVwiaXRlbS5pY29uXCI+PC9pPlxyXG4gICAgICAgIHt7ICdlbnRpdHktdHlwZS50YWJzLicgKyBpdGVtLmxhYmVsICsgJy5sYWJlbCcgfCB0cmFuc2xhdGUgfX1cclxuICAgICAgPC9kaXY+XHJcbiAgICA8L25nLXRlbXBsYXRlPlxyXG4gIDwvcC10YWJNZW51PlxyXG5cclxuICA8cC1zcGxpdHRlclxyXG4gICAgKm5nSWY9XCJlbnRpdHlUeXBlLmlkXCJcclxuICAgIFtzdHlsZV09XCJ7IGhlaWdodDogJzMwMHB4JyB9XCJcclxuICAgIFtwYW5lbFNpemVzXT1cIlszMCwgNzBdXCJcclxuICAgIFttaW5TaXplc109XCJbMjAsIDQwXVwiXHJcbiAgPlxyXG4gICAgPG5nLXRlbXBsYXRlIHBUZW1wbGF0ZT5cclxuICAgICAgPHAtb3JkZXJMaXN0XHJcbiAgICAgICAgKm5nSWY9XCJlbGVtZW50cy5sZW5ndGggPiAwXCJcclxuICAgICAgICBbdmFsdWVdPVwiZWxlbWVudHNcIlxyXG4gICAgICAgIFtkaXNhYmxlZF09XCJkaXNhYmxlZExpc3RcIlxyXG4gICAgICAgIFtzZWxlY3Rpb25dPVwiZWxlbWVudHNTZWxlY3RlZFwiXHJcbiAgICAgICAgKG9uU2VsZWN0aW9uQ2hhbmdlKT1cIm9uU2VsZWN0aW9uQ2hhbmdlZCgkZXZlbnQpXCJcclxuICAgICAgICBmaWx0ZXJCeT1cIm5hbWVcIlxyXG4gICAgICAgIFtkcmFnZHJvcF09XCJ0cnVlXCJcclxuICAgICAgICBzdHlsZUNsYXNzPVwicC10YWJtZW51LW5hdlwiXHJcbiAgICAgICAgKG9uUmVvcmRlcik9XCJyZW9yZGVyRWxlbWVudHMoJGV2ZW50KVwiXHJcbiAgICAgID5cclxuICAgICAgICA8bmctdGVtcGxhdGUgbGV0LWVsZW1lbnQgcFRlbXBsYXRlPVwibGlzdFwiPlxyXG4gICAgICAgICAgPGRpdj5cclxuICAgICAgICAgICAgPGRpdj5cclxuICAgICAgICAgICAgICB7e1xyXG4gICAgICAgICAgICAgICAgZWxlbWVudC5pZCAhPSBudWxsXHJcbiAgICAgICAgICAgICAgICAgID8gKCdlbnRpdGllcy4nICtcclxuICAgICAgICAgICAgICAgICAgICAgIGVudGl0eVR5cGUubmFtZSArXHJcbiAgICAgICAgICAgICAgICAgICAgICAnLmZpZWxkcy4nICtcclxuICAgICAgICAgICAgICAgICAgICAgIGVsZW1lbnQubmFtZSArXHJcbiAgICAgICAgICAgICAgICAgICAgICAnLmxhYmVsJyB8IHRyYW5zbGF0ZSlcclxuICAgICAgICAgICAgICAgICAgOiAoJ2VudGl0eS5uZXcuJyArIGFjdGl2ZUl0ZW0udHlwZS50b0xvd2VyQ2FzZSgpIHwgdHJhbnNsYXRlKVxyXG4gICAgICAgICAgICAgIH19XHJcbiAgICAgICAgICAgIDwvZGl2PlxyXG4gICAgICAgICAgPC9kaXY+XHJcbiAgICAgICAgPC9uZy10ZW1wbGF0ZT5cclxuICAgICAgPC9wLW9yZGVyTGlzdD48L25nLXRlbXBsYXRlXHJcbiAgICA+XHJcbiAgICA8bmctdGVtcGxhdGUgcFRlbXBsYXRlPVwiZHluYW1pY0Zvcm1cIj5cclxuICAgICAgPGRpdj5cclxuICAgICAgICA8cHJnLWR5bmFtaWMtZm9ybVxyXG4gICAgICAgICAgW2ZpZWxkc109XCJlbGVtZW50RHluYW1pY0Zvcm1GaWVsZHNcIlxyXG4gICAgICAgICAgW2FjdGlvbnNdPVwiZWxlbWVudER5bmFtaWNGb3JtQWN0aW9uc1wiXHJcbiAgICAgICAgICBbZW50aXR5XT1cImVsZW1lbnRzU2VsZWN0ZWQubGVuZ3RoID8gZWxlbWVudHNTZWxlY3RlZFswXSA6IG51bGxcIlxyXG4gICAgICAgICAgW3ZpZXdNb2RlXT1cImVsZW1lbnRWaWV3TW9kZVwiXHJcbiAgICAgICAgICAoYWN0aW9uT3V0cHV0KT1cIm9uRW50aXR5QWN0aW9uT3V0cHV0dGVkKCRldmVudClcIlxyXG4gICAgICAgID48L3ByZy1keW5hbWljLWZvcm0+XHJcbiAgICAgIDwvZGl2PlxyXG4gICAgPC9uZy10ZW1wbGF0ZT5cclxuICA8L3Atc3BsaXR0ZXI+XHJcbjwvZGl2PlxyXG4iXX0=