/**
 * Class PrgGridsterConfig to be used on dashboards options
 */
export class PrgGridsterConfig {
    /**
     * Constructor
     * @param {Partial<PrgGridsterConfig>} init
     */
    constructor(init) {
        /**
         * Different types for layout for the grid.
         * @type {PrgGridType}
         */
        this.gridType = PrgGridType.Fit;
        /**
         * scale param to zoom in/zoom out
         * @type {number}
         */
        this.scale = 1;
        /**
         * fixed col width for gridType: fixed
         * @type {number}
         */
        this.fixedColWidth = 250;
        /**
         * fixed row height for gridType: fixed
         * @type {number}
         */
        this.fixedRowHeight = 250;
        /**
         * keep the height from fixed gridType in mobile layout
         * @type {boolean}
         */
        this.keepFixedHeightInMobile = false;
        /**
         * keep the width from fixed gridType in mobile layout
         * @type {boolean}
         */
        this.keepFixedWidthInMobile = false;
        /**
         * sets grid size depending on content
         * @type {boolean}
         */
        this.setGridSize = false;
        /**
         * use the width of the <body> element to determine when to switch to the mobile layout
         * @type {boolean}
         */
        this.useBodyForBreakpoint = false;
        /**
         * compact items
         * @type {PrgCompactType}
         */
        this.compactType = PrgCompactType.None;
        /**
         *  if the screen is not wider that this, remove the grid layout and stack the items
         * @type {number}
         */
        this.mobileBreakpoint = 640;
        /**
         * allow items show in layers
         * @type {boolean}
         */
        this.allowMultiLayer = false;
        /**
         * default layer index of an item in gridster
         * @type {number}
         */
        this.defaultLayerIndex = 1;
        /**
         * max layer index of an item in gridster
         * @type {number}
         */
        this.maxLayerIndex = 2;
        /**
         * base layer index of an item in gridster, final z-index should be baseLayerIndex + layerIndex
         * @type {number}
         */
        this.baseLayerIndex = 1;
        /**
         * minimum amount of columns in the grid
         * @type {number}
         */
        this.minCols = 1;
        /**
         * maximum amount of columns in the grid
         * @type {number}
         */
        this.maxCols = 100;
        /**
         * minimum amount of rows in the grid
         * @type {number}
         */
        this.minRows = 1;
        /**
         * maximum amount of rows in the grid
         * @type {number}
         */
        this.maxRows = 100;
        /**
         * default width of an item in columns
         * @type {number}
         */
        this.defaultItemCols = 1;
        /**
         * default height of an item in rows
         * @type {number}
         */
        this.defaultItemRows = 1;
        /**
         * maximum item number of cols
         * @type {number}
         */
        this.maxItemCols = 50;
        /**
         * maximum item number of rows
         * @type {number}
         */
        this.maxItemRows = 50;
        /**
         * 	minimum item number of cols
         * @type {number}
         */
        this.minItemCols = 1;
        /**
         * minimum item number of row
         * @type {number}
         */
        this.minItemRows = 1;
        /**
         * minimum item area: cols * rows
         * @type {number}
         */
        this.minItemArea = 1;
        /**
         * maximum item area: cols * rows
         * @type {number}
         */
        this.maxItemArea = 2500;
        /**
         * add a number of extra empty rows at the end
         * @type {number}
         */
        this.addEmptyRowsCount = 0;
        /**
         * row height ratio from column width for gridType: scrollVertical and scrollHorizontal
         * @type {number}
         */
        this.rowHeightRatio = 1;
        /**
         * margin between grid items
         * @type {number}
         */
        this.margin = 5;
        /**
         * if margins will apply to the sides of the container
         * @type {boolean}
         */
        this.outerMargin = true;
        /**
         * override top outer margin for grid
         * @type {number | null}
         */
        this.outerMarginTop = null;
        /**
         * override right outer margin for grid
         * @type {number | null}
         */
        this.outerMarginRight = null;
        /**
         * override bottom outer margin for grid
         * @type {number | null}
         */
        this.outerMarginBottom = null;
        /**
         * override left outer margin for grid
         * @type {number | null}
         */
        this.outerMarginLeft = null;
        /**
         * use of transform positioning property
         * @type {boolean}
         */
        this.useTransformPositioning = true;
        /**
         * Scroll sensitivity
         * @type {number | null}
         */
        this.scrollSensitivity = 10;
        /**
         * scroll speed
         * @type {number}
         */
        this.scrollSpeed = 20;
        /**
         * PrgDraggable Object
         * @type {PrgDraggable}
         */
        this.draggable = new PrgDraggable({});
        /**
         *  PrgResizable Object
         * @type {PrgResizable}
         */
        this.resizable = new PrgResizable({});
        /**
         * allow items to switch position if drop on top of another
         * @type {boolean}
         */
        this.swap = true;
        /**
         * swap items while dragging and save new position
         * @type {boolean}
         */
        this.swapWhileDragging = false;
        /**
         * push items when resizing and dragging
         * @type {boolean}
         */
        this.pushItems = false;
        /**
         * disable push on drag
         * @type {boolean}
         */
        this.disablePushOnDrag = false;
        /**
         * disable push on resize
         * @type {boolean}
         */
        this.disablePushOnResize = false;
        /**
         * disable auto-position of items on conflict state
         * @type {boolean}
         */
        this.disableAutoPositionOnConflict = false;
        /**
         * control the directions items are pushed
         * @type {PushDirections}
         */
        this.pushDirections = new PrgPushDirections({});
        /**
         * on resize of item will shrink adjacent items
         * @type {boolean}
         */
        this.pushResizeItems = false;
        /**
         * display background grid of rows and columns
         * @type {displayGrids}
         */
        this.displayGrid = PrgDisplayGrid.Always;
        /**
         * disable the window on resize listener. This will stop grid to recalculate on window resize.
         * @type {boolean}
         */
        this.disableWindowResize = false;
        /**
         * disable console log warnings about misplacement of grid items
         * @type {boolean}
         */
        this.disableWarnings = false;
        /**
         * scroll to new items placed in a scrollable view
         * @type {boolean}
         */
        this.scrollToNewItems = false;
        /**
         * enable/disable auto horizontal scrolling when on edge of grid
         * @type {boolean}
         */
        this.disableScrollHorizontal = false;
        /**
         * enable/disable auto vertical scrolling when on edge of grid
         * @type {boolean}
         */
        this.disableScrollVertical = false;
        /**
         * enable/disable boundary control while dragging items
         * @type {boolean}
         */
        this.enableBoundaryControl = false;
        /**
         * enable empty cell click events
         * @type {boolean}
         */
        this.enableEmptyCellClick = true;
        /**
         * enable empty cell context menu (right click) events
         * @type {boolean}
         */
        this.enableEmptyCellContextMenu = false;
        /**
         * enable empty cell drop events
         * @type {boolean}
         */
        this.enableEmptyCellDrop = false;
        /**
         * enable empty cell drag events
         * @type {boolean}
         */
        this.enableEmptyCellDrag = true;
        /**
         * enable occupied cell drop events
         * @type {boolean}
         */
        this.enableOccupiedCellDrop = false;
        /**
         * limit empty cell drag max cols
         * @type {number}
         */
        this.emptyCellDragMaxCols = 50;
        /**
         * limit empty cell drag max rows
         * @type {number}
         */
        this.emptyCellDragMaxRows = 50;
        /**
         * ignore the gap between rows for items which span multiple rows (see #162, #224) only for gridType: fixed , verticalFixed, horizontalFixed
         * @type {boolean}
         */
        this.ignoreMarginInRow = false;
        /**
         * direction type
         * @type {PrgDirTypes}
         */
        this.dirType = PrgDirTypes.LTR;
        Object.assign(this, init);
    }
}
/**
 * Enum to be used on property gridType of PrgGridsterConfig
 */
export var PrgGridType;
(function (PrgGridType) {
    PrgGridType["Fit"] = "fit";
    PrgGridType["ScrollVertical"] = "scrollVertical";
    PrgGridType["ScrollHorizontal"] = "scrollHorizontal";
    PrgGridType["Fixed"] = "fixed";
    PrgGridType["VerticalFixed"] = "verticalFixed";
    PrgGridType["HorizontalFixed"] = "horizontalFixed";
})(PrgGridType || (PrgGridType = {}));
/**
 * Enum to be used on property displayGrid of PrgGridsterConfig
 */
export var PrgDisplayGrid;
(function (PrgDisplayGrid) {
    PrgDisplayGrid["Always"] = "always";
    PrgDisplayGrid["OnDragAndResize"] = "onDrag&Resize";
    PrgDisplayGrid["None"] = "none";
})(PrgDisplayGrid || (PrgDisplayGrid = {}));
/**
 * Enum to be used on property compactType of PrgGridsterConfig
 */
export var PrgCompactType;
(function (PrgCompactType) {
    PrgCompactType["None"] = "none";
    PrgCompactType["CompactUp"] = "compactUp";
    PrgCompactType["CompactLeft"] = "compactLeft";
    PrgCompactType["CompactUpAndLeft"] = "compactUp&Left";
    PrgCompactType["CompactLeftAndUp"] = "compactLeft&Up";
    PrgCompactType["CompactRight"] = "compactRight";
    PrgCompactType["CompactUpAndRight"] = "compactUp&Right";
    PrgCompactType["CompactRightAndUp"] = "compactRight&Up";
    PrgCompactType["CompactDown"] = "compactDown";
    PrgCompactType["CompactDownAndLeft"] = "compactDown&Left";
    PrgCompactType["CompactLeftAndDown"] = "compactLeft&Down";
    PrgCompactType["CompactDownAndRight"] = "compactDown&Right";
    PrgCompactType["CompactRightAndDown"] = "compactRight&Down";
})(PrgCompactType || (PrgCompactType = {}));
/**
 * Enum to be used on property dirType of PrgGridsterConfig
 */
export var PrgDirTypes;
(function (PrgDirTypes) {
    PrgDirTypes["LTR"] = "ltr";
    PrgDirTypes["RTL"] = "rtl";
})(PrgDirTypes || (PrgDirTypes = {}));
/**
 * Class PrgDragBase
 */
export class PrgDragBase {
    /**
     * Constructor
     * @param {Partial<PrgDragBase>} init
     */
    constructor(init) {
        /**
         * enable/disable draggable items
         * @type {boolean}
         */
        this.enabled = true;
        /**
         * milliseconds to delay the start of drag, useful for touch interaction
         * @type {number}
         */
        this.delayStart = 0;
        Object.assign(this, init);
    }
}
/**
 * Class PrgDraggable
 */
export class PrgDraggable extends PrgDragBase {
    /**
     * Constructor
     * @param {Partial<PrgDraggable>} init
     */
    constructor(init) {
        super(new PrgDragBase({}));
        /**
         * default content class to ignore the drag event from
         * @type {string}
         */
        this.ignoreContentClass = 'gridster-item-content';
        /**
         *  if true drag will start only from elements from dragHandleClass
         * @type {boolean}
         */
        this.ignoreContent = false;
        /**
         * drag event only from this class. If ignoreContent is true.
         * @type {string}
         */
        this.dragHandleClass = 'drag-handle';
        /**
         * enable items drop over another, will work if swap and push is disabled
         * @type {boolean}
         */
        this.dropOverItems = true;
        Object.assign(this, init);
    }
}
/**
 * Class PrgRezisable
 */
export class PrgResizable extends PrgDragBase {
    /**
     * Constructor
     * @param {Partial<PrgResizable>} init
     */
    constructor(init) {
        super(new PrgDragBase({}));
        /**
         * resizable edges of an item
         * @type {{s: boolean, e: boolean, n: boolean, w: boolean, se: boolean, ne: boolean, sw: boolean, nw: boolean}}
         */
        this.handles = new PrgResizableHandles({});
        Object.assign(this, init);
    }
}
/**
 * Class PrgPushDirections
 */
export class PrgPushDirections {
    /**
     * Constructor
     * @param {Partial<PrgPushDirections>} init
     */
    constructor(init) {
        /**
         * Push direction north
         * @type {boolean}
         */
        this.north = true;
        /**
         * Push direction east
         * @type {boolean}
         */
        this.east = true;
        /**
         * Push direction south
         * @type {boolean}
         */
        this.south = true;
        /**
         * Push direction west
         * @type {boolean}
         */
        this.west = true;
        Object.assign(this, init);
    }
}
/**
 * Class PrgResizableHandles
 */
export class PrgResizableHandles {
    /**
     * Constructor
     * @param {Partial<PrgResizableHandles>} init
     */
    constructor(init) {
        /**
         * south
         * @type {boolean}
         */
        this.s = true;
        /**
         * east
         * @type {boolean}
         */
        this.e = true;
        /**
         * north
         * @type {boolean}
         */
        this.n = true;
        /**
         * west
         * @type {boolean}
         */
        this.w = true;
        /**
         * southeast
         * @type {boolean}
         */
        this.se = true;
        /**
         * northeast
         * @type {boolean}
         */
        this.ne = true;
        /**
         * southwest
         * @type {boolean}
         */
        this.sw = true;
        /**
         * northwest
         * @type {boolean}
         */
        this.nw = true;
        Object.assign(this, init);
    }
}
/**
 * Class PrgGridsterItem to be used on items of dashboards
 */
export class PrgGridsterItem {
    /**
     * Constructor
     * @param {Partial<PrgGridsterItem>} init
     */
    constructor(init) {
        /**
         * override grid option resizable.handles for this item
         * @type {PrgResizableHandles}
         */
        this.resizableHandles = new PrgResizableHandles({
            s: null,
            e: null,
            n: null,
            w: null,
            se: null,
            ne: null,
            sw: null,
            nw: null,
        });
        Object.assign(this, init);
    }
}
//# sourceMappingURL=data:application/json;base64,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