import { Injectable } from '@angular/core';
import { PrgError } from '../../../core/models/error.model';
import { PrgSucess } from '../../../core/models/success.model';
import { MockDatabase as MockDatabaseEntityType } from '../../../entity-types/mock/database.mock';
import { AbstractDashboardService } from '../../services/dashboard/abstract-dashboard.service';
import * as i0 from "@angular/core";
import * as i1 from "../../../core/services/notifications/notifications.service";
import * as i2 from "../../../core/services/utility/utility.service";
import * as i3 from "../../../core/services/is-loading-data/isloadingdata.service";
import * as i4 from "../../../core/services/utility/objects-utility.service";
import * as i5 from "../../../core/services/utility/array-utility.service";
import * as i6 from "../../../auth/services/auth/abstract-auth.service";
import * as i7 from "@angular/common/http";
/**
 * Injectable
 */
/**
 * MOCK AbstractDashboardService implementation
 */
export class MockDashboardService extends AbstractDashboardService {
    /**
     * Constructor
     * @param {NotificationsService} notificationsService
     * @param {UtilityService} utilityService
     * @param {IsLoadingDataService} isLoadingData
     * @param {ObjectsUtilityService} objectsUtilityService
     * @param {ArrayUtilityService} arrayUtilityService
     * @param {AbstractAuthService} authService
     * @param {HttpClient} httpClient
     */
    constructor(notificationsService, utilityService, isLoadingData, objectsUtilityService, arrayUtilityService, authService, httpClient) {
        super(httpClient);
        this.notificationsService = notificationsService;
        this.utilityService = utilityService;
        this.isLoadingData = isLoadingData;
        this.objectsUtilityService = objectsUtilityService;
        this.arrayUtilityService = arrayUtilityService;
        this.authService = authService;
        this.httpClient = httpClient;
        /**
         * default time in seconds
         */
        this.sleepTimeSec = 0.5;
    }
    /**
     * Function responsible to store a new Dashboard
     * @param {Dashboard} newDashboard
     * @returns {Promise<Dashboard>}
     */
    createDashboardAsync(newDashboard) {
        return new Promise(async (resolve, reject) => {
            this.isLoadingData.show();
            await this.utilityService.sleepSecAsync(this.sleepTimeSec);
            let checkNameDashboard = MockDatabaseEntityType.dashboard.find((dashboard) => dashboard.name === newDashboard.name);
            this.isLoadingData.hide();
            if (!checkNameDashboard) {
                let dashboard = this.objectsUtilityService.cloneObject(newDashboard);
                dashboard['createdBy'] = this.authService.getLoggedUser()?.id;
                MockDatabaseEntityType.dashboard.push(dashboard);
                const dashboardsUpdated = this.arrayUtilityService.clone(MockDatabaseEntityType.dashboard);
                await this.setDashboardListUpdated(dashboardsUpdated);
                resolve(dashboard);
                this.notificationsService.successNotification(new PrgSucess({
                    titleKey: 'New Dashboard Created.',
                    detailKey: 'Success.',
                }));
            }
            else {
                reject('Dashboard not save');
                this.notificationsService.errorNotification(new PrgError({
                    titleKey: 'Dashboard not save.',
                    detailKey: 'Dashboard name already exists in database',
                }));
            }
        });
    }
    /**
     * Get Dashboard by Id
     * @param {string} dashboardId
     * @returns {Promise<Dashboard>}
     */
    getDashboardByIdAsync(dashboardId) {
        return new Promise(async (resolve, reject) => {
            this.isLoadingData.show();
            await this.utilityService.sleepSecAsync(this.sleepTimeSec);
            const dashboard = this.objectsUtilityService.cloneObject(MockDatabaseEntityType.dashboard.find((dashboard) => {
                return dashboard.id === dashboardId;
            }));
            this.isLoadingData.hide();
            if (dashboard) {
                resolve(dashboard);
            }
            else {
                reject('dashboard could not be loaded');
                this.notificationsService.errorNotification(new PrgError({
                    titleKey: 'dashboard could not be loaded.',
                    detailKey: 'Server is down',
                }));
            }
        });
    }
    /**
     * Function responsible to update a dashboard
     * @param {string} dashboardId
     * @param {Dashboard} dashboard
     * @returns {Promise<Dashboard>}
     */
    updateDashboardAsync(dashboardId, dashboard) {
        return new Promise(async (resolve, reject) => {
            this.isLoadingData.show();
            await this.utilityService.sleepSecAsync(this.sleepTimeSec);
            const dashboardIndexToUpdate = MockDatabaseEntityType.dashboard.findIndex((dashboard) => {
                return dashboard.id == dashboardId;
            });
            if (dashboardIndexToUpdate > -1) {
                MockDatabaseEntityType.dashboard[dashboardIndexToUpdate] =
                    this.objectsUtilityService.cloneObject(dashboard);
                const dashboardsUpdated = this.arrayUtilityService.clone(MockDatabaseEntityType.dashboard);
                await this.setDashboardListUpdated(dashboardsUpdated);
                this.isLoadingData.hide();
                resolve(dashboard);
                this.notificationsService.successNotification(new PrgSucess({
                    titleKey: 'Dashboard updated.',
                    detailKey: 'Success.',
                }));
            }
            else {
                reject('dashboard could not be updated');
                this.notificationsService.errorNotification(new PrgError({
                    titleKey: 'dashboard could not be updated.',
                    detailKey: 'Server is down',
                }));
            }
        });
    }
    /**
     * Get dashboards by User Logged
     * @returns {Promise<Dashboard[]>}
     */
    getDashboardsByUserAsync() {
        return new Promise(async (resolve, reject) => {
            this.isLoadingData.show();
            await this.utilityService.sleepSecAsync(this.sleepTimeSec);
            let userLogged = this.authService.getLoggedUser();
            const dashboards = this.objectsUtilityService.cloneObject(MockDatabaseEntityType.dashboard.filter((dashboard) => {
                return dashboard.createdBy === userLogged.id;
            }));
            this.isLoadingData.hide();
            if (dashboards) {
                resolve(dashboards);
            }
            else {
                reject('dashboards could not be loaded');
                this.notificationsService.errorNotification(new PrgError({
                    titleKey: 'dashboards could not be loaded.',
                    detailKey: 'Server is down',
                }));
            }
        });
    }
}
MockDashboardService.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "14.2.0", ngImport: i0, type: MockDashboardService, deps: [{ token: i1.NotificationsService }, { token: i2.UtilityService }, { token: i3.IsLoadingDataService }, { token: i4.ObjectsUtilityService }, { token: i5.ArrayUtilityService }, { token: i6.AbstractAuthService }, { token: i7.HttpClient }], target: i0.ɵɵFactoryTarget.Injectable });
MockDashboardService.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "14.2.0", ngImport: i0, type: MockDashboardService, providedIn: 'root' });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "14.2.0", ngImport: i0, type: MockDashboardService, decorators: [{
            type: Injectable,
            args: [{
                    providedIn: 'root',
                }]
        }], ctorParameters: function () { return [{ type: i1.NotificationsService }, { type: i2.UtilityService }, { type: i3.IsLoadingDataService }, { type: i4.ObjectsUtilityService }, { type: i5.ArrayUtilityService }, { type: i6.AbstractAuthService }, { type: i7.HttpClient }]; } });
//# sourceMappingURL=data:application/json;base64,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