import { animate, animateChild, group, query, state, style, transition, trigger, } from '@angular/animations';
import { Component, Inject, Input } from '@angular/core';
import { DialogService } from 'primeng/dynamicdialog';
import { ViewMode } from '../../../core/models/constants/view-mode.enum';
import { Dashboard, DashboardItem, MapComponentTypesToClass, } from '../../models/dashboard.model';
import { PrgGridsterConfig, PrgGridsterItem, } from '../../models/prg-gridster-config';
import { PRG_DASHBOARD_CONFIG } from '../../services/prg-dashboard-configuration/prg-dashboard-configuration.service';
import { PrgDashboardItemsOptionsComponent } from '../dashboard-items-options/prg-dashboard-items-options.component';
import { PrgDashboardOptionsComponent } from '../dashboard-options/prg-dashboard-options.component';
import { DASHBOARD_DYNAMIC_FORM } from './dashboard-dynamic-form-structure/dashboard-dynamic-form';
import * as i0 from "@angular/core";
import * as i1 from "primeng/dynamicdialog";
import * as i2 from "../../../layouts/services/main-layout.service";
import * as i3 from "../../../core/services/utility/objects-utility.service";
import * as i4 from "../../../core/services/utility/array-utility.service";
import * as i5 from "../../services/dashboard/abstract-dashboard.service";
import * as i6 from "../../../core/services/utility/utility.service";
import * as i7 from "@angular/router";
import * as i8 from "@angular/common";
import * as i9 from "../../../core/services/notifications/notifications.service";
import * as i10 from "@ngx-translate/core";
import * as i11 from "primeng/tooltip";
import * as i12 from "primeng/api";
import * as i13 from "primeng/table";
import * as i14 from "primeng/button";
import * as i15 from "primeng/image";
import * as i16 from "primeng/ripple";
import * as i17 from "../../../core/components/dynamic-form/prg-dynamic-form.component";
import * as i18 from "angular-gridster2";
import * as i19 from "ng-dynamic-component";
import * as i20 from "primeng/overlaypanel";
import * as i21 from "primeng/dragdrop";
/**
 * Dashboard Component
 */
export class PrgDashboardComponent {
    /**
     * Constructor
     * @param {DialogService} dialogService dialogService
     * @param {MainLayoutService} mainLayoutService
     * @param {ObjectsUtilityService} objectsUtilityService
     * @param {ArrayUtilityService} arrayUtilityService
     * @param {AbstractDashboardService} dashboardService
     * @param {UtilityService} utilityService
     * @param {ActivatedRoute} route
     * @param {Router} router
     * @param {PrgDashboardConfig} prgDashboardConfig
     * @param {LocationStrategy} location
     * @param {NotificationsService} notificationsService
     * @param {TranslateService} translateService
     */
    constructor(dialogService, mainLayoutService, objectsUtilityService, arrayUtilityService, dashboardService, utilityService, route, router, prgDashboardConfig, location, notificationsService, translateService) {
        this.dialogService = dialogService;
        this.mainLayoutService = mainLayoutService;
        this.objectsUtilityService = objectsUtilityService;
        this.arrayUtilityService = arrayUtilityService;
        this.dashboardService = dashboardService;
        this.utilityService = utilityService;
        this.route = route;
        this.router = router;
        this.prgDashboardConfig = prgDashboardConfig;
        this.location = location;
        this.notificationsService = notificationsService;
        this.translateService = translateService;
        /**
         * The dynamic form fields to build dashboard form in template
         */
        this.dashboardDynamicFormFields = this.arrayUtilityService.clone(DASHBOARD_DYNAMIC_FORM.fields);
        /**
         * The view mode for dynamic form
         * @type {ViewMode}
         */
        this.dashboardViewMode = ViewMode.Edit;
        /**
         * The current state of sidebar menu. Expanded is true and collapsed is false.
         *
         * Default is false.
         * @type {boolean}
         */
        this.menuDashboardExpanded = true;
        /**
         * An array of component type list to be shown on sidebar
         * @type {any[]}
         */
        this.componentsType = [];
        /**
         * The base route for editing a dashboard
         * @type {string}
         */
        this.editDashboardPageRoute = '/' +
            this.prgDashboardConfig.dashboardRoutingBasePath +
            '/' +
            this.prgDashboardConfig.dashboardEditRoutingPath;
        /**
         * An array of widgets (dynamic items) displayed on dashboard
         * @type {Array<PrgGridsterItem>}
         */
        this.dashboardItems = [];
        /**
         * A copy dashboard items
         * @type {Array<PrgGridsterItem>}
         * @private
         */
        this.dashboardItemsCopy = [];
        /**
         * A class property used to unsubscribe observables on ngOnDestroy
         * @type {Subscription[]}
         * @private
         */
        this.subscription = [];
    }
    /**
     * ngOnInit
     *
     * Verify if there is any changes on size of main sidebar and update view on dashboard.
     *
     * Verify if there is any changes on Dashboard Options form and reflect them on the view of dashboard.
     *
     * Verify route and if there is a parameter on route, if of dashboard and if so set this as the selected dashboard
     * @returns {Promise<void>}
     */
    async ngOnInit() {
        this.dashboardService.setDashboardOptionsChanges(null);
        this.subscription.push(this.mainLayoutService.getSideBarStateObservable().subscribe(() => {
            this.changedOptions();
        }));
        this.subscription.push(this.dashboardService
            .getDashboardOptionsChangesObservable()
            .subscribe((dashboard) => {
            if (dashboard) {
                if (dashboard.gridsterConfig != null &&
                    dashboard.gridsterItems == null) {
                    this.dashboardOptions = this.objectsUtilityService.cloneObject(dashboard.gridsterConfig);
                    this.dashboardItems = this.arrayUtilityService.clone(this.dashboardItems);
                }
                else {
                    this.dashboardItems = this.arrayUtilityService.clone(dashboard.gridsterItems);
                }
                this.changedOptions();
            }
        }));
        this.dashboardOptions = new PrgGridsterConfig({});
        this.populateComponentsType();
        if (this.router.url.includes(this.editDashboardPageRoute)) {
            await this.getDashboard();
        }
        else {
            await this.setNewDashboardSettings();
        }
        this.changedOptions();
    }
    /**
     * This function will populate components type list on sidebar
     * @private
     */
    populateComponentsType() {
        MapComponentTypesToClass.forEach((value, key) => {
            this.componentsType.push({
                name: key,
                class: value,
            });
        });
    }
    /**
     * Setting up a new dashboard
     * @returns {Promise<void>}
     * @private
     */
    async setNewDashboardSettings() {
        this.dashboard = new Dashboard({});
        this.dashboardCopy = this.objectsUtilityService.cloneObject(this.dashboard);
        this.settingCallBackFunctionDashboard();
        this.dashboardItems = [];
        this.dashboardItemsCopy = this.arrayUtilityService.clone(this.dashboardItems);
    }
    /**
     * This function will get and set the dashboard selected on list
     * @returns {Promise<void>}
     * @private
     */
    async getDashboard() {
        this.idDashboard = this.route?.snapshot?.params['id'];
        try {
            this.dashboard = await this.dashboardService.getDashboardByIdAsync(this.idDashboard);
        }
        catch (e) {
            this.location.back();
        }
        if (this.dashboard) {
            this.dashboardCopy = this.objectsUtilityService.cloneObject(this.dashboard);
            if (this.dashboard?.settings) {
                let dashboardOptionStringToObject = this.utilityService.guiSettingToObject(this.dashboard.settings);
                this.dashboardOptions = this.objectsUtilityService.cloneObject(this.objectsUtilityService.unFlattenObj(dashboardOptionStringToObject, '.'));
            }
            this.settingCallBackFunctionDashboard();
            this.dashboardItems = [];
            if (this.dashboard.items != null && this.dashboard.items.length > 0) {
                this.setDashboardItems();
            }
        }
        else {
            this.location.back();
        }
    }
    /**
     * Setting up items for dashboard
     * @private
     */
    setDashboardItems() {
        this.dashboard.items.map((item) => {
            this.dashboardItems.push(new PrgGridsterItem({
                x: item.x,
                y: item.y,
                cols: item.cols,
                rows: item.rows,
                dashboardId: item.dashboardId,
                item: MapComponentTypesToClass.get(item.componentType),
                settings: item.settings,
                id: item.id ? item.id : this.utilityService.newGuid(),
            }));
        });
        this.dashboardItemsCopy = this.arrayUtilityService.clone(this.dashboardItems);
    }
    /**
     * Setting up call back for dashboard
     * @private
     */
    settingCallBackFunctionDashboard() {
        this.dashboardOptions = {
            ...this.dashboardOptions,
            itemResizeCallback: this.onItemChangeSized.bind(this),
            emptyCellClickCallback: this.emptyCellClick.bind(this),
            emptyCellDropCallback: this.emptyCellClick.bind(this),
        };
        this.dashboardOptionsCopy = this.objectsUtilityService.cloneObject(this.dashboardOptions);
    }
    /**
     * This function is responsible to delete an item of dashboard
     * @param {MouseEvent} event
     * @param {PrgGridsterItem} item
     */
    onDeleteItem(event, item) {
        event.preventDefault();
        event.stopPropagation();
        this.dashboardItems.splice(this.dashboardItems.indexOf(item), 1);
    }
    /**
     * This function is responsible to refresh grid options
     * @private
     */
    changedOptions() {
        //Update Grid on Changes
        if (this.dashboardOptions?.api &&
            this.dashboardOptions?.api?.optionsChanged) {
            this.dashboardOptions?.api?.optionsChanged();
        }
    }
    /**
     * This function will add an item on the first possible position of the dashboard
     * @param {ComponentType} componentType
     */
    onAddItemFirstPossiblePosition(componentType = null) {
        if (componentType) {
            this.componentTypeSelected = componentType;
        }
        if (this.componentTypeSelected != null) {
            this.dashboardItems.push({
                x: 0,
                y: 0,
                cols: 1,
                rows: 1,
                settings: null,
                item: this.componentTypeSelected.class,
                id: this.utilityService.newGuid(),
            });
        }
    }
    /**
     * This function is responsible to clear all items of dashboard
     * @returns {Promise<void>}
     */
    async clearAllItems() {
        if (await this.notificationsService.prgConfirmationService('messages.clear-confirmation', await this.translateService.get('dashboard ?').toPromise())) {
            this.dashboardItems = [];
        }
        else {
            return;
        }
    }
    /**
     * ngOnDestroy
     *
     * Unsubscribe subscriptions
     */
    ngOnDestroy() {
        this.subscription.forEach((subscription) => {
            subscription.unsubscribe();
        });
        this.subscription = [];
    }
    /**
     * This function will add an item on an empty cell that was clicked
     * @param {MouseEvent} event
     * @param {PrgGridsterItem} item
     * @private
     */
    emptyCellClick(event = null, item) {
        if (this.componentTypeSelected != null) {
            this.dashboardItems.push({
                ...item,
                settings: null,
                item: this.componentTypeSelected.class,
                id: this.utilityService.newGuid(),
            });
        }
    }
    /**
     * This function is responsible to open dashboard options modal and handle actions when it closes
     */
    openOptionsDashboard() {
        this.dynamicDialogRef = this.dialogService.open(PrgDashboardOptionsComponent, {
            header: 'DashBoardOptions',
            showHeader: true,
            width: '50%',
            height: '60vh',
            baseZIndex: 10000,
            data: {
                dashboardOptions: this.dashboardOptions,
                dashboardItems: this.dashboardItems,
            },
            draggable: true,
            resizable: true,
            maximizable: true,
        });
        this.subscription.push(this.dynamicDialogRef.onClose.subscribe((dashboard) => {
            if (dashboard) {
                this.dashboardOptions =
                    this.objectsUtilityService.cloneObject(dashboard);
                this.dashboardOptionsCopy = this.objectsUtilityService.cloneObject(this.dashboardOptions);
            }
            else {
                this.dashboardOptions = this.objectsUtilityService.cloneObject(this.dashboardOptionsCopy);
                this.dashboardItems = this.arrayUtilityService.clone(this.dashboardItems);
            }
            this.changedOptions();
        }));
    }
    /**
     *  This function will save the dashboard
     * @param entity
     * @returns {Promise<void>}
     */
    async saveDashboard(entity) {
        let isNewDashboard = false;
        this.dashboard = this.objectsUtilityService.cloneObject(entity);
        if (entity.id == null) {
            this.dashboard.id = this.utilityService.newGuid();
            isNewDashboard = true;
        }
        this.prepareDashboardItemsToSave();
        try {
            if (isNewDashboard) {
                await this.dashboardService.createDashboardAsync(this.dashboard);
            }
            else {
                await this.dashboardService.updateDashboardAsync(this.dashboard.id, this.dashboard);
            }
            this.dashboardCopy = this.objectsUtilityService.cloneObject(this.dashboard);
            this.dashboardOptionsCopy = this.objectsUtilityService.cloneObject(this.dashboardOptions);
            this.dashboardItemsCopy = this.arrayUtilityService.clone(this.dashboardItems);
        }
        catch (e) { }
        /*
        this.changedOptions();*/
    }
    /**
     * This function is auxiliary function of saveDashboard. Prepares dashboard items to be saved.
     * @private
     */
    prepareDashboardItemsToSave() {
        let itemsToSave = [];
        this.dashboardItems.map((dashboardItem) => {
            let auxComponentType = this.utilityService.getKeyByValueOnMap(MapComponentTypesToClass, dashboardItem.item);
            itemsToSave.push(new DashboardItem({
                x: dashboardItem.x,
                y: dashboardItem.y,
                id: dashboardItem.id,
                cols: dashboardItem.cols,
                rows: dashboardItem.rows,
                settings: dashboardItem.settings,
                dashboardId: this.dashboard.id,
                componentType: auxComponentType,
            }));
        });
        this.dashboard = {
            ...this.dashboard,
            items: itemsToSave,
            settings: JSON.stringify(this.objectsUtilityService.flattenObj(this.dashboardOptions, '.')),
        };
    }
    /**
     * This function is call whenever height or width of each item changes
     * @param {PrgGridsterItem} item
     * @param {GridsterItemComponentInterface} itemComponent
     * @private
     */
    onItemChangeSized(item, itemComponent) {
        this.dashboardService.setDashboardItemsResize(itemComponent);
    }
    /**
     * This function is call when a drag occurs on dashboard.Set componentTypeSelected.
     * @param componentType
     */
    onDragItem(componentType) {
        this.componentTypeSelected = componentType;
    }
    /**
     * This function is responsible to handle actions after an item was drop in an empty cell on the dashboard
     * @param event
     * @param {GridsterComponent} gridsterComponent
     */
    onDropItemEmptyCell(event, gridsterComponent) {
        event.preventDefault();
        event.stopPropagation();
        gridsterComponent.emptyCell.emptyCellDragDrop(event);
    }
    /**
     * This function is responsible to handle actions after an item was drop in an occupied cell on the dashboard
     * @param event
     * @param {PrgGridsterItem} item
     * @param {GridsterComponent} gridsterComponent
     * @constructor
     */
    OnDropItem(event, item, gridsterComponent) {
        if (this.componentTypeSelected != null) {
            let newPos = this.dashboardOptions.api.getFirstPossiblePosition(item);
            let index = this.dashboardItems.indexOf(item);
            this.dashboardItems[index] = { ...this.dashboardItems[index], ...newPos };
            this.dashboardItems.push({
                ...item,
                item: this.componentTypeSelected.class,
                settings: null,
                id: this.utilityService.newGuid(),
            });
        }
    }
    /**
     * This function is call whenever sidebar expands or collapses
     */
    onShowHideSideBarDetails() {
        this.menuDashboardExpanded = !this.menuDashboardExpanded;
        this.changedOptions();
    }
    /**
     * This function is responsible to open dashboard items options modal and handle actions when it closes
     * @param {PrgGridsterItem} item
     */
    onOpenItemOptions(item) {
        this.dashboardItemsCopy = this.arrayUtilityService.clone(this.dashboardItems);
        this.dynamicDialogRefItems = this.dialogService.open(PrgDashboardItemsOptionsComponent, {
            header: 'DashBoardItemOptions',
            showHeader: true,
            width: '50%',
            height: '60vh',
            baseZIndex: 10000,
            data: {
                item: item,
                index: this.dashboardItems.indexOf(item),
            },
            draggable: true,
            resizable: true,
            maximizable: true,
        });
        this.subscription.push(this.dynamicDialogRefItems.onClose.subscribe((item) => {
            if (item) {
                this.dashboardItems[item.index] =
                    this.objectsUtilityService.cloneObject(item.item);
            }
            this.dashboardItemsCopy = this.arrayUtilityService.clone(this.dashboardItems);
        }));
    }
}
PrgDashboardComponent.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "14.2.0", ngImport: i0, type: PrgDashboardComponent, deps: [{ token: i1.DialogService }, { token: i2.MainLayoutService }, { token: i3.ObjectsUtilityService }, { token: i4.ArrayUtilityService }, { token: i5.AbstractDashboardService }, { token: i6.UtilityService }, { token: i7.ActivatedRoute }, { token: i7.Router }, { token: PRG_DASHBOARD_CONFIG }, { token: i8.LocationStrategy }, { token: i9.NotificationsService }, { token: i10.TranslateService }], target: i0.ɵɵFactoryTarget.Component });
PrgDashboardComponent.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "14.0.0", version: "14.2.0", type: PrgDashboardComponent, selector: "prg-dashboard", inputs: { editDashboardPageRoute: "editDashboardPageRoute" }, providers: [DialogService], ngImport: i0, template: "<div class=\"grid-layout\">\r\n  <div class=\"grid grid-nogutter\">\r\n  <div class=\"col grid-nogutter grid-layout \" >\r\n    <gridster    pDroppable=\"dd\"   (onDrop)=\"onDropItemEmptyCell($event,gridsterComponent)\" #gridsterComponent  [options]=\"dashboardOptions\" >\r\n\r\n    <gridster-item pDroppable=\"dd\"    (onDrop)=\"OnDropItem($event,item,gridsterComponent)\"   [item]=\"item\" *ngFor=\"let item of dashboardItems\" >\r\n\r\n        <div class=\"relative\" >\r\n          <div class=\"container-buttons\"  >\r\n            <button    (click)=\"onDeleteItem($event,item)\"  pButton pRipple type=\"button\" icon=\"pi pi-times\" class=\"p-button-rounded p-button-danger mr-1\"></button>\r\n            <button (click)=\"onOpenItemOptions(item)\"    pButton pRipple type=\"button\" icon=\"pi pi pi-cog\" class=\"ml-1 p-button-rounded p-button-info\"></button>\r\n\r\n            </div>\r\n      <ndc-dynamic  [ndcDynamicInputs]=\"{itemId:item.id,itemSettings:item.settings}\"  [ndcDynamicComponent]=\"item.item\">\r\n\r\n      </ndc-dynamic>\r\n        </div>\r\n    </gridster-item>\r\n  </gridster>\r\n  </div>\r\n  <div class=\"col-fixed grid-nogutter ml-2 \">\r\n    <aside class=\"shadow-5 \" >\r\n      <div  class=\"flex flex-column sideBarDashboard p-1\" [@OnExpandSideBarDashboard]=\"menuDashboardExpanded?'open':'closed'\">\r\n      <div class=\"text-center mt-2\">\r\n\r\n        <button  tooltipPosition=\"bottom\" [ngClass]=\"menuDashboardExpanded?'':'mb-2'\" pTooltip=\"Dashboard Options\"  (click)=\"openOptionsDashboard()\" pButton pRipple type=\"button\" icon=\"pi pi pi-cog\" class=\"text-center p-button-rounded p-button-info\"></button>\r\n        <button tooltipPosition=\"bottom\"  pTooltip=\"Clear All Items\" [ngClass]=\"menuDashboardExpanded?'mx-3':'mb-2'\" (click)=\"clearAllItems()\"  pButton pRipple type=\"button\" icon=\"pi pi-times\" class=\"text-center p-button-rounded p-button-danger\"></button>\r\n        <button pTooltip=\"Save Dashboard\" [ngClass]=\"menuDashboardExpanded?'':'mb-2'\"   [disabled]=\"!dynamicFormComponent?.form?.valid\" (click)=\"saveDashboard(dynamicFormComponent.entityTemp)\"  pButton pRipple type=\"button\" icon=\"pi pi-save\" class=\"text-center p-button-rounded p-button-success\"></button>\r\n        <button  pTooltip=\"Select Item\" [ngClass]=\"menuDashboardExpanded?'mx-2':'mb-2'\"  *ngIf=\"!menuDashboardExpanded\"  (click)=\"op.toggle($event)\"  pButton pRipple type=\"button\" icon=\"pi pi-search\" class=\"text-center p-button-rounded p-button-info\"></button>\r\n        <p-overlayPanel  #op [dismissable]=\"true\" [showCloseIcon]=\"true\">\r\n          <ng-template pTemplate>\r\n            <p-table styleClass=\"tableOverlay\"  [value]=\"componentsType\" selectionMode=\"single\" [(selection)]=\"componentTypeSelected\"  [paginator]=\"true\" [rows]=\"5\" responsiveLayout=\"scroll\">\r\n              <ng-template pTemplate=\"header\">\r\n                <tr>\r\n                  <th style=\"text-align: center\" pSortableColumn=\"name\">Name<p-sortIcon field=\"name\"></p-sortIcon></th>\r\n                  <th  style=\"text-align: center\">Image</th>\r\n                </tr>\r\n              </ng-template>\r\n              <ng-template pTemplate=\"body\" let-rowData let-componentType>\r\n                <tr  (dblclick)=\"onAddItemFirstPossiblePosition(componentType)\" [pSelectableRow]=\"rowData\" pDraggable=\"dd\" (onDragStart)=\"onDragItem(componentType)\" >\r\n                  <td      style=\"text-align: center\">{{'dashboard.components.'+ componentType.name + '.label'|translate}}</td>\r\n                  <td    style=\"height: auto;width: 150px\"> <p-image [src]=\"'dashboard.components.'+ componentType.name + '.imageUrl'|translate\"  class=\"w-2\" alt=\"\"></p-image></td>\r\n                </tr>\r\n              </ng-template>\r\n            </p-table>\r\n          </ng-template>\r\n        </p-overlayPanel>\r\n      </div>\r\n        <section  [@OnExpandSideBarTextDashboard]=\"menuDashboardExpanded?'show-content':'hide-content'\">\r\n          <div class=\"pt-2 \" [hidden]=\"!menuDashboardExpanded\">\r\n\r\n\r\n            <prg-dynamic-form #dynamicFormComponent [viewMode]=\"dashboardViewMode\"\r\n                              [fields]=\"dashboardDynamicFormFields\"\r\n                              [entity]=\"dashboard\">\r\n            </prg-dynamic-form>\r\n\r\n            <p-table [paginator]=\"true\"  selectionMode=\"single\" [rows]=\"3\" [value]=\"componentsType\" responsiveLayout=\"scroll\" [(selection)]=\"componentTypeSelected\">\r\n              <ng-template pTemplate=\"header\">\r\n                <tr>\r\n                  <th style=\"text-align: center\" pSortableColumn=\"name\">Name<p-sortIcon field=\"name\"></p-sortIcon></th>\r\n                  <th  style=\"text-align: center\">Image</th>\r\n                </tr>\r\n              </ng-template>\r\n              <ng-template pTemplate=\"body\" let-rowData let-componentType>\r\n                <tr [pSelectableRow]=\"rowData\"  (dblclick)=\"onAddItemFirstPossiblePosition(componentType)\"  pDraggable=\"dd\" (onDragStart)=\"onDragItem(componentType)\" >\r\n                  <td      style=\"text-align: center\">{{'dashboard.components.'+ componentType.name + '.label'|translate}}</td>\r\n                  <td    style=\"height: auto;width: 150px\"> <p-image [src]=\"'dashboard.components.'+ componentType.name + '.imageUrl'|translate\"  class=\"w-2\" alt=\"\"></p-image></td>\r\n                </tr>\r\n              </ng-template>\r\n            </p-table>\r\n\r\n          </div>\r\n        </section>\r\n\r\n      <div class=\"flex flex-column pb-2 flex-grow-1 justify-content-end\" [ngClass]=\"menuDashboardExpanded?'align-items-start':'align-items-center'\">\r\n\r\n        <button (click)=\" onShowHideSideBarDetails()\" pButton pRipple type=\"button\" icon=\"{{menuDashboardExpanded?'pi pi-angle-double-right':'pi pi-angle-double-left'}} text-2xl\" class=\"p-button-rounded p-button-text p-button-plain\"></button></div>\r\n      </div>\r\n    </aside>\r\n  </div>\r\n  </div>\r\n\r\n</div>\r\n\r\n\r\n<!--\"\r\n\"-->\r\n", styles: [".sideBarDashboard{height:calc(100vh - 70px)}:host ::ng-deep .p-multiselect{min-width:10rem;width:15rem}.grid-layout{height:calc(100vh - 70px);width:100%}.container-buttons{opacity:.5;position:absolute;top:50%;left:50%;transform:translate(-50%,-50%);-ms-transform:translate(-50%,-50%);z-index:1}.container-buttons:hover{opacity:1}\n"], dependencies: [{ kind: "directive", type: i11.Tooltip, selector: "[pTooltip]", inputs: ["tooltipPosition", "tooltipEvent", "appendTo", "positionStyle", "tooltipStyleClass", "tooltipZIndex", "escape", "showDelay", "hideDelay", "life", "positionTop", "positionLeft", "fitContent", "pTooltip", "tooltipDisabled", "tooltipOptions"] }, { kind: "directive", type: i12.PrimeTemplate, selector: "[pTemplate]", inputs: ["type", "pTemplate"] }, { kind: "directive", type: i8.NgClass, selector: "[ngClass]", inputs: ["class", "ngClass"] }, { kind: "directive", type: i8.NgForOf, selector: "[ngFor][ngForOf]", inputs: ["ngForOf", "ngForTrackBy", "ngForTemplate"] }, { kind: "directive", type: i8.NgIf, selector: "[ngIf]", inputs: ["ngIf", "ngIfThen", "ngIfElse"] }, { kind: "component", type: i13.Table, selector: "p-table", inputs: ["frozenColumns", "frozenValue", "style", "styleClass", "tableStyle", "tableStyleClass", "paginator", "pageLinks", "rowsPerPageOptions", "alwaysShowPaginator", "paginatorPosition", "paginatorDropdownAppendTo", "paginatorDropdownScrollHeight", "currentPageReportTemplate", "showCurrentPageReport", "showJumpToPageDropdown", "showJumpToPageInput", "showFirstLastIcon", "showPageLinks", "defaultSortOrder", "sortMode", "resetPageOnSort", "selectionMode", "selectionPageOnly", "contextMenuSelection", "contextMenuSelectionMode", "dataKey", "metaKeySelection", "rowSelectable", "rowTrackBy", "lazy", "lazyLoadOnInit", "compareSelectionBy", "csvSeparator", "exportFilename", "filters", "globalFilterFields", "filterDelay", "filterLocale", "expandedRowKeys", "editingRowKeys", "rowExpandMode", "scrollable", "scrollDirection", "rowGroupMode", "scrollHeight", "virtualScroll", "virtualScrollItemSize", "virtualScrollOptions", "virtualScrollDelay", "frozenWidth", "responsive", "contextMenu", "resizableColumns", "columnResizeMode", "reorderableColumns", "loading", "loadingIcon", "showLoader", "rowHover", "customSort", "showInitialSortBadge", "autoLayout", "exportFunction", "exportHeader", "stateKey", "stateStorage", "editMode", "groupRowsBy", "groupRowsByOrder", "responsiveLayout", "breakpoint", "virtualRowHeight", "value", "columns", "first", "rows", "totalRecords", "sortField", "sortOrder", "multiSortMeta", "selection", "selectAll"], outputs: ["selectAllChange", "selectionChange", "contextMenuSelectionChange", "onRowSelect", "onRowUnselect", "onPage", "onSort", "onFilter", "onLazyLoad", "onRowExpand", "onRowCollapse", "onContextMenuSelect", "onColResize", "onColReorder", "onRowReorder", "onEditInit", "onEditComplete", "onEditCancel", "onHeaderCheckboxToggle", "sortFunction", "firstChange", "rowsChange", "onStateSave", "onStateRestore"] }, { kind: "directive", type: i13.SortableColumn, selector: "[pSortableColumn]", inputs: ["pSortableColumn", "pSortableColumnDisabled"] }, { kind: "directive", type: i13.SelectableRow, selector: "[pSelectableRow]", inputs: ["pSelectableRow", "pSelectableRowIndex", "pSelectableRowDisabled"] }, { kind: "component", type: i13.SortIcon, selector: "p-sortIcon", inputs: ["field"] }, { kind: "directive", type: i14.ButtonDirective, selector: "[pButton]", inputs: ["iconPos", "loadingIcon", "label", "icon", "loading"] }, { kind: "component", type: i15.Image, selector: "p-image", inputs: ["imageClass", "imageStyle", "styleClass", "style", "src", "alt", "width", "height", "appendTo", "preview", "showTransitionOptions", "hideTransitionOptions"], outputs: ["onShow", "onHide", "onImageError"] }, { kind: "directive", type: i16.Ripple, selector: "[pRipple]" }, { kind: "component", type: i17.PrgDynamicFormComponent, selector: "prg-dynamic-form", inputs: ["context", "fields", "actions", "entity", "viewMode", "formGroupType"], outputs: ["actionOutput", "fieldChangeOutput"] }, { kind: "component", type: i18.GridsterComponent, selector: "gridster", inputs: ["options"] }, { kind: "component", type: i18.GridsterItemComponent, selector: "gridster-item", inputs: ["item"], outputs: ["itemInit", "itemChange", "itemResize"] }, { kind: "component", type: i19.DynamicComponent, selector: "ndc-dynamic", inputs: ["ndcDynamicComponent", "ndcDynamicInjector", "ndcDynamicProviders", "ndcDynamicContent"], outputs: ["ndcDynamicCreated"] }, { kind: "directive", type: i19.DynamicIoDirective, selector: "[ndcDynamicInputs],[ndcDynamicOutputs]", inputs: ["ndcDynamicInputs", "ndcDynamicOutputs"], exportAs: ["ndcDynamicIo"] }, { kind: "component", type: i20.OverlayPanel, selector: "p-overlayPanel", inputs: ["dismissable", "showCloseIcon", "style", "styleClass", "appendTo", "autoZIndex", "ariaCloseLabel", "baseZIndex", "focusOnShow", "showTransitionOptions", "hideTransitionOptions"], outputs: ["onShow", "onHide"] }, { kind: "directive", type: i21.Draggable, selector: "[pDraggable]", inputs: ["pDraggable", "dragEffect", "dragHandle", "pDraggableDisabled"], outputs: ["onDragStart", "onDragEnd", "onDrag"] }, { kind: "directive", type: i21.Droppable, selector: "[pDroppable]", inputs: ["pDroppable", "pDroppableDisabled", "dropEffect"], outputs: ["onDragEnter", "onDragLeave", "onDrop"] }, { kind: "pipe", type: i10.TranslatePipe, name: "translate" }], animations: [
        trigger('OnExpandSideBarDashboard', [
            state('open', style({
                width: '100%',
            })),
            state('closed', style({
                width: '70px',
            })),
            transition('open => closed', [
                group([
                    query(':self', [animate('0s')]),
                    query('@OnExpandSideBarTextDashboard', [animateChild()]),
                ]),
            ]),
            transition('closed => open', [
                group([
                    query(':self', [animate('0.2s')]),
                    query('@OnExpandSideBarTextDashboard', [animateChild()]),
                ]),
            ]),
        ]),
        trigger('OnExpandSideBarTextDashboard', [
            // ...
            state('show-content', style({
                opacity: '1',
            })),
            state('hide-content', style({
                opacity: '0',
                overflow: 'hidden',
            })),
            transition('show-content => hide-content', [animate('0s')]),
            transition('hide-content => show-content', [animate('0.2s 0.3s')]),
        ]),
    ] });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "14.2.0", ngImport: i0, type: PrgDashboardComponent, decorators: [{
            type: Component,
            args: [{ selector: 'prg-dashboard', providers: [DialogService], animations: [
                        trigger('OnExpandSideBarDashboard', [
                            state('open', style({
                                width: '100%',
                            })),
                            state('closed', style({
                                width: '70px',
                            })),
                            transition('open => closed', [
                                group([
                                    query(':self', [animate('0s')]),
                                    query('@OnExpandSideBarTextDashboard', [animateChild()]),
                                ]),
                            ]),
                            transition('closed => open', [
                                group([
                                    query(':self', [animate('0.2s')]),
                                    query('@OnExpandSideBarTextDashboard', [animateChild()]),
                                ]),
                            ]),
                        ]),
                        trigger('OnExpandSideBarTextDashboard', [
                            // ...
                            state('show-content', style({
                                opacity: '1',
                            })),
                            state('hide-content', style({
                                opacity: '0',
                                overflow: 'hidden',
                            })),
                            transition('show-content => hide-content', [animate('0s')]),
                            transition('hide-content => show-content', [animate('0.2s 0.3s')]),
                        ]),
                    ], template: "<div class=\"grid-layout\">\r\n  <div class=\"grid grid-nogutter\">\r\n  <div class=\"col grid-nogutter grid-layout \" >\r\n    <gridster    pDroppable=\"dd\"   (onDrop)=\"onDropItemEmptyCell($event,gridsterComponent)\" #gridsterComponent  [options]=\"dashboardOptions\" >\r\n\r\n    <gridster-item pDroppable=\"dd\"    (onDrop)=\"OnDropItem($event,item,gridsterComponent)\"   [item]=\"item\" *ngFor=\"let item of dashboardItems\" >\r\n\r\n        <div class=\"relative\" >\r\n          <div class=\"container-buttons\"  >\r\n            <button    (click)=\"onDeleteItem($event,item)\"  pButton pRipple type=\"button\" icon=\"pi pi-times\" class=\"p-button-rounded p-button-danger mr-1\"></button>\r\n            <button (click)=\"onOpenItemOptions(item)\"    pButton pRipple type=\"button\" icon=\"pi pi pi-cog\" class=\"ml-1 p-button-rounded p-button-info\"></button>\r\n\r\n            </div>\r\n      <ndc-dynamic  [ndcDynamicInputs]=\"{itemId:item.id,itemSettings:item.settings}\"  [ndcDynamicComponent]=\"item.item\">\r\n\r\n      </ndc-dynamic>\r\n        </div>\r\n    </gridster-item>\r\n  </gridster>\r\n  </div>\r\n  <div class=\"col-fixed grid-nogutter ml-2 \">\r\n    <aside class=\"shadow-5 \" >\r\n      <div  class=\"flex flex-column sideBarDashboard p-1\" [@OnExpandSideBarDashboard]=\"menuDashboardExpanded?'open':'closed'\">\r\n      <div class=\"text-center mt-2\">\r\n\r\n        <button  tooltipPosition=\"bottom\" [ngClass]=\"menuDashboardExpanded?'':'mb-2'\" pTooltip=\"Dashboard Options\"  (click)=\"openOptionsDashboard()\" pButton pRipple type=\"button\" icon=\"pi pi pi-cog\" class=\"text-center p-button-rounded p-button-info\"></button>\r\n        <button tooltipPosition=\"bottom\"  pTooltip=\"Clear All Items\" [ngClass]=\"menuDashboardExpanded?'mx-3':'mb-2'\" (click)=\"clearAllItems()\"  pButton pRipple type=\"button\" icon=\"pi pi-times\" class=\"text-center p-button-rounded p-button-danger\"></button>\r\n        <button pTooltip=\"Save Dashboard\" [ngClass]=\"menuDashboardExpanded?'':'mb-2'\"   [disabled]=\"!dynamicFormComponent?.form?.valid\" (click)=\"saveDashboard(dynamicFormComponent.entityTemp)\"  pButton pRipple type=\"button\" icon=\"pi pi-save\" class=\"text-center p-button-rounded p-button-success\"></button>\r\n        <button  pTooltip=\"Select Item\" [ngClass]=\"menuDashboardExpanded?'mx-2':'mb-2'\"  *ngIf=\"!menuDashboardExpanded\"  (click)=\"op.toggle($event)\"  pButton pRipple type=\"button\" icon=\"pi pi-search\" class=\"text-center p-button-rounded p-button-info\"></button>\r\n        <p-overlayPanel  #op [dismissable]=\"true\" [showCloseIcon]=\"true\">\r\n          <ng-template pTemplate>\r\n            <p-table styleClass=\"tableOverlay\"  [value]=\"componentsType\" selectionMode=\"single\" [(selection)]=\"componentTypeSelected\"  [paginator]=\"true\" [rows]=\"5\" responsiveLayout=\"scroll\">\r\n              <ng-template pTemplate=\"header\">\r\n                <tr>\r\n                  <th style=\"text-align: center\" pSortableColumn=\"name\">Name<p-sortIcon field=\"name\"></p-sortIcon></th>\r\n                  <th  style=\"text-align: center\">Image</th>\r\n                </tr>\r\n              </ng-template>\r\n              <ng-template pTemplate=\"body\" let-rowData let-componentType>\r\n                <tr  (dblclick)=\"onAddItemFirstPossiblePosition(componentType)\" [pSelectableRow]=\"rowData\" pDraggable=\"dd\" (onDragStart)=\"onDragItem(componentType)\" >\r\n                  <td      style=\"text-align: center\">{{'dashboard.components.'+ componentType.name + '.label'|translate}}</td>\r\n                  <td    style=\"height: auto;width: 150px\"> <p-image [src]=\"'dashboard.components.'+ componentType.name + '.imageUrl'|translate\"  class=\"w-2\" alt=\"\"></p-image></td>\r\n                </tr>\r\n              </ng-template>\r\n            </p-table>\r\n          </ng-template>\r\n        </p-overlayPanel>\r\n      </div>\r\n        <section  [@OnExpandSideBarTextDashboard]=\"menuDashboardExpanded?'show-content':'hide-content'\">\r\n          <div class=\"pt-2 \" [hidden]=\"!menuDashboardExpanded\">\r\n\r\n\r\n            <prg-dynamic-form #dynamicFormComponent [viewMode]=\"dashboardViewMode\"\r\n                              [fields]=\"dashboardDynamicFormFields\"\r\n                              [entity]=\"dashboard\">\r\n            </prg-dynamic-form>\r\n\r\n            <p-table [paginator]=\"true\"  selectionMode=\"single\" [rows]=\"3\" [value]=\"componentsType\" responsiveLayout=\"scroll\" [(selection)]=\"componentTypeSelected\">\r\n              <ng-template pTemplate=\"header\">\r\n                <tr>\r\n                  <th style=\"text-align: center\" pSortableColumn=\"name\">Name<p-sortIcon field=\"name\"></p-sortIcon></th>\r\n                  <th  style=\"text-align: center\">Image</th>\r\n                </tr>\r\n              </ng-template>\r\n              <ng-template pTemplate=\"body\" let-rowData let-componentType>\r\n                <tr [pSelectableRow]=\"rowData\"  (dblclick)=\"onAddItemFirstPossiblePosition(componentType)\"  pDraggable=\"dd\" (onDragStart)=\"onDragItem(componentType)\" >\r\n                  <td      style=\"text-align: center\">{{'dashboard.components.'+ componentType.name + '.label'|translate}}</td>\r\n                  <td    style=\"height: auto;width: 150px\"> <p-image [src]=\"'dashboard.components.'+ componentType.name + '.imageUrl'|translate\"  class=\"w-2\" alt=\"\"></p-image></td>\r\n                </tr>\r\n              </ng-template>\r\n            </p-table>\r\n\r\n          </div>\r\n        </section>\r\n\r\n      <div class=\"flex flex-column pb-2 flex-grow-1 justify-content-end\" [ngClass]=\"menuDashboardExpanded?'align-items-start':'align-items-center'\">\r\n\r\n        <button (click)=\" onShowHideSideBarDetails()\" pButton pRipple type=\"button\" icon=\"{{menuDashboardExpanded?'pi pi-angle-double-right':'pi pi-angle-double-left'}} text-2xl\" class=\"p-button-rounded p-button-text p-button-plain\"></button></div>\r\n      </div>\r\n    </aside>\r\n  </div>\r\n  </div>\r\n\r\n</div>\r\n\r\n\r\n<!--\"\r\n\"-->\r\n", styles: [".sideBarDashboard{height:calc(100vh - 70px)}:host ::ng-deep .p-multiselect{min-width:10rem;width:15rem}.grid-layout{height:calc(100vh - 70px);width:100%}.container-buttons{opacity:.5;position:absolute;top:50%;left:50%;transform:translate(-50%,-50%);-ms-transform:translate(-50%,-50%);z-index:1}.container-buttons:hover{opacity:1}\n"] }]
        }], ctorParameters: function () { return [{ type: i1.DialogService }, { type: i2.MainLayoutService }, { type: i3.ObjectsUtilityService }, { type: i4.ArrayUtilityService }, { type: i5.AbstractDashboardService }, { type: i6.UtilityService }, { type: i7.ActivatedRoute }, { type: i7.Router }, { type: undefined, decorators: [{
                    type: Inject,
                    args: [PRG_DASHBOARD_CONFIG]
                }] }, { type: i8.LocationStrategy }, { type: i9.NotificationsService }, { type: i10.TranslateService }]; }, propDecorators: { editDashboardPageRoute: [{
                type: Input
            }] } });
//# sourceMappingURL=data:application/json;base64,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