import { Component, Input } from '@angular/core';
import { MapComponentTypesToClass, } from '../../models/dashboard.model';
import { PrgDraggable, PrgGridsterConfig, PrgGridsterItem, PrgResizable, } from '../../models/prg-gridster-config';
import * as i0 from "@angular/core";
import * as i1 from "../../../core/services/utility/objects-utility.service";
import * as i2 from "../../../core/services/utility/array-utility.service";
import * as i3 from "../../services/dashboard/abstract-dashboard.service";
import * as i4 from "../../../core/services/utility/utility.service";
import * as i5 from "@angular/router";
import * as i6 from "@angular/common";
import * as i7 from "../../../layouts/services/main-layout.service";
import * as i8 from "primeng/dropdown";
import * as i9 from "angular-gridster2";
import * as i10 from "ng-dynamic-component";
/**
 * Display dashboard component
 */
export class PrgDashboardDisplayComponent {
    /**
     * Constructor
     * @param {ObjectsUtilityService} objectsUtilityService
     * @param {ArrayUtilityService} arrayUtilityService
     * @param {AbstractDashboardService} dashboardService
     * @param {UtilityService} utilityService
     * @param {ActivatedRoute} route
     * @param {LocationStrategy} location
     * @param {MainLayoutService} mainLayoutService
     */
    constructor(objectsUtilityService, arrayUtilityService, dashboardService, utilityService, route, location, mainLayoutService) {
        this.objectsUtilityService = objectsUtilityService;
        this.arrayUtilityService = arrayUtilityService;
        this.dashboardService = dashboardService;
        this.utilityService = utilityService;
        this.route = route;
        this.location = location;
        this.mainLayoutService = mainLayoutService;
        /**
         * Show/hide header.Default is true.
         * @type {boolean}
         */
        this.displayHeader = true;
        /**
         * Show/hide dashboard selection dropdown.Default is true.
         * @type {boolean}
         */
        this.showDropdown = true;
        /**
         * An array of widgets (dynamic items) displayed on dashboard
         * @type {Array<PrgGridsterItem>}
         */
        this.dashboardItems = [];
        /**
         * Show/Hide dashboard on template.Default is false.
         * @type {boolean}
         */
        this.displayDashboard = false;
        /**
         * A class property used to unsubscribe observables on ngOnDestroy
         * @type {Subscription[]}
         * @private
         */
        this.subscription = [];
    }
    /**
     * ngOnInit
     *
     * Verify if there is any changes on size of main sidebar and update view on dashboard.
     *
     * Load all dashboard created by logged user.
     *
     * Verify if there is a parameter on route, name of dashboard and if so set this as the selected dashboard
     *
     * @returns {Promise<void>}
     */
    async ngOnInit() {
        this.subscription.push(this.mainLayoutService.getSideBarStateObservable().subscribe(() => {
            this.changedOptions();
        }));
        this.mainLayoutService.setDisplayHeaderState(this.displayHeader);
        this.dashboardsByUser =
            await this.dashboardService.getDashboardsByUserAsync();
        this.nameDashboard = this.route?.snapshot?.params['name'];
        if (this.nameDashboard) {
            this.selectedDashboard = this.dashboardsByUser.find((dashboard) => dashboard.name === this.nameDashboard);
            if (this.showDropdown != null) {
                await this.settingDashboard();
                this.displayDashboard = true;
            }
            else {
                this.location.back();
            }
        }
    }
    /**
     * Setting up the dashboard selected
     * @returns {Promise<void>}
     * @private
     */
    async settingDashboard() {
        this.dashboardOptions = new PrgGridsterConfig({});
        if (this.selectedDashboard) {
            if (this.selectedDashboard?.settings) {
                let dashboardOptionStringToObject = this.utilityService.guiSettingToObject(this.selectedDashboard.settings);
                this.dashboardOptions = this.objectsUtilityService.cloneObject(this.objectsUtilityService.unFlattenObj(dashboardOptionStringToObject, '.'));
            }
            this.settingCallBackFunctionDashboard();
            this.dashboardItems = [];
            if (this.selectedDashboard.items != null &&
                this.selectedDashboard.items.length > 0) {
                this.setDashboardItems();
            }
        }
        else {
            this.location.back();
        }
    }
    /**
     * Setting up items for current dashboard selected
     * @private
     */
    setDashboardItems() {
        this.selectedDashboard.items.map((item) => {
            this.dashboardItems.push(new PrgGridsterItem({
                x: item.x,
                y: item.y,
                cols: item.cols,
                rows: item.rows,
                dashboardId: item.dashboardId,
                item: MapComponentTypesToClass.get(item.componentType),
                settings: item.settings,
                id: item.id,
            }));
        });
    }
    /**
     * Setting up call back functions and additional options for the current dashboard selected
     * @private
     */
    settingCallBackFunctionDashboard() {
        this.dashboardOptions = {
            ...this.dashboardOptions,
            itemResizeCallback: this.onItemChangeSized.bind(this),
            draggable: new PrgDraggable({
                enabled: false,
            }),
            resizable: new PrgResizable({
                enabled: false,
            }),
            swap: false,
        };
    }
    /**
     * This function is call whenever height or width of each item changes
     * @param {PrgGridsterItem} item
     * @param {GridsterItemComponentInterface} itemComponent
     * @private
     */
    onItemChangeSized(item, itemComponent) {
        this.dashboardService.setDashboardItemsResize(itemComponent);
    }
    /**
     * This function is responsible to refresh grid options
     * @private
     */
    changedOptions() {
        if (this.dashboardOptions?.api &&
            this.dashboardOptions?.api?.optionsChanged) {
            this.dashboardOptions?.api?.optionsChanged();
        }
    }
    /**
     * This function is call when a dashboard is selected on dropdown
     * @param $event
     * @returns {Promise<void>}
     */
    async onChangeDashboard($event) {
        this.displayDashboard = false;
        this.selectedDashboard = this.dashboardsByUser.find((dashboard) => $event.value === dashboard.id);
        await this.settingDashboard();
        this.displayDashboard = true;
    }
    /**
     * ngOnDestroy
     *
     * Unsubscribe  subscriptions and set the display header state to true
     */
    ngOnDestroy() {
        this.mainLayoutService.setDisplayHeaderState(true);
        this.subscription.forEach((subscription) => {
            subscription.unsubscribe();
        });
        this.subscription = [];
    }
}
PrgDashboardDisplayComponent.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "14.2.0", ngImport: i0, type: PrgDashboardDisplayComponent, deps: [{ token: i1.ObjectsUtilityService }, { token: i2.ArrayUtilityService }, { token: i3.AbstractDashboardService }, { token: i4.UtilityService }, { token: i5.ActivatedRoute }, { token: i6.LocationStrategy }, { token: i7.MainLayoutService }], target: i0.ɵɵFactoryTarget.Component });
PrgDashboardDisplayComponent.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "14.0.0", version: "14.2.0", type: PrgDashboardDisplayComponent, selector: "prg-dashboard-display", inputs: { displayHeader: "displayHeader", showDropdown: "showDropdown" }, ngImport: i0, template: "<div >\r\n  <p-dropdown *ngIf=\"showDropdown\" placeholder=\"Select a Dashboard\" (onChange)=\"onChangeDashboard($event)\" optionValue=\"id\" optionLabel=\"name\" [options]=\"dashboardsByUser\" ></p-dropdown>\r\n\r\n    <div class=\"col grid-nogutter \"\r\n         [ngClass]=\"displayHeader&&showDropdown?'grid-layout-with-header-dropdown':displayHeader&&!showDropdown?'grid-layout-with-header':!displayHeader&&showDropdown?'grid-layout-with-dropdown':'grid-layout'\">\r\n      <gridster  *ngIf=\"displayDashboard\"     [options]=\"dashboardOptions\" >\r\n\r\n        <gridster-item    [item]=\"item\" *ngFor=\"let item of dashboardItems\" >\r\n\r\n\r\n            <ndc-dynamic  [ndcDynamicInputs]=\"{itemId:item.id,itemSettings:item.settings}\"  [ndcDynamicComponent]=\"item.item\">\r\n\r\n            </ndc-dynamic>\r\n\r\n        </gridster-item>\r\n      </gridster>\r\n\r\n</div>\r\n\r\n</div>\r\n", styles: [".display-with-header-dropdown,.grid-layout-with-header-dropdown{height:calc(100vh - 130px);width:100%}.display-with-header,.grid-layout-with-header{height:calc(100vh - 80px);width:100%}.display-with-dropdown,.grid-layout-with-dropdown{height:calc(100vh - 60px);width:100%}.display,.grid-layout{height:calc(100vh - 10px);width:100%}\n"], dependencies: [{ kind: "directive", type: i6.NgClass, selector: "[ngClass]", inputs: ["class", "ngClass"] }, { kind: "directive", type: i6.NgForOf, selector: "[ngFor][ngForOf]", inputs: ["ngForOf", "ngForTrackBy", "ngForTemplate"] }, { kind: "directive", type: i6.NgIf, selector: "[ngIf]", inputs: ["ngIf", "ngIfThen", "ngIfElse"] }, { kind: "component", type: i8.Dropdown, selector: "p-dropdown", inputs: ["scrollHeight", "filter", "name", "style", "panelStyle", "styleClass", "panelStyleClass", "readonly", "required", "editable", "appendTo", "tabindex", "placeholder", "filterPlaceholder", "filterLocale", "inputId", "selectId", "dataKey", "filterBy", "autofocus", "resetFilterOnHide", "dropdownIcon", "optionLabel", "optionValue", "optionDisabled", "optionGroupLabel", "optionGroupChildren", "autoDisplayFirst", "group", "showClear", "emptyFilterMessage", "emptyMessage", "lazy", "virtualScroll", "virtualScrollItemSize", "virtualScrollOptions", "autoZIndex", "baseZIndex", "showTransitionOptions", "hideTransitionOptions", "ariaFilterLabel", "ariaLabel", "ariaLabelledBy", "filterMatchMode", "maxlength", "tooltip", "tooltipPosition", "tooltipPositionStyle", "tooltipStyleClass", "autofocusFilter", "disabled", "itemSize", "options", "filterValue"], outputs: ["onChange", "onFilter", "onFocus", "onBlur", "onClick", "onShow", "onHide", "onClear", "onLazyLoad"] }, { kind: "component", type: i9.GridsterComponent, selector: "gridster", inputs: ["options"] }, { kind: "component", type: i9.GridsterItemComponent, selector: "gridster-item", inputs: ["item"], outputs: ["itemInit", "itemChange", "itemResize"] }, { kind: "component", type: i10.DynamicComponent, selector: "ndc-dynamic", inputs: ["ndcDynamicComponent", "ndcDynamicInjector", "ndcDynamicProviders", "ndcDynamicContent"], outputs: ["ndcDynamicCreated"] }, { kind: "directive", type: i10.DynamicIoDirective, selector: "[ndcDynamicInputs],[ndcDynamicOutputs]", inputs: ["ndcDynamicInputs", "ndcDynamicOutputs"], exportAs: ["ndcDynamicIo"] }] });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "14.2.0", ngImport: i0, type: PrgDashboardDisplayComponent, decorators: [{
            type: Component,
            args: [{ selector: 'prg-dashboard-display', template: "<div >\r\n  <p-dropdown *ngIf=\"showDropdown\" placeholder=\"Select a Dashboard\" (onChange)=\"onChangeDashboard($event)\" optionValue=\"id\" optionLabel=\"name\" [options]=\"dashboardsByUser\" ></p-dropdown>\r\n\r\n    <div class=\"col grid-nogutter \"\r\n         [ngClass]=\"displayHeader&&showDropdown?'grid-layout-with-header-dropdown':displayHeader&&!showDropdown?'grid-layout-with-header':!displayHeader&&showDropdown?'grid-layout-with-dropdown':'grid-layout'\">\r\n      <gridster  *ngIf=\"displayDashboard\"     [options]=\"dashboardOptions\" >\r\n\r\n        <gridster-item    [item]=\"item\" *ngFor=\"let item of dashboardItems\" >\r\n\r\n\r\n            <ndc-dynamic  [ndcDynamicInputs]=\"{itemId:item.id,itemSettings:item.settings}\"  [ndcDynamicComponent]=\"item.item\">\r\n\r\n            </ndc-dynamic>\r\n\r\n        </gridster-item>\r\n      </gridster>\r\n\r\n</div>\r\n\r\n</div>\r\n", styles: [".display-with-header-dropdown,.grid-layout-with-header-dropdown{height:calc(100vh - 130px);width:100%}.display-with-header,.grid-layout-with-header{height:calc(100vh - 80px);width:100%}.display-with-dropdown,.grid-layout-with-dropdown{height:calc(100vh - 60px);width:100%}.display,.grid-layout{height:calc(100vh - 10px);width:100%}\n"] }]
        }], ctorParameters: function () { return [{ type: i1.ObjectsUtilityService }, { type: i2.ArrayUtilityService }, { type: i3.AbstractDashboardService }, { type: i4.UtilityService }, { type: i5.ActivatedRoute }, { type: i6.LocationStrategy }, { type: i7.MainLayoutService }]; }, propDecorators: { displayHeader: [{
                type: Input
            }], showDropdown: [{
                type: Input
            }] } });
//# sourceMappingURL=data:application/json;base64,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