import { Injectable, EventEmitter } from "@angular/core";
import { BehaviorSubject } from "rxjs";
import * as i0 from "@angular/core";
/**
 * Injectable
 */
/**
 * Generic context service, provides a way to store information in any context
 */
export class ContextGenericService {
    constructor() {
        /**
         * Map that holds all the variables and actions.
         * <br />-> key - owner name
         * <br />-> value object that holds a map for variables and a map for actions
         */
        this.context = new Map();
        /**
         * Array that holds all subscriptions, used to clean up on destroy
         */
        this.subscriptions = [];
    }
    /**
     * Clean up all
     */
    ngOnDestroy() {
        while (this.subscriptions.length > 0) {
            this.unsubscribe(this.subscriptions[0]);
        }
    }
    /**
     * Get owner context using name, if the context does not exist, it is created
     * @param ownerName Name of the owner
     * @returns The owner full context
     */
    getOwnerContext(ownerName) {
        let ownerContext = this.context[ownerName];
        if (!ownerContext) {
            // owner context does not exist, create it
            ownerContext = {
                variables: new Map(),
                actions: new Map(),
            };
            this.context[ownerName] = ownerContext;
        }
        return ownerContext;
    }
    /**
     * Get a variable subject using owner and name, if the variable does not exist, it is created
     * @param ownerName Name of the owner
     * @param variableName Name of the variable
     * @returns Subject of the variable
     */
    getVariable(ownerName, variableName) {
        // Check if owner context exist
        let ownerContext = this.getOwnerContext(ownerName);
        // check if variable exist for this owner
        let variable = ownerContext.variables[variableName];
        if (!variable) {
            // variable does not exist, create it
            variable = new BehaviorSubject(null);
            ownerContext.variables[variableName] = variable;
        }
        return variable;
    }
    /**
     * Subscribes to a variable subject using owner and name, if the variable does not exist, it is created
     * @param ownerName Name of the owner
     * @param variableName Name of the variable
     * @param callback callback function of the subscriber
     * @returns Susbcription
     */
    subscribeVariable(ownerName, variableName, callback) {
        const subject = this.getVariable(ownerName, variableName);
        const subscription = subject.asObservable().subscribe((data) => {
            callback(this.cloneObject(data));
        });
        this.subscriptions.push(subscription);
        return subscription;
    }
    /**
     *  // TODO: This method should be changed to a generic utils class
     */
    cloneObject(obj) {
        return obj ? Object.assign({}, obj) : null; // TODO: check best way to clone a full deep copy
    }
    /**
     * Set data to a variable subject using owner and name, if the variable does not exist, it is created
     * @param ownerName Name of the owner
     * @param variableName Name of the variable
     * @param data Data to be set
     */
    setVariableData(ownerName, variableName, data) {
        const subject = this.getVariable(ownerName, variableName);
        const newValue = this.cloneObject(data);
        subject.next(newValue);
    }
    /**
     * Get an action event using owner and name, if the action does not exist, it is created
     * @param ownerName Name of the owner
     * @param actionName Name of the action
     * @returns Event of the action
     */
    getAction(ownerName, actionName) {
        // Check if owner context exist
        let ownerContext = this.getOwnerContext(ownerName);
        // check if variable exist for this owner
        let action = ownerContext.actions[actionName];
        if (!action) {
            // variable does not exist, create it
            action = new EventEmitter(true);
            ownerContext.actions[actionName] = action;
        }
        return action;
    }
    /**
     * Subscribes to an action using owner and name, if the variable does not exist, it is created
     * @param ownerName Name of the owner
     * @param actionName Name of the action
     * @param callback callback function of the subscriber
     * @returns Susbcription
     */
    subscribeAction(ownerName, actionName, callback) {
        const subject = this.getAction(ownerName, actionName);
        const subscription = subject.subscribe((data) => {
            callback(this.cloneObject(data));
        });
        this.subscriptions.push(subscription);
        return subscription;
    }
    /**
     * Send action using owner and name, if the action does not exist, it is created
     * @param ownerName Name of the owner
     * @param actionName Name of the action
     * @param data Data to be set
     */
    sendAction(ownerName, actionName, data) {
        const action = this.getAction(ownerName, actionName);
        action.emit(this.cloneObject(data));
    }
    /**
     * Unsubscribes a subscription
     * @param subscription Subscription to unsubscribe
     */
    unsubscribe(subscription) {
        if (!subscription)
            return;
        if (Array.isArray(subscription)) {
            // TODO:
        }
        else {
            const index = this.subscriptions.findIndex((s) => s === subscription);
            subscription.unsubscribe();
            if (index >= 0) {
                this.subscriptions.splice(index, 1);
            }
        }
    }
}
ContextGenericService.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "14.2.0", ngImport: i0, type: ContextGenericService, deps: [], target: i0.ɵɵFactoryTarget.Injectable });
ContextGenericService.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "14.2.0", ngImport: i0, type: ContextGenericService });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "14.2.0", ngImport: i0, type: ContextGenericService, decorators: [{
            type: Injectable
        }] });
//# sourceMappingURL=data:application/json;base64,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