/**
 * Injectable
 */
import { Injectable } from '@angular/core';
import { UniversalState } from '../../../core/models/base-entity.model';
import { PrgError } from '../../../core/models/error.model';
import { PrgSucess } from '../../../core/models/success.model';
import { ConfigItemsPath, } from '../../models/configuration-item.model';
import { AbstractConfigurationItemsService } from '../../services/configuration-items/abstract-configuration-items.service';
import { MockDatabase } from '../database.mock';
import * as i0 from "@angular/core";
import * as i1 from "../../../core/services/notifications/notifications.service";
import * as i2 from "../../../core/services/utility/utility.service";
import * as i3 from "../../../core/services/is-loading-data/isloadingdata.service";
import * as i4 from "../../../core/services/utility/objects-utility.service";
import * as i5 from "../../../core/services/utility/array-utility.service";
import * as i6 from "../../../auth/services/auth/abstract-auth.service";
/**
 * Injectable
 */
/**
 * MOCK AbstractConfigurationItemsService implementation
 */
export class MockConfigurationItemsService extends AbstractConfigurationItemsService {
    /**
     * Constructor
     * @param {NotificationsService} notificationsService
     * @param {UtilityService} utilityService
     * @param {IsLoadingDataService} isLoadingData
     * @param {ObjectsUtilityService} objectsUtilityService
     * @param {ArrayUtilityService} arrayUtilityService
     * @param {AbstractAuthService} authService
     */
    constructor(notificationsService, utilityService, isLoadingData, objectsUtilityService, arrayUtilityService, authService) {
        super();
        this.notificationsService = notificationsService;
        this.utilityService = utilityService;
        this.isLoadingData = isLoadingData;
        this.objectsUtilityService = objectsUtilityService;
        this.arrayUtilityService = arrayUtilityService;
        this.authService = authService;
        /**
         * default time in seconds
         */
        this.sleepTimeSec = 0.5;
    }
    /**
     * Get a list of all configuration items
     * @returns {Promise<ConfigurationItem[]>}
     */
    getAllConfigurationItemsAsync() {
        return new Promise(async (resolve, reject) => {
            this.isLoadingData.show();
            await this.utilityService.sleepSecAsync(this.sleepTimeSec);
            const configsItems = this.arrayUtilityService.clone(MockDatabase.ConfigurationItems);
            this.isLoadingData.hide();
            if (configsItems) {
                resolve(configsItems);
            }
            else {
                reject('Config Items could not be loaded');
                this.notificationsService.errorNotification(new PrgError({
                    titleKey: 'Config Items could not be loaded.',
                    detailKey: 'Server is down',
                }));
            }
        });
    }
    /**
     * Update a configuration items
     * @param configItemData
     * @returns {Promise<ConfigurationItem[]>}
     */
    updateConfigurationItemsAsync(configItemData) {
        return new Promise(async (resolve, reject) => {
            this.isLoadingData.show();
            await this.utilityService.sleepSecAsync(this.sleepTimeSec);
            this.isLoadingData.hide();
            if (configItemData.path && configItemData.path != '') {
                Object.keys(configItemData).map((key) => {
                    if (key != 'path') {
                        let configItemToUpdate = MockDatabase.ConfigurationItems.find((configItem) => configItem.name === key &&
                            configItem.path === configItemData.path);
                        if (configItemToUpdate) {
                            configItemToUpdate.value = configItemData[key];
                        }
                        else {
                            let newConfigItem = this.objectsUtilityService.cloneObject({
                                path: configItemData.path,
                                name: key,
                                value: configItemData[key],
                                id: this.utilityService.newGuid(),
                                guiSettings: null,
                                universalStateId: UniversalState.Active,
                                createdOn: new Date(),
                                operationId: null,
                                modifiedBy: this.authService.getLoggedUser()?.id
                                    ? this.authService.getLoggedUser().id
                                    : null,
                                modifiedOn: new Date(),
                                createdBy: this.authService.getLoggedUser()?.id
                                    ? this.authService.getLoggedUser().id
                                    : null,
                                transactionId: null,
                                readonly: null,
                            });
                            MockDatabase.ConfigurationItems.push(newConfigItem);
                        }
                    }
                });
                this.notificationsService.successNotification(new PrgSucess({
                    titleKey: 'Configuration Item Updated.',
                    detailKey: 'Success.',
                }));
                this.setConfigurationItems(MockDatabase.ConfigurationItems);
                resolve(MockDatabase.ConfigurationItems);
            }
            else {
                reject('Configuration Items could not be save.');
                this.notificationsService.errorNotification(new PrgError({
                    titleKey: 'Error.',
                    detailKey: 'Config Item Path is not present',
                }));
            }
        });
    }
    /**
     * Get configuration items by path
     * @param {string} path
     * @returns {Promise<any>}
     */
    getConfigurationItemsByPathAsync(path) {
        return new Promise(async (resolve, reject) => {
            this.isLoadingData.show();
            await this.utilityService.sleepSecAsync(this.sleepTimeSec);
            const configItemsBypath = this.arrayUtilityService.clone(MockDatabase.ConfigurationItems.filter((configItems) => {
                return configItems.path.includes(path);
            }));
            const userAndDefaultsConfigsItems = {};
            const configItemsDefaults = configItemsBypath.filter((configItemsDefaults) => {
                return configItemsDefaults.path === path + ConfigItemsPath.Defaults;
            });
            const configItemsUser = configItemsBypath.filter((configItemsUser) => {
                return (configItemsUser.path ===
                    path + this.authService.getLoggedUser()?.email);
            });
            Object.assign(userAndDefaultsConfigsItems, {
                user: configItemsUser,
                defaults: configItemsDefaults,
            });
            this.isLoadingData.hide();
            if (userAndDefaultsConfigsItems) {
                resolve(userAndDefaultsConfigsItems);
            }
            else {
                reject('Config Items could not be loaded');
                this.notificationsService.errorNotification(new PrgError({
                    titleKey: 'Config Items could not be loaded.',
                    detailKey: 'Server is down',
                }));
            }
        });
    }
}
MockConfigurationItemsService.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "14.2.0", ngImport: i0, type: MockConfigurationItemsService, deps: [{ token: i1.NotificationsService }, { token: i2.UtilityService }, { token: i3.IsLoadingDataService }, { token: i4.ObjectsUtilityService }, { token: i5.ArrayUtilityService }, { token: i6.AbstractAuthService }], target: i0.ɵɵFactoryTarget.Injectable });
MockConfigurationItemsService.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "14.2.0", ngImport: i0, type: MockConfigurationItemsService, providedIn: 'root' });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "14.2.0", ngImport: i0, type: MockConfigurationItemsService, decorators: [{
            type: Injectable,
            args: [{
                    providedIn: 'root',
                }]
        }], ctorParameters: function () { return [{ type: i1.NotificationsService }, { type: i2.UtilityService }, { type: i3.IsLoadingDataService }, { type: i4.ObjectsUtilityService }, { type: i5.ArrayUtilityService }, { type: i6.AbstractAuthService }]; } });
//# sourceMappingURL=data:application/json;base64,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