import { Component } from '@angular/core';
import { Validators } from '@angular/forms';
import { BaseActionKey, } from '../../../core/components/dynamic-form/models/base-action';
import { ViewMode } from '../../../core/models/constants/view-mode.enum';
import { CONFIG_ITEMS_DYNAMIC_FORM } from '../../dynamic-form-structure/config-items-dynamic-form';
import { ConfigItemsPath, } from '../../models/configuration-item.model';
import * as i0 from "@angular/core";
import * as i1 from "../../services/configuration-items/abstract-configuration-items.service";
import * as i2 from "../../../core/services/utility/array-utility.service";
import * as i3 from "../../../core/services/utility/utility.service";
import * as i4 from "@ngx-translate/core";
import * as i5 from "../../../core/services/utility/objects-utility.service";
import * as i6 from "@angular/common";
import * as i7 from "primeng/tree";
import * as i8 from "../../../core/components/dynamic-form/prg-dynamic-form.component";
/**
 * Config items admin component
 */
export class PrgConfigItemsAdminComponent {
    /**
     * Constructor
     * @param {AbstractConfigurationItemsService} configurationItemsService
     * @param {ArrayUtilityService} arrayUtilityService
     * @param {UtilityService} utilityService
     * @param {TranslateService} translateService
     * @param {ObjectsUtilityService} objectUtility
     */
    constructor(configurationItemsService, arrayUtilityService, utilityService, translateService, objectUtility) {
        this.configurationItemsService = configurationItemsService;
        this.arrayUtilityService = arrayUtilityService;
        this.utilityService = utilityService;
        this.translateService = translateService;
        this.objectUtility = objectUtility;
        /**
         * The view mode type of the form. Default is "read"
         * @type {ViewMode}
         */
        this.configItemsViewMode = ViewMode.Read;
        /**
         * A boolean property of the class to show/hide form. Default is false(hide)
         * @type {boolean}
         */
        this.initForm = false;
        /**
         * An object with the data to build form , keys and values
         * @type {any}
         */
        this.entityConfigItems = {};
        /**
         * The dynamic form fields to build dynamic form config items in template
         */
        this.configItemsDynamicFormFields = this.arrayUtilityService.clone(CONFIG_ITEMS_DYNAMIC_FORM.fields);
        /**
         * The dynamic form actions to build dynamic form config items in template
         */
        this.configItemsDynamicFormActions = this.arrayUtilityService.clone(CONFIG_ITEMS_DYNAMIC_FORM.actions);
        /**
         * A copy of entityConfigItems object
         * @type {any}
         * @private
         */
        this.entityConfigItemCopy = {};
        /**
         * The list of configuration items
         * @type {ConfigurationItem[]}
         */
        this.configItemsList = [];
        /**
         * The configuration items defaults selected on tree
         * @type {ConfigurationItem[]}
         */
        this.configItemsSelectedDefaults = [];
        /**
         * The configuration items selected on tree
         * @type {ConfigurationItem[]}
         */
        this.configItemsSelected = [];
    }
    /**
     * ngOnInit
     *
     * Get all configuration item
     * @returns {Promise<void>}
     */
    async ngOnInit() {
        //fetch all configuration items
        this.configItemsList =
            await this.configurationItemsService.getAllConfigurationItemsAsync();
        this.setTreeData();
    }
    /**
     * This function call rearrange all config item paths and build tree menu
     * @private
     */
    setTreeData() {
        //get only the path of all configuration items
        const configItemsPath = this.configItemsList.map((configItem) => configItem.path);
        //Build tree
        this.configItemsTree = this.arrayUtilityService.clone(this.utilityService.arrayOfPathsToTree(configItemsPath, ':'));
    }
    /**
     * This function is responsible toe execute actions whenever a node is selected on tree
     * @param {TreeNode} node
     */
    onSelectItem(node) {
        // Return View mode back to read
        this.configItemsViewMode = ViewMode.Read;
        // Set config items selected
        this.setConfigItemsSelected(node);
        //Dynamic Form Fields
        this.configItemsDynamicFormFields = this.arrayUtilityService.clone(this.getAndParseGuiSettingsItems());
        // Assign config item data to form - entities
        this.assignConfigItemDataToForm();
        // Title of Form
        this.setTitleForm();
        //Initiate Form
        this.initForm = true;
    }
    /**
     * This function is responsible to get the complete path of item selected and their default path if
     * they exist. Then it will get the data of config items selected
     * @param {TreeNode} node
     * @private
     */
    setConfigItemsSelected(node) {
        const pathConfigItem = PrgConfigItemsAdminComponent.getCompletePathOfItem(node);
        // Path of default config-items
        const pathConfigItemDefaults = pathConfigItem.substring(0, pathConfigItem.lastIndexOf(':')) +
            ':' +
            ConfigItemsPath.Defaults;
        // Config-items of the selectable node
        this.configItemsSelected = this.configItemsList.filter((configItem) => configItem.path === pathConfigItem);
        // Config-items Defaults of the selectable node
        this.configItemsSelectedDefaults = this.configItemsList.filter((configItem) => configItem.path === pathConfigItemDefaults);
    }
    /**
     * This function is responsible to get the complete path of the item in tree
     * @param {TreeNode} node
     * @returns {string}
     * @private
     */
    static getCompletePathOfItem(node) {
        // Find the complete path of the item
        const completePathItem = [];
        let flag = true;
        let parent = node?.parent;
        while (flag == true) {
            if (parent) {
                completePathItem.unshift(parent.label);
                parent = parent?.parent;
            }
            else {
                flag = false;
            }
        }
        completePathItem.push(node?.label);
        return completePathItem.join(':');
    }
    /**
     * This function it will set the title of the form
     * @private
     */
    setTitleForm() {
        // Title of Form
        if (this.entityConfigItems.path.includes(ConfigItemsPath.UserPreferences)) {
            this.titleForm = this.entityConfigItems.path
                .substring(0, this.entityConfigItems.path.lastIndexOf(':'))
                .replaceAll(':', '.')
                .toLowerCase();
        }
        else {
            this.titleForm = this.entityConfigItems?.path
                .replaceAll(':', '.')
                .toLowerCase();
        }
    }
    /**
     * This function will assign config items data to form
     * @private
     */
    assignConfigItemDataToForm() {
        // Assign config item data to form - entities
        const mapConfigItemsIntoEntity = [];
        this.configItemsSelected.map((configItems) => {
            mapConfigItemsIntoEntity[configItems.name] = configItems.value;
            mapConfigItemsIntoEntity['path'] = configItems.path;
        });
        if (this.configItemsSelectedDefaults != null &&
            this.configItemsSelectedDefaults.length > 0) {
            this.configItemsSelectedDefaults.map(async (configItems) => {
                if (!mapConfigItemsIntoEntity[configItems.name]) {
                    mapConfigItemsIntoEntity[configItems.name] = null;
                }
                let fieldGuiSettings = this.configItemsDynamicFormFields.find((value) => value.key === configItems.name);
                if (fieldGuiSettings.basePathTranslation.includes('lookup-tables')) {
                    fieldGuiSettings.placeholder =
                        this.translateService.instant(fieldGuiSettings.basePathTranslation +
                            '.' +
                            fieldGuiSettings.key +
                            '.items.' +
                            configItems.value +
                            '.name') + ' (default)';
                }
                else {
                    fieldGuiSettings.placeholder = configItems.value + ' (default)';
                }
            });
        }
        this.entityConfigItems = Object.assign({}, mapConfigItemsIntoEntity);
        this.entityConfigItemCopy = this.objectUtility.cloneObject(this.entityConfigItems);
    }
    /**
     * This function is responsible to get the gui settings of the item selected and parse it
     * @returns {any}
     * @private
     */
    getAndParseGuiSettingsItems() {
        //Gui Settings
        let guiSettings;
        if (this.configItemsSelectedDefaults == null ||
            this.configItemsSelectedDefaults.length == 0) {
            guiSettings = this.configItemsSelected.map((configItem) => {
                return this.utilityService.guiSettingToObject(configItem.guiSettings);
            });
        }
        else {
            guiSettings = this.configItemsSelectedDefaults.map((configItem) => {
                return this.utilityService.guiSettingToObject(configItem.guiSettings);
            });
            if (this.selectedNode.key.includes(ConfigItemsPath.Defaults)) {
                guiSettings.forEach((gui) => {
                    gui.validators = Validators.required;
                });
            }
        }
        return guiSettings;
    }
    /**
     * This function is responsible to handle actions from the dynamic config items form
     * @param {DynamicFormActionOutput} event
     * @returns {Promise<void>}
     */
    async onActionConfigItems(event) {
        switch (event.action) {
            case BaseActionKey.Save:
                await this.configItemsSave(event);
                break;
            case BaseActionKey.Cancel:
                this.configItemsViewMode = ViewMode.Read;
                this.entityConfigItems = this.objectUtility.cloneObject(this.entityConfigItemCopy);
                break;
            case BaseActionKey.Edit:
                this.configItemsViewMode = ViewMode.Edit;
                break;
            default:
                break;
        }
    }
    /**
     * This function is responsible to save a new config item or update an existing one
     * @param {DynamicFormActionOutput} event
     * @returns {Promise<void>}
     * @private
     */
    async configItemsSave(event) {
        this.configItemsViewMode = ViewMode.Read;
        try {
            this.initForm = false;
            await this.configurationItemsService.updateConfigurationItemsAsync(event.formEntity);
            this.configItemsList = this.arrayUtilityService.clone(this.configurationItemsService.getConfigurationItemsValue());
            this.setTreeData();
            if (this.selectedNode) {
                this.onSelectItem(this.selectedNode);
            }
            this.initForm = true;
        }
        catch (e) { }
    }
}
PrgConfigItemsAdminComponent.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "14.2.0", ngImport: i0, type: PrgConfigItemsAdminComponent, deps: [{ token: i1.AbstractConfigurationItemsService }, { token: i2.ArrayUtilityService }, { token: i3.UtilityService }, { token: i4.TranslateService }, { token: i5.ObjectsUtilityService }], target: i0.ɵɵFactoryTarget.Component });
PrgConfigItemsAdminComponent.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "14.0.0", version: "14.2.0", type: PrgConfigItemsAdminComponent, selector: "prg-config-items-admin", ngImport: i0, template: "<div class=\"grid grid-nogutter\">\r\n  <div class=\"col-fixed grid-nogutter\">\r\n    <p-tree [(selection)]=\"selectedNode\"  [filter]=\"true\" filterPlaceholder=\"Search Item...\" selectionMode=\"single\"\r\n            (onNodeSelect)=\"onSelectItem($event.node)\" layout=\"vertical\" [value]=\"configItemsTree\"></p-tree>\r\n  </div>\r\n <div class=\"col grid-nogutter px-2 \" *ngIf=\"initForm\">\r\n   <p [ngClass]=\"['text-center' ,'bg-blue-50' ,'text-3xl' ,'border-round', 'mb-2', 'py-3', 'w-full', 'border-300', 'border-1' ,'shadow-1', 'mt-0' ,'text-700']\"  >{{titleForm + '.label'|translate|uppercase}}</p>\r\n\r\n   <prg-dynamic-form [viewMode]=\"configItemsViewMode\" [actions]=\"configItemsDynamicFormActions\"\r\n                     [fields]=\"configItemsDynamicFormFields\" [entity]=\"entityConfigItems\"\r\n                     (actionOutput)=\"onActionConfigItems($event)\">\r\n   </prg-dynamic-form>\r\n </div>\r\n</div>\r\n\r\n", styles: [":host ::ng-deep .p-dropdown .p-dropdown-label.p-placeholder{color:#08f}:host ::ng-deep p-inputnumber.p-inputnumber-clearable .p-inputnumber-input{width:260px}:host ::ng-deep p-inputnumber.p-inputnumber-clearable .p-inputnumber-input{padding-right:.75rem}:host ::ng-deep .p-inputnumber-buttons-stacked .p-inputnumber-input::placeholder{color:#08f!important;font-size:15px}\n"], dependencies: [{ kind: "directive", type: i6.NgClass, selector: "[ngClass]", inputs: ["class", "ngClass"] }, { kind: "directive", type: i6.NgIf, selector: "[ngIf]", inputs: ["ngIf", "ngIfThen", "ngIfElse"] }, { kind: "component", type: i7.Tree, selector: "p-tree", inputs: ["value", "selectionMode", "selection", "style", "styleClass", "contextMenu", "layout", "draggableScope", "droppableScope", "draggableNodes", "droppableNodes", "metaKeySelection", "propagateSelectionUp", "propagateSelectionDown", "loading", "loadingIcon", "emptyMessage", "ariaLabel", "togglerAriaLabel", "ariaLabelledBy", "validateDrop", "filter", "filterBy", "filterMode", "filterPlaceholder", "filteredNodes", "filterLocale", "scrollHeight", "lazy", "virtualScroll", "virtualScrollItemSize", "virtualScrollOptions", "indentation", "_templateMap", "trackBy", "virtualNodeHeight"], outputs: ["selectionChange", "onNodeSelect", "onNodeUnselect", "onNodeExpand", "onNodeCollapse", "onNodeContextMenuSelect", "onNodeDrop", "onLazyLoad", "onScroll", "onScrollIndexChange", "onFilter"] }, { kind: "component", type: i8.PrgDynamicFormComponent, selector: "prg-dynamic-form", inputs: ["context", "fields", "actions", "entity", "viewMode", "formGroupType"], outputs: ["actionOutput", "fieldChangeOutput"] }, { kind: "pipe", type: i6.UpperCasePipe, name: "uppercase" }, { kind: "pipe", type: i4.TranslatePipe, name: "translate" }] });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "14.2.0", ngImport: i0, type: PrgConfigItemsAdminComponent, decorators: [{
            type: Component,
            args: [{ selector: 'prg-config-items-admin', template: "<div class=\"grid grid-nogutter\">\r\n  <div class=\"col-fixed grid-nogutter\">\r\n    <p-tree [(selection)]=\"selectedNode\"  [filter]=\"true\" filterPlaceholder=\"Search Item...\" selectionMode=\"single\"\r\n            (onNodeSelect)=\"onSelectItem($event.node)\" layout=\"vertical\" [value]=\"configItemsTree\"></p-tree>\r\n  </div>\r\n <div class=\"col grid-nogutter px-2 \" *ngIf=\"initForm\">\r\n   <p [ngClass]=\"['text-center' ,'bg-blue-50' ,'text-3xl' ,'border-round', 'mb-2', 'py-3', 'w-full', 'border-300', 'border-1' ,'shadow-1', 'mt-0' ,'text-700']\"  >{{titleForm + '.label'|translate|uppercase}}</p>\r\n\r\n   <prg-dynamic-form [viewMode]=\"configItemsViewMode\" [actions]=\"configItemsDynamicFormActions\"\r\n                     [fields]=\"configItemsDynamicFormFields\" [entity]=\"entityConfigItems\"\r\n                     (actionOutput)=\"onActionConfigItems($event)\">\r\n   </prg-dynamic-form>\r\n </div>\r\n</div>\r\n\r\n", styles: [":host ::ng-deep .p-dropdown .p-dropdown-label.p-placeholder{color:#08f}:host ::ng-deep p-inputnumber.p-inputnumber-clearable .p-inputnumber-input{width:260px}:host ::ng-deep p-inputnumber.p-inputnumber-clearable .p-inputnumber-input{padding-right:.75rem}:host ::ng-deep .p-inputnumber-buttons-stacked .p-inputnumber-input::placeholder{color:#08f!important;font-size:15px}\n"] }]
        }], ctorParameters: function () { return [{ type: i1.AbstractConfigurationItemsService }, { type: i2.ArrayUtilityService }, { type: i3.UtilityService }, { type: i4.TranslateService }, { type: i5.ObjectsUtilityService }]; } });
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicHJnLWNvbmZpZy1pdGVtcy1hZG1pbi5jb21wb25lbnQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi8uLi8uLi8uLi8uLi8uLi9wcm9qZWN0cy9QcmdGcmFtZXdvcmsuRkUuQ29yZUxpYi9zcmMvbGliL21vZHVsZXMvY29uZmlndXJhdGlvbi1pdGVtcy9jb21wb25lbnRzL2NvbmZpZy1pdGVtcy1hZG1pbi9wcmctY29uZmlnLWl0ZW1zLWFkbWluLmNvbXBvbmVudC50cyIsIi4uLy4uLy4uLy4uLy4uLy4uLy4uLy4uL3Byb2plY3RzL1ByZ0ZyYW1ld29yay5GRS5Db3JlTGliL3NyYy9saWIvbW9kdWxlcy9jb25maWd1cmF0aW9uLWl0ZW1zL2NvbXBvbmVudHMvY29uZmlnLWl0ZW1zLWFkbWluL3ByZy1jb25maWctaXRlbXMtYWRtaW4uY29tcG9uZW50Lmh0bWwiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQUEsT0FBTyxFQUFFLFNBQVMsRUFBVSxNQUFNLGVBQWUsQ0FBQztBQUNsRCxPQUFPLEVBQUUsVUFBVSxFQUFFLE1BQU0sZ0JBQWdCLENBQUM7QUFHNUMsT0FBTyxFQUVMLGFBQWEsR0FFZCxNQUFNLDBEQUEwRCxDQUFDO0FBQ2xFLE9BQU8sRUFBRSxRQUFRLEVBQUUsTUFBTSwrQ0FBK0MsQ0FBQztBQUl6RSxPQUFPLEVBQUUseUJBQXlCLEVBQUUsTUFBTSx3REFBd0QsQ0FBQztBQUNuRyxPQUFPLEVBQ0wsZUFBZSxHQUVoQixNQUFNLHVDQUF1QyxDQUFDOzs7Ozs7Ozs7O0FBRy9DOztHQUVHO0FBTUgsTUFBTSxPQUFPLDRCQUE0QjtJQTBFdkM7Ozs7Ozs7T0FPRztJQUNILFlBQ1UseUJBQTRELEVBQzVELG1CQUF3QyxFQUN4QyxjQUE4QixFQUM5QixnQkFBa0MsRUFDbEMsYUFBb0M7UUFKcEMsOEJBQXlCLEdBQXpCLHlCQUF5QixDQUFtQztRQUM1RCx3QkFBbUIsR0FBbkIsbUJBQW1CLENBQXFCO1FBQ3hDLG1CQUFjLEdBQWQsY0FBYyxDQUFnQjtRQUM5QixxQkFBZ0IsR0FBaEIsZ0JBQWdCLENBQWtCO1FBQ2xDLGtCQUFhLEdBQWIsYUFBYSxDQUF1QjtRQTFFOUM7OztXQUdHO1FBQ0ksd0JBQW1CLEdBQWEsUUFBUSxDQUFDLElBQUksQ0FBQztRQUVyRDs7O1dBR0c7UUFDSSxhQUFRLEdBQVksS0FBSyxDQUFDO1FBRWpDOzs7V0FHRztRQUNJLHNCQUFpQixHQUFRLEVBQUUsQ0FBQztRQVFuQzs7V0FFRztRQUNJLGlDQUE0QixHQUFVLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLENBQ3pFLHlCQUF5QixDQUFDLE1BQU0sQ0FDakMsQ0FBQztRQUVGOztXQUVHO1FBQ0ksa0NBQTZCLEdBQ2xDLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLENBQUMseUJBQXlCLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDcEU7Ozs7V0FJRztRQUNLLHlCQUFvQixHQUFRLEVBQUUsQ0FBQztRQUV2Qzs7O1dBR0c7UUFDSyxvQkFBZSxHQUF3QixFQUFFLENBQUM7UUFFbEQ7OztXQUdHO1FBQ0ssZ0NBQTJCLEdBQXdCLEVBQUUsQ0FBQztRQUU5RDs7O1dBR0c7UUFDSyx3QkFBbUIsR0FBd0IsRUFBRSxDQUFDO0lBZ0JuRCxDQUFDO0lBRUo7Ozs7O09BS0c7SUFDSSxLQUFLLENBQUMsUUFBUTtRQUNuQiwrQkFBK0I7UUFDL0IsSUFBSSxDQUFDLGVBQWU7WUFDbEIsTUFBTSxJQUFJLENBQUMseUJBQXlCLENBQUMsNkJBQTZCLEVBQUUsQ0FBQztRQUN2RSxJQUFJLENBQUMsV0FBVyxFQUFFLENBQUM7SUFDckIsQ0FBQztJQUVEOzs7T0FHRztJQUNLLFdBQVc7UUFDakIsOENBQThDO1FBQzlDLE1BQU0sZUFBZSxHQUFhLElBQUksQ0FBQyxlQUFlLENBQUMsR0FBRyxDQUN4RCxDQUFDLFVBQVUsRUFBRSxFQUFFLENBQUMsVUFBVSxDQUFDLElBQUksQ0FDaEMsQ0FBQztRQUVGLFlBQVk7UUFDWixJQUFJLENBQUMsZUFBZSxHQUFHLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLENBQ25ELElBQUksQ0FBQyxjQUFjLENBQUMsa0JBQWtCLENBQUMsZUFBZSxFQUFFLEdBQUcsQ0FBQyxDQUM3RCxDQUFDO0lBQ0osQ0FBQztJQUVEOzs7T0FHRztJQUNJLFlBQVksQ0FBQyxJQUFjO1FBQ2hDLGdDQUFnQztRQUNoQyxJQUFJLENBQUMsbUJBQW1CLEdBQUcsUUFBUSxDQUFDLElBQUksQ0FBQztRQUN6Qyw0QkFBNEI7UUFDNUIsSUFBSSxDQUFDLHNCQUFzQixDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ2xDLHFCQUFxQjtRQUNyQixJQUFJLENBQUMsNEJBQTRCLEdBQUcsSUFBSSxDQUFDLG1CQUFtQixDQUFDLEtBQUssQ0FDaEUsSUFBSSxDQUFDLDJCQUEyQixFQUFFLENBQ25DLENBQUM7UUFDRiw2Q0FBNkM7UUFDN0MsSUFBSSxDQUFDLDBCQUEwQixFQUFFLENBQUM7UUFDbEMsZ0JBQWdCO1FBQ2hCLElBQUksQ0FBQyxZQUFZLEVBQUUsQ0FBQztRQUVwQixlQUFlO1FBQ2YsSUFBSSxDQUFDLFFBQVEsR0FBRyxJQUFJLENBQUM7SUFDdkIsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ssc0JBQXNCLENBQUMsSUFBYztRQUMzQyxNQUFNLGNBQWMsR0FDbEIsNEJBQTRCLENBQUMscUJBQXFCLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFM0QsK0JBQStCO1FBQy9CLE1BQU0sc0JBQXNCLEdBQzFCLGNBQWMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxFQUFFLGNBQWMsQ0FBQyxXQUFXLENBQUMsR0FBRyxDQUFDLENBQUM7WUFDNUQsR0FBRztZQUNILGVBQWUsQ0FBQyxRQUFRLENBQUM7UUFFM0Isc0NBQXNDO1FBQ3RDLElBQUksQ0FBQyxtQkFBbUIsR0FBRyxJQUFJLENBQUMsZUFBZSxDQUFDLE1BQU0sQ0FDcEQsQ0FBQyxVQUFVLEVBQUUsRUFBRSxDQUFDLFVBQVUsQ0FBQyxJQUFJLEtBQUssY0FBYyxDQUNuRCxDQUFDO1FBRUYsK0NBQStDO1FBQy9DLElBQUksQ0FBQywyQkFBMkIsR0FBRyxJQUFJLENBQUMsZUFBZSxDQUFDLE1BQU0sQ0FDNUQsQ0FBQyxVQUFVLEVBQUUsRUFBRSxDQUFDLFVBQVUsQ0FBQyxJQUFJLEtBQUssc0JBQXNCLENBQzNELENBQUM7SUFDSixDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSyxNQUFNLENBQUMscUJBQXFCLENBQUMsSUFBYztRQUNqRCxxQ0FBcUM7UUFDckMsTUFBTSxnQkFBZ0IsR0FBVSxFQUFFLENBQUM7UUFDbkMsSUFBSSxJQUFJLEdBQUcsSUFBSSxDQUFDO1FBQ2hCLElBQUksTUFBTSxHQUFHLElBQUksRUFBRSxNQUFNLENBQUM7UUFDMUIsT0FBTyxJQUFJLElBQUksSUFBSSxFQUFFO1lBQ25CLElBQUksTUFBTSxFQUFFO2dCQUNWLGdCQUFnQixDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUM7Z0JBQ3ZDLE1BQU0sR0FBRyxNQUFNLEVBQUUsTUFBTSxDQUFDO2FBQ3pCO2lCQUFNO2dCQUNMLElBQUksR0FBRyxLQUFLLENBQUM7YUFDZDtTQUNGO1FBQ0QsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsQ0FBQztRQUNuQyxPQUFPLGdCQUFnQixDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBRUQ7OztPQUdHO0lBQ0ssWUFBWTtRQUNsQixnQkFBZ0I7UUFDaEIsSUFBSSxJQUFJLENBQUMsaUJBQWlCLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxlQUFlLENBQUMsZUFBZSxDQUFDLEVBQUU7WUFDekUsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLENBQUMsaUJBQWlCLENBQUMsSUFBSTtpQkFDekMsU0FBUyxDQUFDLENBQUMsRUFBRSxJQUFJLENBQUMsaUJBQWlCLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxHQUFHLENBQUMsQ0FBQztpQkFDMUQsVUFBVSxDQUFDLEdBQUcsRUFBRSxHQUFHLENBQUM7aUJBQ3BCLFdBQVcsRUFBRSxDQUFDO1NBQ2xCO2FBQU07WUFDTCxJQUFJLENBQUMsU0FBUyxHQUFHLElBQUksQ0FBQyxpQkFBaUIsRUFBRSxJQUFJO2lCQUMxQyxVQUFVLENBQUMsR0FBRyxFQUFFLEdBQUcsQ0FBQztpQkFDcEIsV0FBVyxFQUFFLENBQUM7U0FDbEI7SUFDSCxDQUFDO0lBRUQ7OztPQUdHO0lBQ0ssMEJBQTBCO1FBQ2hDLDZDQUE2QztRQUM3QyxNQUFNLHdCQUF3QixHQUFVLEVBQUUsQ0FBQztRQUMzQyxJQUFJLENBQUMsbUJBQW1CLENBQUMsR0FBRyxDQUFDLENBQUMsV0FBVyxFQUFFLEVBQUU7WUFDM0Msd0JBQXdCLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxHQUFHLFdBQVcsQ0FBQyxLQUFLLENBQUM7WUFDL0Qsd0JBQXdCLENBQUMsTUFBTSxDQUFDLEdBQUcsV0FBVyxDQUFDLElBQUksQ0FBQztRQUN0RCxDQUFDLENBQUMsQ0FBQztRQUVILElBQ0UsSUFBSSxDQUFDLDJCQUEyQixJQUFJLElBQUk7WUFDeEMsSUFBSSxDQUFDLDJCQUEyQixDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQzNDO1lBQ0EsSUFBSSxDQUFDLDJCQUEyQixDQUFDLEdBQUcsQ0FBQyxLQUFLLEVBQUUsV0FBVyxFQUFFLEVBQUU7Z0JBQ3pELElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLEVBQUU7b0JBQy9DLHdCQUF3QixDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsR0FBRyxJQUFJLENBQUM7aUJBQ25EO2dCQUNELElBQUksZ0JBQWdCLEdBQUcsSUFBSSxDQUFDLDRCQUE0QixDQUFDLElBQUksQ0FDM0QsQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLEtBQUssQ0FBQyxHQUFHLEtBQUssV0FBVyxDQUFDLElBQUksQ0FDMUMsQ0FBQztnQkFFRixJQUFJLGdCQUFnQixDQUFDLG1CQUFtQixDQUFDLFFBQVEsQ0FBQyxlQUFlLENBQUMsRUFBRTtvQkFDbEUsZ0JBQWdCLENBQUMsV0FBVzt3QkFDMUIsSUFBSSxDQUFDLGdCQUFnQixDQUFDLE9BQU8sQ0FDM0IsZ0JBQWdCLENBQUMsbUJBQW1COzRCQUNsQyxHQUFHOzRCQUNILGdCQUFnQixDQUFDLEdBQUc7NEJBQ3BCLFNBQVM7NEJBQ1QsV0FBVyxDQUFDLEtBQUs7NEJBQ2pCLE9BQU8sQ0FDVixHQUFHLFlBQVksQ0FBQztpQkFDcEI7cUJBQU07b0JBQ0wsZ0JBQWdCLENBQUMsV0FBVyxHQUFHLFdBQVcsQ0FBQyxLQUFLLEdBQUcsWUFBWSxDQUFDO2lCQUNqRTtZQUNILENBQUMsQ0FBQyxDQUFDO1NBQ0o7UUFDRCxJQUFJLENBQUMsaUJBQWlCLEdBQUcsTUFBTSxDQUFDLE1BQU0sQ0FBQyxFQUFFLEVBQUUsd0JBQXdCLENBQUMsQ0FBQztRQUNyRSxJQUFJLENBQUMsb0JBQW9CLEdBQUcsSUFBSSxDQUFDLGFBQWEsQ0FBQyxXQUFXLENBQ3hELElBQUksQ0FBQyxpQkFBaUIsQ0FDdkIsQ0FBQztJQUNKLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssMkJBQTJCO1FBQ2pDLGNBQWM7UUFDZCxJQUFJLFdBQVcsQ0FBQztRQUVoQixJQUNFLElBQUksQ0FBQywyQkFBMkIsSUFBSSxJQUFJO1lBQ3hDLElBQUksQ0FBQywyQkFBMkIsQ0FBQyxNQUFNLElBQUksQ0FBQyxFQUM1QztZQUNBLFdBQVcsR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQUMsR0FBRyxDQUFDLENBQUMsVUFBVSxFQUFFLEVBQUU7Z0JBQ3hELE9BQU8sSUFBSSxDQUFDLGNBQWMsQ0FBQyxrQkFBa0IsQ0FBQyxVQUFVLENBQUMsV0FBVyxDQUFDLENBQUM7WUFDeEUsQ0FBQyxDQUFDLENBQUM7U0FDSjthQUFNO1lBQ0wsV0FBVyxHQUFHLElBQUksQ0FBQywyQkFBMkIsQ0FBQyxHQUFHLENBQUMsQ0FBQyxVQUFVLEVBQUUsRUFBRTtnQkFDaEUsT0FBTyxJQUFJLENBQUMsY0FBYyxDQUFDLGtCQUFrQixDQUFDLFVBQVUsQ0FBQyxXQUFXLENBQUMsQ0FBQztZQUN4RSxDQUFDLENBQUMsQ0FBQztZQUNILElBQUksSUFBSSxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsUUFBUSxDQUFDLGVBQWUsQ0FBQyxRQUFRLENBQUMsRUFBRTtnQkFDNUQsV0FBVyxDQUFDLE9BQU8sQ0FBQyxDQUFDLEdBQUcsRUFBRSxFQUFFO29CQUMxQixHQUFHLENBQUMsVUFBVSxHQUFHLFVBQVUsQ0FBQyxRQUFRLENBQUM7Z0JBQ3ZDLENBQUMsQ0FBQyxDQUFDO2FBQ0o7U0FDRjtRQUNELE9BQU8sV0FBVyxDQUFDO0lBQ3JCLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksS0FBSyxDQUFDLG1CQUFtQixDQUM5QixLQUE4QjtRQUU5QixRQUFRLEtBQUssQ0FBQyxNQUFNLEVBQUU7WUFDcEIsS0FBSyxhQUFhLENBQUMsSUFBSTtnQkFDckIsTUFBTSxJQUFJLENBQUMsZUFBZSxDQUFDLEtBQUssQ0FBQyxDQUFDO2dCQUNsQyxNQUFNO1lBRVIsS0FBSyxhQUFhLENBQUMsTUFBTTtnQkFDdkIsSUFBSSxDQUFDLG1CQUFtQixHQUFHLFFBQVEsQ0FBQyxJQUFJLENBQUM7Z0JBQ3pDLElBQUksQ0FBQyxpQkFBaUIsR0FBRyxJQUFJLENBQUMsYUFBYSxDQUFDLFdBQVcsQ0FDckQsSUFBSSxDQUFDLG9CQUFvQixDQUMxQixDQUFDO2dCQUVGLE1BQU07WUFFUixLQUFLLGFBQWEsQ0FBQyxJQUFJO2dCQUNyQixJQUFJLENBQUMsbUJBQW1CLEdBQUcsUUFBUSxDQUFDLElBQUksQ0FBQztnQkFDekMsTUFBTTtZQUNSO2dCQUNFLE1BQU07U0FDVDtJQUNILENBQUM7SUFFRDs7Ozs7T0FLRztJQUNLLEtBQUssQ0FBQyxlQUFlLENBQUMsS0FBOEI7UUFDMUQsSUFBSSxDQUFDLG1CQUFtQixHQUFHLFFBQVEsQ0FBQyxJQUFJLENBQUM7UUFFekMsSUFBSTtZQUNGLElBQUksQ0FBQyxRQUFRLEdBQUcsS0FBSyxDQUFDO1lBQ3RCLE1BQU0sSUFBSSxDQUFDLHlCQUF5QixDQUFDLDZCQUE2QixDQUNoRSxLQUFLLENBQUMsVUFBVSxDQUNqQixDQUFDO1lBQ0YsSUFBSSxDQUFDLGVBQWUsR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQUMsS0FBSyxDQUNuRCxJQUFJLENBQUMseUJBQXlCLENBQUMsMEJBQTBCLEVBQUUsQ0FDNUQsQ0FBQztZQUVGLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQztZQUNuQixJQUFJLElBQUksQ0FBQyxZQUFZLEVBQUU7Z0JBQ3JCLElBQUksQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDO2FBQ3RDO1lBQ0QsSUFBSSxDQUFDLFFBQVEsR0FBRyxJQUFJLENBQUM7U0FDdEI7UUFBQyxPQUFPLENBQUMsRUFBRSxHQUFFO0lBQ2hCLENBQUM7O3lIQWhWVSw0QkFBNEI7NkdBQTVCLDRCQUE0Qiw4REM1QnpDLG83QkFlQTsyRkRhYSw0QkFBNEI7a0JBTHhDLFNBQVM7K0JBQ0Usd0JBQXdCIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQ29tcG9uZW50LCBPbkluaXQgfSBmcm9tICdAYW5ndWxhci9jb3JlJztcclxuaW1wb3J0IHsgVmFsaWRhdG9ycyB9IGZyb20gJ0Bhbmd1bGFyL2Zvcm1zJztcclxuaW1wb3J0IHsgVHJhbnNsYXRlU2VydmljZSB9IGZyb20gJ0BuZ3gtdHJhbnNsYXRlL2NvcmUnO1xyXG5pbXBvcnQgeyBUcmVlTm9kZSB9IGZyb20gJ3ByaW1lbmcvYXBpJztcclxuaW1wb3J0IHtcclxuICBCYXNlQWN0aW9uLFxyXG4gIEJhc2VBY3Rpb25LZXksXHJcbiAgRHluYW1pY0Zvcm1BY3Rpb25PdXRwdXQsXHJcbn0gZnJvbSAnLi4vLi4vLi4vY29yZS9jb21wb25lbnRzL2R5bmFtaWMtZm9ybS9tb2RlbHMvYmFzZS1hY3Rpb24nO1xyXG5pbXBvcnQgeyBWaWV3TW9kZSB9IGZyb20gJy4uLy4uLy4uL2NvcmUvbW9kZWxzL2NvbnN0YW50cy92aWV3LW1vZGUuZW51bSc7XHJcbmltcG9ydCB7IEFycmF5VXRpbGl0eVNlcnZpY2UgfSBmcm9tICcuLi8uLi8uLi9jb3JlL3NlcnZpY2VzL3V0aWxpdHkvYXJyYXktdXRpbGl0eS5zZXJ2aWNlJztcclxuaW1wb3J0IHsgT2JqZWN0c1V0aWxpdHlTZXJ2aWNlIH0gZnJvbSAnLi4vLi4vLi4vY29yZS9zZXJ2aWNlcy91dGlsaXR5L29iamVjdHMtdXRpbGl0eS5zZXJ2aWNlJztcclxuaW1wb3J0IHsgVXRpbGl0eVNlcnZpY2UgfSBmcm9tICcuLi8uLi8uLi9jb3JlL3NlcnZpY2VzL3V0aWxpdHkvdXRpbGl0eS5zZXJ2aWNlJztcclxuaW1wb3J0IHsgQ09ORklHX0lURU1TX0RZTkFNSUNfRk9STSB9IGZyb20gJy4uLy4uL2R5bmFtaWMtZm9ybS1zdHJ1Y3R1cmUvY29uZmlnLWl0ZW1zLWR5bmFtaWMtZm9ybSc7XHJcbmltcG9ydCB7XHJcbiAgQ29uZmlnSXRlbXNQYXRoLFxyXG4gIENvbmZpZ3VyYXRpb25JdGVtLFxyXG59IGZyb20gJy4uLy4uL21vZGVscy9jb25maWd1cmF0aW9uLWl0ZW0ubW9kZWwnO1xyXG5pbXBvcnQgeyBBYnN0cmFjdENvbmZpZ3VyYXRpb25JdGVtc1NlcnZpY2UgfSBmcm9tICcuLi8uLi9zZXJ2aWNlcy9jb25maWd1cmF0aW9uLWl0ZW1zL2Fic3RyYWN0LWNvbmZpZ3VyYXRpb24taXRlbXMuc2VydmljZSc7XHJcblxyXG4vKipcclxuICogQ29uZmlnIGl0ZW1zIGFkbWluIGNvbXBvbmVudFxyXG4gKi9cclxuQENvbXBvbmVudCh7XHJcbiAgc2VsZWN0b3I6ICdwcmctY29uZmlnLWl0ZW1zLWFkbWluJyxcclxuICB0ZW1wbGF0ZVVybDogJy4vcHJnLWNvbmZpZy1pdGVtcy1hZG1pbi5jb21wb25lbnQuaHRtbCcsXHJcbiAgc3R5bGVVcmxzOiBbJy4vcHJnLWNvbmZpZy1pdGVtcy1hZG1pbi5jb21wb25lbnQuc2NzcyddLFxyXG59KVxyXG5leHBvcnQgY2xhc3MgUHJnQ29uZmlnSXRlbXNBZG1pbkNvbXBvbmVudCBpbXBsZW1lbnRzIE9uSW5pdCB7XHJcbiAgLyoqXHJcbiAgICogVGhlIHN0cnVjdHVyZSBvZiB0aGUgdHJlZSBpbmNsdWRpbmcgcGF0aHMgYW5kIGRhdGFcclxuICAgKiBAdHlwZSB7VHJlZU5vZGVbXX1cclxuICAgKi9cclxuICBwdWJsaWMgY29uZmlnSXRlbXNUcmVlOiBUcmVlTm9kZVtdO1xyXG5cclxuICAvKipcclxuICAgKiBUaGUgY3VycmVudCBzZWxlY3RlZCBub2RlIG9mIHRoZSB0cmVlXHJcbiAgICogQHR5cGUge1RyZWVOb2RlfVxyXG4gICAqL1xyXG4gIHB1YmxpYyBzZWxlY3RlZE5vZGU6IFRyZWVOb2RlO1xyXG5cclxuICAvKipcclxuICAgKiBUaGUgdmlldyBtb2RlIHR5cGUgb2YgdGhlIGZvcm0uIERlZmF1bHQgaXMgXCJyZWFkXCJcclxuICAgKiBAdHlwZSB7Vmlld01vZGV9XHJcbiAgICovXHJcbiAgcHVibGljIGNvbmZpZ0l0ZW1zVmlld01vZGU6IFZpZXdNb2RlID0gVmlld01vZGUuUmVhZDtcclxuXHJcbiAgLyoqXHJcbiAgICogQSBib29sZWFuIHByb3BlcnR5IG9mIHRoZSBjbGFzcyB0byBzaG93L2hpZGUgZm9ybS4gRGVmYXVsdCBpcyBmYWxzZShoaWRlKVxyXG4gICAqIEB0eXBlIHtib29sZWFufVxyXG4gICAqL1xyXG4gIHB1YmxpYyBpbml0Rm9ybTogYm9vbGVhbiA9IGZhbHNlO1xyXG5cclxuICAvKipcclxuICAgKiBBbiBvYmplY3Qgd2l0aCB0aGUgZGF0YSB0byBidWlsZCBmb3JtICwga2V5cyBhbmQgdmFsdWVzXHJcbiAgICogQHR5cGUge2FueX1cclxuICAgKi9cclxuICBwdWJsaWMgZW50aXR5Q29uZmlnSXRlbXM6IGFueSA9IHt9O1xyXG5cclxuICAvKipcclxuICAgKiBUaGUgdGl0bGUgb2YgdGhlIGZvcm0gb2YgY29uZmlndXJhdGlvbiBpdGVtXHJcbiAgICogQHR5cGUge3N0cmluZ31cclxuICAgKi9cclxuICBwdWJsaWMgdGl0bGVGb3JtOiBzdHJpbmc7XHJcblxyXG4gIC8qKlxyXG4gICAqIFRoZSBkeW5hbWljIGZvcm0gZmllbGRzIHRvIGJ1aWxkIGR5bmFtaWMgZm9ybSBjb25maWcgaXRlbXMgaW4gdGVtcGxhdGVcclxuICAgKi9cclxuICBwdWJsaWMgY29uZmlnSXRlbXNEeW5hbWljRm9ybUZpZWxkczogYW55W10gPSB0aGlzLmFycmF5VXRpbGl0eVNlcnZpY2UuY2xvbmUoXHJcbiAgICBDT05GSUdfSVRFTVNfRFlOQU1JQ19GT1JNLmZpZWxkc1xyXG4gICk7XHJcblxyXG4gIC8qKlxyXG4gICAqIFRoZSBkeW5hbWljIGZvcm0gYWN0aW9ucyB0byBidWlsZCBkeW5hbWljIGZvcm0gY29uZmlnIGl0ZW1zIGluIHRlbXBsYXRlXHJcbiAgICovXHJcbiAgcHVibGljIGNvbmZpZ0l0ZW1zRHluYW1pY0Zvcm1BY3Rpb25zOiBCYXNlQWN0aW9uW10gPVxyXG4gICAgdGhpcy5hcnJheVV0aWxpdHlTZXJ2aWNlLmNsb25lKENPTkZJR19JVEVNU19EWU5BTUlDX0ZPUk0uYWN0aW9ucyk7XHJcbiAgLyoqXHJcbiAgICogQSBjb3B5IG9mIGVudGl0eUNvbmZpZ0l0ZW1zIG9iamVjdFxyXG4gICAqIEB0eXBlIHthbnl9XHJcbiAgICogQHByaXZhdGVcclxuICAgKi9cclxuICBwcml2YXRlIGVudGl0eUNvbmZpZ0l0ZW1Db3B5OiBhbnkgPSB7fTtcclxuXHJcbiAgLyoqXHJcbiAgICogVGhlIGxpc3Qgb2YgY29uZmlndXJhdGlvbiBpdGVtc1xyXG4gICAqIEB0eXBlIHtDb25maWd1cmF0aW9uSXRlbVtdfVxyXG4gICAqL1xyXG4gIHByaXZhdGUgY29uZmlnSXRlbXNMaXN0OiBDb25maWd1cmF0aW9uSXRlbVtdID0gW107XHJcblxyXG4gIC8qKlxyXG4gICAqIFRoZSBjb25maWd1cmF0aW9uIGl0ZW1zIGRlZmF1bHRzIHNlbGVjdGVkIG9uIHRyZWVcclxuICAgKiBAdHlwZSB7Q29uZmlndXJhdGlvbkl0ZW1bXX1cclxuICAgKi9cclxuICBwcml2YXRlIGNvbmZpZ0l0ZW1zU2VsZWN0ZWREZWZhdWx0czogQ29uZmlndXJhdGlvbkl0ZW1bXSA9IFtdO1xyXG5cclxuICAvKipcclxuICAgKiBUaGUgY29uZmlndXJhdGlvbiBpdGVtcyBzZWxlY3RlZCBvbiB0cmVlXHJcbiAgICogQHR5cGUge0NvbmZpZ3VyYXRpb25JdGVtW119XHJcbiAgICovXHJcbiAgcHJpdmF0ZSBjb25maWdJdGVtc1NlbGVjdGVkOiBDb25maWd1cmF0aW9uSXRlbVtdID0gW107XHJcblxyXG4gIC8qKlxyXG4gICAqIENvbnN0cnVjdG9yXHJcbiAgICogQHBhcmFtIHtBYnN0cmFjdENvbmZpZ3VyYXRpb25JdGVtc1NlcnZpY2V9IGNvbmZpZ3VyYXRpb25JdGVtc1NlcnZpY2VcclxuICAgKiBAcGFyYW0ge0FycmF5VXRpbGl0eVNlcnZpY2V9IGFycmF5VXRpbGl0eVNlcnZpY2VcclxuICAgKiBAcGFyYW0ge1V0aWxpdHlTZXJ2aWNlfSB1dGlsaXR5U2VydmljZVxyXG4gICAqIEBwYXJhbSB7VHJhbnNsYXRlU2VydmljZX0gdHJhbnNsYXRlU2VydmljZVxyXG4gICAqIEBwYXJhbSB7T2JqZWN0c1V0aWxpdHlTZXJ2aWNlfSBvYmplY3RVdGlsaXR5XHJcbiAgICovXHJcbiAgY29uc3RydWN0b3IoXHJcbiAgICBwcml2YXRlIGNvbmZpZ3VyYXRpb25JdGVtc1NlcnZpY2U6IEFic3RyYWN0Q29uZmlndXJhdGlvbkl0ZW1zU2VydmljZSxcclxuICAgIHByaXZhdGUgYXJyYXlVdGlsaXR5U2VydmljZTogQXJyYXlVdGlsaXR5U2VydmljZSxcclxuICAgIHByaXZhdGUgdXRpbGl0eVNlcnZpY2U6IFV0aWxpdHlTZXJ2aWNlLFxyXG4gICAgcHJpdmF0ZSB0cmFuc2xhdGVTZXJ2aWNlOiBUcmFuc2xhdGVTZXJ2aWNlLFxyXG4gICAgcHJpdmF0ZSBvYmplY3RVdGlsaXR5OiBPYmplY3RzVXRpbGl0eVNlcnZpY2VcclxuICApIHt9XHJcblxyXG4gIC8qKlxyXG4gICAqIG5nT25Jbml0XHJcbiAgICpcclxuICAgKiBHZXQgYWxsIGNvbmZpZ3VyYXRpb24gaXRlbVxyXG4gICAqIEByZXR1cm5zIHtQcm9taXNlPHZvaWQ+fVxyXG4gICAqL1xyXG4gIHB1YmxpYyBhc3luYyBuZ09uSW5pdCgpOiBQcm9taXNlPHZvaWQ+IHtcclxuICAgIC8vZmV0Y2ggYWxsIGNvbmZpZ3VyYXRpb24gaXRlbXNcclxuICAgIHRoaXMuY29uZmlnSXRlbXNMaXN0ID1cclxuICAgICAgYXdhaXQgdGhpcy5jb25maWd1cmF0aW9uSXRlbXNTZXJ2aWNlLmdldEFsbENvbmZpZ3VyYXRpb25JdGVtc0FzeW5jKCk7XHJcbiAgICB0aGlzLnNldFRyZWVEYXRhKCk7XHJcbiAgfVxyXG5cclxuICAvKipcclxuICAgKiBUaGlzIGZ1bmN0aW9uIGNhbGwgcmVhcnJhbmdlIGFsbCBjb25maWcgaXRlbSBwYXRocyBhbmQgYnVpbGQgdHJlZSBtZW51XHJcbiAgICogQHByaXZhdGVcclxuICAgKi9cclxuICBwcml2YXRlIHNldFRyZWVEYXRhKCk6IHZvaWQge1xyXG4gICAgLy9nZXQgb25seSB0aGUgcGF0aCBvZiBhbGwgY29uZmlndXJhdGlvbiBpdGVtc1xyXG4gICAgY29uc3QgY29uZmlnSXRlbXNQYXRoOiBzdHJpbmdbXSA9IHRoaXMuY29uZmlnSXRlbXNMaXN0Lm1hcChcclxuICAgICAgKGNvbmZpZ0l0ZW0pID0+IGNvbmZpZ0l0ZW0ucGF0aFxyXG4gICAgKTtcclxuXHJcbiAgICAvL0J1aWxkIHRyZWVcclxuICAgIHRoaXMuY29uZmlnSXRlbXNUcmVlID0gdGhpcy5hcnJheVV0aWxpdHlTZXJ2aWNlLmNsb25lKFxyXG4gICAgICB0aGlzLnV0aWxpdHlTZXJ2aWNlLmFycmF5T2ZQYXRoc1RvVHJlZShjb25maWdJdGVtc1BhdGgsICc6JylcclxuICAgICk7XHJcbiAgfVxyXG5cclxuICAvKipcclxuICAgKiBUaGlzIGZ1bmN0aW9uIGlzIHJlc3BvbnNpYmxlIHRvZSBleGVjdXRlIGFjdGlvbnMgd2hlbmV2ZXIgYSBub2RlIGlzIHNlbGVjdGVkIG9uIHRyZWVcclxuICAgKiBAcGFyYW0ge1RyZWVOb2RlfSBub2RlXHJcbiAgICovXHJcbiAgcHVibGljIG9uU2VsZWN0SXRlbShub2RlOiBUcmVlTm9kZSk6IHZvaWQge1xyXG4gICAgLy8gUmV0dXJuIFZpZXcgbW9kZSBiYWNrIHRvIHJlYWRcclxuICAgIHRoaXMuY29uZmlnSXRlbXNWaWV3TW9kZSA9IFZpZXdNb2RlLlJlYWQ7XHJcbiAgICAvLyBTZXQgY29uZmlnIGl0ZW1zIHNlbGVjdGVkXHJcbiAgICB0aGlzLnNldENvbmZpZ0l0ZW1zU2VsZWN0ZWQobm9kZSk7XHJcbiAgICAvL0R5bmFtaWMgRm9ybSBGaWVsZHNcclxuICAgIHRoaXMuY29uZmlnSXRlbXNEeW5hbWljRm9ybUZpZWxkcyA9IHRoaXMuYXJyYXlVdGlsaXR5U2VydmljZS5jbG9uZShcclxuICAgICAgdGhpcy5nZXRBbmRQYXJzZUd1aVNldHRpbmdzSXRlbXMoKVxyXG4gICAgKTtcclxuICAgIC8vIEFzc2lnbiBjb25maWcgaXRlbSBkYXRhIHRvIGZvcm0gLSBlbnRpdGllc1xyXG4gICAgdGhpcy5hc3NpZ25Db25maWdJdGVtRGF0YVRvRm9ybSgpO1xyXG4gICAgLy8gVGl0bGUgb2YgRm9ybVxyXG4gICAgdGhpcy5zZXRUaXRsZUZvcm0oKTtcclxuXHJcbiAgICAvL0luaXRpYXRlIEZvcm1cclxuICAgIHRoaXMuaW5pdEZvcm0gPSB0cnVlO1xyXG4gIH1cclxuXHJcbiAgLyoqXHJcbiAgICogVGhpcyBmdW5jdGlvbiBpcyByZXNwb25zaWJsZSB0byBnZXQgdGhlIGNvbXBsZXRlIHBhdGggb2YgaXRlbSBzZWxlY3RlZCBhbmQgdGhlaXIgZGVmYXVsdCBwYXRoIGlmXHJcbiAgICogdGhleSBleGlzdC4gVGhlbiBpdCB3aWxsIGdldCB0aGUgZGF0YSBvZiBjb25maWcgaXRlbXMgc2VsZWN0ZWRcclxuICAgKiBAcGFyYW0ge1RyZWVOb2RlfSBub2RlXHJcbiAgICogQHByaXZhdGVcclxuICAgKi9cclxuICBwcml2YXRlIHNldENvbmZpZ0l0ZW1zU2VsZWN0ZWQobm9kZTogVHJlZU5vZGUpOiB2b2lkIHtcclxuICAgIGNvbnN0IHBhdGhDb25maWdJdGVtID1cclxuICAgICAgUHJnQ29uZmlnSXRlbXNBZG1pbkNvbXBvbmVudC5nZXRDb21wbGV0ZVBhdGhPZkl0ZW0obm9kZSk7XHJcblxyXG4gICAgLy8gUGF0aCBvZiBkZWZhdWx0IGNvbmZpZy1pdGVtc1xyXG4gICAgY29uc3QgcGF0aENvbmZpZ0l0ZW1EZWZhdWx0cyA9XHJcbiAgICAgIHBhdGhDb25maWdJdGVtLnN1YnN0cmluZygwLCBwYXRoQ29uZmlnSXRlbS5sYXN0SW5kZXhPZignOicpKSArXHJcbiAgICAgICc6JyArXHJcbiAgICAgIENvbmZpZ0l0ZW1zUGF0aC5EZWZhdWx0cztcclxuXHJcbiAgICAvLyBDb25maWctaXRlbXMgb2YgdGhlIHNlbGVjdGFibGUgbm9kZVxyXG4gICAgdGhpcy5jb25maWdJdGVtc1NlbGVjdGVkID0gdGhpcy5jb25maWdJdGVtc0xpc3QuZmlsdGVyKFxyXG4gICAgICAoY29uZmlnSXRlbSkgPT4gY29uZmlnSXRlbS5wYXRoID09PSBwYXRoQ29uZmlnSXRlbVxyXG4gICAgKTtcclxuXHJcbiAgICAvLyBDb25maWctaXRlbXMgRGVmYXVsdHMgb2YgdGhlIHNlbGVjdGFibGUgbm9kZVxyXG4gICAgdGhpcy5jb25maWdJdGVtc1NlbGVjdGVkRGVmYXVsdHMgPSB0aGlzLmNvbmZpZ0l0ZW1zTGlzdC5maWx0ZXIoXHJcbiAgICAgIChjb25maWdJdGVtKSA9PiBjb25maWdJdGVtLnBhdGggPT09IHBhdGhDb25maWdJdGVtRGVmYXVsdHNcclxuICAgICk7XHJcbiAgfVxyXG5cclxuICAvKipcclxuICAgKiBUaGlzIGZ1bmN0aW9uIGlzIHJlc3BvbnNpYmxlIHRvIGdldCB0aGUgY29tcGxldGUgcGF0aCBvZiB0aGUgaXRlbSBpbiB0cmVlXHJcbiAgICogQHBhcmFtIHtUcmVlTm9kZX0gbm9kZVxyXG4gICAqIEByZXR1cm5zIHtzdHJpbmd9XHJcbiAgICogQHByaXZhdGVcclxuICAgKi9cclxuICBwcml2YXRlIHN0YXRpYyBnZXRDb21wbGV0ZVBhdGhPZkl0ZW0obm9kZTogVHJlZU5vZGUpOiBzdHJpbmcge1xyXG4gICAgLy8gRmluZCB0aGUgY29tcGxldGUgcGF0aCBvZiB0aGUgaXRlbVxyXG4gICAgY29uc3QgY29tcGxldGVQYXRoSXRlbTogYW55W10gPSBbXTtcclxuICAgIGxldCBmbGFnID0gdHJ1ZTtcclxuICAgIGxldCBwYXJlbnQgPSBub2RlPy5wYXJlbnQ7XHJcbiAgICB3aGlsZSAoZmxhZyA9PSB0cnVlKSB7XHJcbiAgICAgIGlmIChwYXJlbnQpIHtcclxuICAgICAgICBjb21wbGV0ZVBhdGhJdGVtLnVuc2hpZnQocGFyZW50LmxhYmVsKTtcclxuICAgICAgICBwYXJlbnQgPSBwYXJlbnQ/LnBhcmVudDtcclxuICAgICAgfSBlbHNlIHtcclxuICAgICAgICBmbGFnID0gZmFsc2U7XHJcbiAgICAgIH1cclxuICAgIH1cclxuICAgIGNvbXBsZXRlUGF0aEl0ZW0ucHVzaChub2RlPy5sYWJlbCk7XHJcbiAgICByZXR1cm4gY29tcGxldGVQYXRoSXRlbS5qb2luKCc6Jyk7XHJcbiAgfVxyXG5cclxuICAvKipcclxuICAgKiBUaGlzIGZ1bmN0aW9uIGl0IHdpbGwgc2V0IHRoZSB0aXRsZSBvZiB0aGUgZm9ybVxyXG4gICAqIEBwcml2YXRlXHJcbiAgICovXHJcbiAgcHJpdmF0ZSBzZXRUaXRsZUZvcm0oKTogdm9pZCB7XHJcbiAgICAvLyBUaXRsZSBvZiBGb3JtXHJcbiAgICBpZiAodGhpcy5lbnRpdHlDb25maWdJdGVtcy5wYXRoLmluY2x1ZGVzKENvbmZpZ0l0ZW1zUGF0aC5Vc2VyUHJlZmVyZW5jZXMpKSB7XHJcbiAgICAgIHRoaXMudGl0bGVGb3JtID0gdGhpcy5lbnRpdHlDb25maWdJdGVtcy5wYXRoXHJcbiAgICAgICAgLnN1YnN0cmluZygwLCB0aGlzLmVudGl0eUNvbmZpZ0l0ZW1zLnBhdGgubGFzdEluZGV4T2YoJzonKSlcclxuICAgICAgICAucmVwbGFjZUFsbCgnOicsICcuJylcclxuICAgICAgICAudG9Mb3dlckNhc2UoKTtcclxuICAgIH0gZWxzZSB7XHJcbiAgICAgIHRoaXMudGl0bGVGb3JtID0gdGhpcy5lbnRpdHlDb25maWdJdGVtcz8ucGF0aFxyXG4gICAgICAgIC5yZXBsYWNlQWxsKCc6JywgJy4nKVxyXG4gICAgICAgIC50b0xvd2VyQ2FzZSgpO1xyXG4gICAgfVxyXG4gIH1cclxuXHJcbiAgLyoqXHJcbiAgICogVGhpcyBmdW5jdGlvbiB3aWxsIGFzc2lnbiBjb25maWcgaXRlbXMgZGF0YSB0byBmb3JtXHJcbiAgICogQHByaXZhdGVcclxuICAgKi9cclxuICBwcml2YXRlIGFzc2lnbkNvbmZpZ0l0ZW1EYXRhVG9Gb3JtKCk6IHZvaWQge1xyXG4gICAgLy8gQXNzaWduIGNvbmZpZyBpdGVtIGRhdGEgdG8gZm9ybSAtIGVudGl0aWVzXHJcbiAgICBjb25zdCBtYXBDb25maWdJdGVtc0ludG9FbnRpdHk6IGFueVtdID0gW107XHJcbiAgICB0aGlzLmNvbmZpZ0l0ZW1zU2VsZWN0ZWQubWFwKChjb25maWdJdGVtcykgPT4ge1xyXG4gICAgICBtYXBDb25maWdJdGVtc0ludG9FbnRpdHlbY29uZmlnSXRlbXMubmFtZV0gPSBjb25maWdJdGVtcy52YWx1ZTtcclxuICAgICAgbWFwQ29uZmlnSXRlbXNJbnRvRW50aXR5WydwYXRoJ10gPSBjb25maWdJdGVtcy5wYXRoO1xyXG4gICAgfSk7XHJcblxyXG4gICAgaWYgKFxyXG4gICAgICB0aGlzLmNvbmZpZ0l0ZW1zU2VsZWN0ZWREZWZhdWx0cyAhPSBudWxsICYmXHJcbiAgICAgIHRoaXMuY29uZmlnSXRlbXNTZWxlY3RlZERlZmF1bHRzLmxlbmd0aCA+IDBcclxuICAgICkge1xyXG4gICAgICB0aGlzLmNvbmZpZ0l0ZW1zU2VsZWN0ZWREZWZhdWx0cy5tYXAoYXN5bmMgKGNvbmZpZ0l0ZW1zKSA9PiB7XHJcbiAgICAgICAgaWYgKCFtYXBDb25maWdJdGVtc0ludG9FbnRpdHlbY29uZmlnSXRlbXMubmFtZV0pIHtcclxuICAgICAgICAgIG1hcENvbmZpZ0l0ZW1zSW50b0VudGl0eVtjb25maWdJdGVtcy5uYW1lXSA9IG51bGw7XHJcbiAgICAgICAgfVxyXG4gICAgICAgIGxldCBmaWVsZEd1aVNldHRpbmdzID0gdGhpcy5jb25maWdJdGVtc0R5bmFtaWNGb3JtRmllbGRzLmZpbmQoXHJcbiAgICAgICAgICAodmFsdWUpID0+IHZhbHVlLmtleSA9PT0gY29uZmlnSXRlbXMubmFtZVxyXG4gICAgICAgICk7XHJcblxyXG4gICAgICAgIGlmIChmaWVsZEd1aVNldHRpbmdzLmJhc2VQYXRoVHJhbnNsYXRpb24uaW5jbHVkZXMoJ2xvb2t1cC10YWJsZXMnKSkge1xyXG4gICAgICAgICAgZmllbGRHdWlTZXR0aW5ncy5wbGFjZWhvbGRlciA9XHJcbiAgICAgICAgICAgIHRoaXMudHJhbnNsYXRlU2VydmljZS5pbnN0YW50KFxyXG4gICAgICAgICAgICAgIGZpZWxkR3VpU2V0dGluZ3MuYmFzZVBhdGhUcmFuc2xhdGlvbiArXHJcbiAgICAgICAgICAgICAgICAnLicgK1xyXG4gICAgICAgICAgICAgICAgZmllbGRHdWlTZXR0aW5ncy5rZXkgK1xyXG4gICAgICAgICAgICAgICAgJy5pdGVtcy4nICtcclxuICAgICAgICAgICAgICAgIGNvbmZpZ0l0ZW1zLnZhbHVlICtcclxuICAgICAgICAgICAgICAgICcubmFtZSdcclxuICAgICAgICAgICAgKSArICcgKGRlZmF1bHQpJztcclxuICAgICAgICB9IGVsc2Uge1xyXG4gICAgICAgICAgZmllbGRHdWlTZXR0aW5ncy5wbGFjZWhvbGRlciA9IGNvbmZpZ0l0ZW1zLnZhbHVlICsgJyAoZGVmYXVsdCknO1xyXG4gICAgICAgIH1cclxuICAgICAgfSk7XHJcbiAgICB9XHJcbiAgICB0aGlzLmVudGl0eUNvbmZpZ0l0ZW1zID0gT2JqZWN0LmFzc2lnbih7fSwgbWFwQ29uZmlnSXRlbXNJbnRvRW50aXR5KTtcclxuICAgIHRoaXMuZW50aXR5Q29uZmlnSXRlbUNvcHkgPSB0aGlzLm9iamVjdFV0aWxpdHkuY2xvbmVPYmplY3QoXHJcbiAgICAgIHRoaXMuZW50aXR5Q29uZmlnSXRlbXNcclxuICAgICk7XHJcbiAgfVxyXG5cclxuICAvKipcclxuICAgKiBUaGlzIGZ1bmN0aW9uIGlzIHJlc3BvbnNpYmxlIHRvIGdldCB0aGUgZ3VpIHNldHRpbmdzIG9mIHRoZSBpdGVtIHNlbGVjdGVkIGFuZCBwYXJzZSBpdFxyXG4gICAqIEByZXR1cm5zIHthbnl9XHJcbiAgICogQHByaXZhdGVcclxuICAgKi9cclxuICBwcml2YXRlIGdldEFuZFBhcnNlR3VpU2V0dGluZ3NJdGVtcygpOiBhbnkge1xyXG4gICAgLy9HdWkgU2V0dGluZ3NcclxuICAgIGxldCBndWlTZXR0aW5ncztcclxuXHJcbiAgICBpZiAoXHJcbiAgICAgIHRoaXMuY29uZmlnSXRlbXNTZWxlY3RlZERlZmF1bHRzID09IG51bGwgfHxcclxuICAgICAgdGhpcy5jb25maWdJdGVtc1NlbGVjdGVkRGVmYXVsdHMubGVuZ3RoID09IDBcclxuICAgICkge1xyXG4gICAgICBndWlTZXR0aW5ncyA9IHRoaXMuY29uZmlnSXRlbXNTZWxlY3RlZC5tYXAoKGNvbmZpZ0l0ZW0pID0+IHtcclxuICAgICAgICByZXR1cm4gdGhpcy51dGlsaXR5U2VydmljZS5ndWlTZXR0aW5nVG9PYmplY3QoY29uZmlnSXRlbS5ndWlTZXR0aW5ncyk7XHJcbiAgICAgIH0pO1xyXG4gICAgfSBlbHNlIHtcclxuICAgICAgZ3VpU2V0dGluZ3MgPSB0aGlzLmNvbmZpZ0l0ZW1zU2VsZWN0ZWREZWZhdWx0cy5tYXAoKGNvbmZpZ0l0ZW0pID0+IHtcclxuICAgICAgICByZXR1cm4gdGhpcy51dGlsaXR5U2VydmljZS5ndWlTZXR0aW5nVG9PYmplY3QoY29uZmlnSXRlbS5ndWlTZXR0aW5ncyk7XHJcbiAgICAgIH0pO1xyXG4gICAgICBpZiAodGhpcy5zZWxlY3RlZE5vZGUua2V5LmluY2x1ZGVzKENvbmZpZ0l0ZW1zUGF0aC5EZWZhdWx0cykpIHtcclxuICAgICAgICBndWlTZXR0aW5ncy5mb3JFYWNoKChndWkpID0+IHtcclxuICAgICAgICAgIGd1aS52YWxpZGF0b3JzID0gVmFsaWRhdG9ycy5yZXF1aXJlZDtcclxuICAgICAgICB9KTtcclxuICAgICAgfVxyXG4gICAgfVxyXG4gICAgcmV0dXJuIGd1aVNldHRpbmdzO1xyXG4gIH1cclxuXHJcbiAgLyoqXHJcbiAgICogVGhpcyBmdW5jdGlvbiBpcyByZXNwb25zaWJsZSB0byBoYW5kbGUgYWN0aW9ucyBmcm9tIHRoZSBkeW5hbWljIGNvbmZpZyBpdGVtcyBmb3JtXHJcbiAgICogQHBhcmFtIHtEeW5hbWljRm9ybUFjdGlvbk91dHB1dH0gZXZlbnRcclxuICAgKiBAcmV0dXJucyB7UHJvbWlzZTx2b2lkPn1cclxuICAgKi9cclxuICBwdWJsaWMgYXN5bmMgb25BY3Rpb25Db25maWdJdGVtcyhcclxuICAgIGV2ZW50OiBEeW5hbWljRm9ybUFjdGlvbk91dHB1dFxyXG4gICk6IFByb21pc2U8dm9pZD4ge1xyXG4gICAgc3dpdGNoIChldmVudC5hY3Rpb24pIHtcclxuICAgICAgY2FzZSBCYXNlQWN0aW9uS2V5LlNhdmU6XHJcbiAgICAgICAgYXdhaXQgdGhpcy5jb25maWdJdGVtc1NhdmUoZXZlbnQpO1xyXG4gICAgICAgIGJyZWFrO1xyXG5cclxuICAgICAgY2FzZSBCYXNlQWN0aW9uS2V5LkNhbmNlbDpcclxuICAgICAgICB0aGlzLmNvbmZpZ0l0ZW1zVmlld01vZGUgPSBWaWV3TW9kZS5SZWFkO1xyXG4gICAgICAgIHRoaXMuZW50aXR5Q29uZmlnSXRlbXMgPSB0aGlzLm9iamVjdFV0aWxpdHkuY2xvbmVPYmplY3QoXHJcbiAgICAgICAgICB0aGlzLmVudGl0eUNvbmZpZ0l0ZW1Db3B5XHJcbiAgICAgICAgKTtcclxuXHJcbiAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICBjYXNlIEJhc2VBY3Rpb25LZXkuRWRpdDpcclxuICAgICAgICB0aGlzLmNvbmZpZ0l0ZW1zVmlld01vZGUgPSBWaWV3TW9kZS5FZGl0O1xyXG4gICAgICAgIGJyZWFrO1xyXG4gICAgICBkZWZhdWx0OlxyXG4gICAgICAgIGJyZWFrO1xyXG4gICAgfVxyXG4gIH1cclxuXHJcbiAgLyoqXHJcbiAgICogVGhpcyBmdW5jdGlvbiBpcyByZXNwb25zaWJsZSB0byBzYXZlIGEgbmV3IGNvbmZpZyBpdGVtIG9yIHVwZGF0ZSBhbiBleGlzdGluZyBvbmVcclxuICAgKiBAcGFyYW0ge0R5bmFtaWNGb3JtQWN0aW9uT3V0cHV0fSBldmVudFxyXG4gICAqIEByZXR1cm5zIHtQcm9taXNlPHZvaWQ+fVxyXG4gICAqIEBwcml2YXRlXHJcbiAgICovXHJcbiAgcHJpdmF0ZSBhc3luYyBjb25maWdJdGVtc1NhdmUoZXZlbnQ6IER5bmFtaWNGb3JtQWN0aW9uT3V0cHV0KTogUHJvbWlzZTx2b2lkPiB7XHJcbiAgICB0aGlzLmNvbmZpZ0l0ZW1zVmlld01vZGUgPSBWaWV3TW9kZS5SZWFkO1xyXG5cclxuICAgIHRyeSB7XHJcbiAgICAgIHRoaXMuaW5pdEZvcm0gPSBmYWxzZTtcclxuICAgICAgYXdhaXQgdGhpcy5jb25maWd1cmF0aW9uSXRlbXNTZXJ2aWNlLnVwZGF0ZUNvbmZpZ3VyYXRpb25JdGVtc0FzeW5jKFxyXG4gICAgICAgIGV2ZW50LmZvcm1FbnRpdHlcclxuICAgICAgKTtcclxuICAgICAgdGhpcy5jb25maWdJdGVtc0xpc3QgPSB0aGlzLmFycmF5VXRpbGl0eVNlcnZpY2UuY2xvbmUoXHJcbiAgICAgICAgdGhpcy5jb25maWd1cmF0aW9uSXRlbXNTZXJ2aWNlLmdldENvbmZpZ3VyYXRpb25JdGVtc1ZhbHVlKClcclxuICAgICAgKTtcclxuXHJcbiAgICAgIHRoaXMuc2V0VHJlZURhdGEoKTtcclxuICAgICAgaWYgKHRoaXMuc2VsZWN0ZWROb2RlKSB7XHJcbiAgICAgICAgdGhpcy5vblNlbGVjdEl0ZW0odGhpcy5zZWxlY3RlZE5vZGUpO1xyXG4gICAgICB9XHJcbiAgICAgIHRoaXMuaW5pdEZvcm0gPSB0cnVlO1xyXG4gICAgfSBjYXRjaCAoZSkge31cclxuICB9XHJcbn1cclxuIiwiPGRpdiBjbGFzcz1cImdyaWQgZ3JpZC1ub2d1dHRlclwiPlxyXG4gIDxkaXYgY2xhc3M9XCJjb2wtZml4ZWQgZ3JpZC1ub2d1dHRlclwiPlxyXG4gICAgPHAtdHJlZSBbKHNlbGVjdGlvbildPVwic2VsZWN0ZWROb2RlXCIgIFtmaWx0ZXJdPVwidHJ1ZVwiIGZpbHRlclBsYWNlaG9sZGVyPVwiU2VhcmNoIEl0ZW0uLi5cIiBzZWxlY3Rpb25Nb2RlPVwic2luZ2xlXCJcclxuICAgICAgICAgICAgKG9uTm9kZVNlbGVjdCk9XCJvblNlbGVjdEl0ZW0oJGV2ZW50Lm5vZGUpXCIgbGF5b3V0PVwidmVydGljYWxcIiBbdmFsdWVdPVwiY29uZmlnSXRlbXNUcmVlXCI+PC9wLXRyZWU+XHJcbiAgPC9kaXY+XHJcbiA8ZGl2IGNsYXNzPVwiY29sIGdyaWQtbm9ndXR0ZXIgcHgtMiBcIiAqbmdJZj1cImluaXRGb3JtXCI+XHJcbiAgIDxwIFtuZ0NsYXNzXT1cIlsndGV4dC1jZW50ZXInICwnYmctYmx1ZS01MCcgLCd0ZXh0LTN4bCcgLCdib3JkZXItcm91bmQnLCAnbWItMicsICdweS0zJywgJ3ctZnVsbCcsICdib3JkZXItMzAwJywgJ2JvcmRlci0xJyAsJ3NoYWRvdy0xJywgJ210LTAnICwndGV4dC03MDAnXVwiICA+e3t0aXRsZUZvcm0gKyAnLmxhYmVsJ3x0cmFuc2xhdGV8dXBwZXJjYXNlfX08L3A+XHJcblxyXG4gICA8cHJnLWR5bmFtaWMtZm9ybSBbdmlld01vZGVdPVwiY29uZmlnSXRlbXNWaWV3TW9kZVwiIFthY3Rpb25zXT1cImNvbmZpZ0l0ZW1zRHluYW1pY0Zvcm1BY3Rpb25zXCJcclxuICAgICAgICAgICAgICAgICAgICAgW2ZpZWxkc109XCJjb25maWdJdGVtc0R5bmFtaWNGb3JtRmllbGRzXCIgW2VudGl0eV09XCJlbnRpdHlDb25maWdJdGVtc1wiXHJcbiAgICAgICAgICAgICAgICAgICAgIChhY3Rpb25PdXRwdXQpPVwib25BY3Rpb25Db25maWdJdGVtcygkZXZlbnQpXCI+XHJcbiAgIDwvcHJnLWR5bmFtaWMtZm9ybT5cclxuIDwvZGl2PlxyXG48L2Rpdj5cclxuXHJcbiJdfQ==