/**
 * This class define methods to validate passwords
 */
export class PrgAuthCustomPasswordRequirements {
    /**
     * Method to validate passwords through Regular Expression (RegExp)
     * @param {RegExp} regex
     * @param {ValidationErrors} error
     * @param {boolean} passwordRequirements
     * @returns {ValidatorFn}
     */
    static patternValidator(regex, error, passwordRequirements) {
        return (control) => {
            if (!control.value || !passwordRequirements) {
                return null;
            }
            const valid = regex.test(control.value);
            return valid ? null : error;
        };
    }
    /**
     * Method to validate passwords with a number of unique chars
     * @param {number} numberUniqueChars
     * @param {ValidationErrors} error
     * @returns {ValidatorFn}
     */
    static uniqueCharsValidator(numberUniqueChars, error) {
        return (control) => {
            if (!control.value) {
                return null;
            }
            const splitControlValueIntoArray = Array.from(control.value);
            const uniqueCharsArray = [...new Set(splitControlValueIntoArray)];
            if (uniqueCharsArray.length >= numberUniqueChars) {
                return null;
            }
            else {
                return error;
            }
        };
    }
}
/**
 * Method to validate if password and confirm password match
 * @param {AbstractControl} group
 * @returns {ValidationErrors | null}
 */
PrgAuthCustomPasswordRequirements.passwordMatchValidator = (group) => {
    let pass = group.get('password').value;
    let confirmPass = group.get('confirmPassword').value;
    return pass === confirmPass ? null : { passDontMatch: true };
};
//# sourceMappingURL=data:application/json;base64,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