import { __decorate, __param } from "tslib";
import { Inject } from '@angular/core';
import { BehaviorSubject } from 'rxjs';
/**
 * AbstractAuthService class that defines an
 * <br />abstract authentication service to
 * <br />be extended by the mock and real
 * <br />authentication services
 */
let AbstractAuthService = class AbstractAuthService {
    /**
     * constructor
     * @param authStorage
     * @protected
     */
    constructor(authStorage) {
        this.authStorage = authStorage;
        /**
         * Currently logged user
         */
        this.loggedUserSubject = new BehaviorSubject(null);
        /**
         * Currently stored username to be used
         * <br />on the login form
         * // CR: devemos explicar melhor para que serve esta propriedade, eu não estava a perceber até olhar para o código do login,
         *        talvez mudar também o nome para algo mais explicito
         */
        this.storedUserNameSubject = new BehaviorSubject('');
        this.getUserNameFromStorage();
        this.getUserLoggedFromStorage();
    }
    /**
     * Setter for loggedUserSubject
     *
     * @param {User} userLogged The User object
     */
    setLoggedUser(userLogged) {
        // CR: falta guardar isto na storage (session ou local), não?
        this.authStorage.storage.setItem(this.authStorage.keyLoggedUser, JSON.stringify(userLogged));
        this.loggedUserSubject.next(userLogged);
    }
    /**
     * Setter for storedUserNameSubject and saving username on local storage
     *
     * @param {string} value The storedUserName string
     */
    setStoredUserName(value) {
        // ok CR: falta guardar isto na storage (session ou local), não?
        this.authStorage.storage.setItem(this.authStorage.keyStoredUserName, JSON.stringify(value));
        this.storedUserNameSubject.next(value);
    }
    /**
     * Remove username from local storage
     *
     */
    removeStoredUserName() {
        this.authStorage.storage.removeItem(this.authStorage.keyStoredUserName);
        this.storedUserNameSubject.next(null);
    }
    /**
     * Getter for loggedUserSubject
     *
     * @returns The loggedUserSubject as an observable
     */
    getLoggedUserObservable() {
        return this.loggedUserSubject.asObservable();
    }
    /**
     * Getter for loggedUserSubject
     *
     * @returns The loggedUserSubject current value
     */
    getLoggedUser() {
        return this.loggedUserSubject.getValue();
    }
    /**
     * Getter for storedUserNameSubject
     *
     * @returns The storedUserNameSubject as an observable
     */
    getStoredUserNameObservable() {
        return this.storedUserNameSubject.asObservable();
    }
    /**
     * Getter for storedUserNameSubject
     *
     * @returns The storedUserNameSubject current value
     */
    getStoredUserName() {
        return this.storedUserNameSubject.getValue();
    }
    /**
     * Get username from local storage and if it exists,
     *  the value is emitted on storedUserNameSubject
     */
    getUserNameFromStorage() {
        const userName = JSON.parse(this.authStorage.storage.getItem(this.authStorage.keyStoredUserName));
        if (userName == null || userName.length == 0) {
            return;
        }
        else {
            this.setStoredUserName(userName);
        }
    }
    /**
     * Get the user logged from local storage
     * @private
     */
    getUserLoggedFromStorage() {
        const user = JSON.parse(this.authStorage.storage.getItem(this.authStorage.keyLoggedUser));
        if (user == null) {
            return;
        }
        else {
            this.setLoggedUser(user);
        }
    }
    /**
     * User logout Base Method (Async)
     *
     * @returns Boolean promise, true when successful
     */
    baseLogoutAsync() {
        return new Promise((resolve) => {
            this.authStorage.storage.removeItem(this.authStorage.keyLoggedUser);
            this.setLoggedUser(null);
            resolve(true);
        });
    }
};
AbstractAuthService = __decorate([
    __param(0, Inject('AUTHSTORAGE'))
], AbstractAuthService);
export { AbstractAuthService };
//# sourceMappingURL=data:application/json;base64,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