import { Inject, Injectable } from '@angular/core';
import { PrgError } from '../../../core/models/error.model';
import { PrgSucess } from '../../../core/models/success.model';
import { User } from '../../models/user.model';
import { AbstractAuthService } from '../../services/auth/abstract-auth.service';
import { UsersBDMOCk } from '../database.mock';
import * as i0 from "@angular/core";
import * as i1 from "../../../core/services/notifications/notifications.service";
import * as i2 from "../../../core/services/utility/utility.service";
import * as i3 from "../../../core/services/is-loading-data/isloadingdata.service";
/**
 * Injectable
 */
/**
 * MOCK AbstractAuthService implementation
 */
export class MockAuthService extends AbstractAuthService {
    /**
     * constructor
     * @param authStorage
     * @param notificationsService
     * @param utilityService
     * @param isLoadingData
     */
    constructor(authStorage, notificationsService, utilityService, isLoadingData) {
        super(authStorage);
        this.notificationsService = notificationsService;
        this.utilityService = utilityService;
        this.isLoadingData = isLoadingData;
        /**
         * default time in seconds
         */
        this.sleepTimeSec = 2;
    }
    /**
     * User login (Async)
     *
     * @param {string} email The user e-mail
     * @param {string} password The user password
     * @param {boolean} storeUserName Store username
     * @returns Boolean promise, true when successful
     */
    loginAsync(email, password, storeUserName) {
        return new Promise(async (resolve, reject) => {
            this.isLoadingData.show();
            await this.utilityService.sleepSecAsync(this.sleepTimeSec);
            const user = UsersBDMOCk.Users.find((u) => u.email === email || u.name === email);
            this.isLoadingData.hide();
            if (user && password === user.password) {
                this.setLoggedUser(user);
                if (storeUserName) {
                    this.setStoredUserName(email);
                }
                else {
                    this.removeStoredUserName();
                }
                resolve(true);
            }
            else {
                reject('Authentication has failed');
                this.notificationsService.errorNotification(new PrgError({
                    titleKey: 'Login Error.',
                    detailKey: 'Invalid e-mail or password',
                }));
            }
        });
    }
    /**
     * User logout (Async)
     *
     * @returns Boolean promise, true when successful
     */
    logoutAsync() {
        return new Promise(async (resolve) => {
            const checkUserLogOut = await this.baseLogoutAsync();
            if (checkUserLogOut) {
                resolve(true);
            }
        });
    }
    /**
     * User sign up (Async)
     *
     * @returns Boolean promise, true when successful
     */
    signupAsync(email) {
        return new Promise(async (resolve, reject) => {
            this.isLoadingData.show();
            await this.utilityService.sleepSecAsync(this.sleepTimeSec);
            const checkUser = UsersBDMOCk.Users.find((u) => u.email === email);
            this.isLoadingData.hide();
            if (!checkUser) {
                let user = new User();
                user.name = email;
                user.id = this.utilityService.newGuid();
                user.token = null;
                user.email = email;
                UsersBDMOCk.Users.push(user);
                resolve(true);
                this.notificationsService.successNotification(new PrgSucess({
                    titleKey: 'Registration completed.',
                    detailKey: 'An email as been sent with a temporary password.',
                }));
            }
            else {
                reject('Registration has failed.');
                this.notificationsService.errorNotification(new PrgError({
                    titleKey: 'Registration Error.',
                    detailKey: `The email ${email} already exists in database`,
                }));
            }
        });
    }
    /**
     * Request to reset the password (Async)
     *
     * @param {string} email The user e-mail
     * @returns Boolean promise, true when successful
     */
    requestResetPasswordAsync(email) {
        return new Promise(async (resolve, reject) => {
            this.isLoadingData.show();
            await this.utilityService.sleepSecAsync(this.sleepTimeSec);
            const checkUser = UsersBDMOCk.Users.find((u) => u.email === email);
            this.isLoadingData.hide();
            if (checkUser) {
                resolve(true);
                this.notificationsService.successNotification(new PrgSucess({
                    titleKey: 'Request Password completed.',
                    detailKey: 'An email as been sent with a link to reset password.',
                }));
            }
            else {
                reject('Request Password Failed');
                this.notificationsService.errorNotification(new PrgError({
                    titleKey: 'Request Password Failed',
                    detailKey: `The email ${email} doesn't exist in database.`,
                }));
            }
        });
    }
    /**
     * Getter for token reset password (Async)
     *
     * @param {string} tokenReset The token reset password
     * @returns Boolean promise, true when successful
     */
    getTokenResetPasswordAsync(tokenReset) {
        return new Promise((resolve) => {
            const checkUser = UsersBDMOCk.Users.find((u) => u.tokenResetPassword === tokenReset);
            if (checkUser) {
                resolve(true);
            }
            else {
                resolve(false);
            }
        });
    }
    /**
     * Method for reset password (Async)
     *
     * @param {string} newPassword The new password
     * @param {string} tokenReset The token reset password
     * @returns Boolean promise, true when successful
     */
    resetPasswordAsync(newPassword, tokenReset) {
        return new Promise(async (resolve, reject) => {
            this.isLoadingData.show();
            await this.utilityService.sleepSecAsync(this.sleepTimeSec);
            const checkUser = UsersBDMOCk.Users.find((u) => u.tokenResetPassword === tokenReset);
            this.isLoadingData.hide();
            if (checkUser) {
                checkUser.password = newPassword;
                resolve(true);
                this.notificationsService.successNotification(new PrgSucess({
                    titleKey: 'New Password',
                    detailKey: 'Password Changed Successfully',
                }));
            }
            else {
                reject('The password change operation has failed.');
                this.notificationsService.errorNotification(new PrgError({
                    titleKey: 'Password Error.',
                    detailKey: `The password change operation has failed.`,
                }));
            }
        });
    }
}
MockAuthService.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "14.2.0", ngImport: i0, type: MockAuthService, deps: [{ token: 'AUTHSTORAGE' }, { token: i1.NotificationsService }, { token: i2.UtilityService }, { token: i3.IsLoadingDataService }], target: i0.ɵɵFactoryTarget.Injectable });
MockAuthService.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "14.2.0", ngImport: i0, type: MockAuthService, providedIn: 'root' });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "14.2.0", ngImport: i0, type: MockAuthService, decorators: [{
            type: Injectable,
            args: [{
                    providedIn: 'root',
                }]
        }], ctorParameters: function () { return [{ type: undefined, decorators: [{
                    type: Inject,
                    args: ['AUTHSTORAGE']
                }] }, { type: i1.NotificationsService }, { type: i2.UtilityService }, { type: i3.IsLoadingDataService }]; } });
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXV0aC5zZXJ2aWNlLm1vY2suanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi8uLi8uLi8uLi8uLi8uLi9wcm9qZWN0cy9QcmdGcmFtZXdvcmsuRkUuQ29yZUxpYi9zcmMvbGliL21vZHVsZXMvYXV0aC9tb2NrL3NlcnZpY2VzL2F1dGguc2VydmljZS5tb2NrLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFBLE9BQU8sRUFBRSxNQUFNLEVBQUUsVUFBVSxFQUFFLE1BQU0sZUFBZSxDQUFDO0FBQ25ELE9BQU8sRUFBRSxRQUFRLEVBQUUsTUFBTSxrQ0FBa0MsQ0FBQztBQUM1RCxPQUFPLEVBQUUsU0FBUyxFQUFFLE1BQU0sb0NBQW9DLENBQUM7QUFLL0QsT0FBTyxFQUFFLElBQUksRUFBRSxNQUFNLHlCQUF5QixDQUFDO0FBQy9DLE9BQU8sRUFBRSxtQkFBbUIsRUFBRSxNQUFNLDJDQUEyQyxDQUFDO0FBQ2hGLE9BQU8sRUFBRSxXQUFXLEVBQUUsTUFBTSxrQkFBa0IsQ0FBQzs7Ozs7QUFFL0M7O0dBRUc7QUFJSDs7R0FFRztBQUNILE1BQU0sT0FBTyxlQUFnQixTQUFRLG1CQUFtQjtJQU10RDs7Ozs7O09BTUc7SUFDSCxZQUN5QixXQUF3QixFQUN2QyxvQkFBMEMsRUFDMUMsY0FBOEIsRUFDOUIsYUFBbUM7UUFFM0MsS0FBSyxDQUFDLFdBQVcsQ0FBQyxDQUFDO1FBSlgseUJBQW9CLEdBQXBCLG9CQUFvQixDQUFzQjtRQUMxQyxtQkFBYyxHQUFkLGNBQWMsQ0FBZ0I7UUFDOUIsa0JBQWEsR0FBYixhQUFhLENBQXNCO1FBaEI3Qzs7V0FFRztRQUNLLGlCQUFZLEdBQUcsQ0FBQyxDQUFDO0lBZ0J6QixDQUFDO0lBRUQ7Ozs7Ozs7T0FPRztJQUNJLFVBQVUsQ0FDZixLQUFhLEVBQ2IsUUFBZ0IsRUFDaEIsYUFBc0I7UUFFdEIsT0FBTyxJQUFJLE9BQU8sQ0FBVSxLQUFLLEVBQUUsT0FBTyxFQUFFLE1BQU0sRUFBRSxFQUFFO1lBQ3BELElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLENBQUM7WUFDMUIsTUFBTSxJQUFJLENBQUMsY0FBYyxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLENBQUM7WUFDM0QsTUFBTSxJQUFJLEdBQUcsV0FBVyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQ2pDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsS0FBSyxLQUFLLEtBQUssSUFBSSxDQUFDLENBQUMsSUFBSSxLQUFLLEtBQUssQ0FDN0MsQ0FBQztZQUNGLElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLENBQUM7WUFDMUIsSUFBSSxJQUFJLElBQUksUUFBUSxLQUFLLElBQUksQ0FBQyxRQUFRLEVBQUU7Z0JBQ3RDLElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLENBQUM7Z0JBQ3pCLElBQUksYUFBYSxFQUFFO29CQUNqQixJQUFJLENBQUMsaUJBQWlCLENBQUMsS0FBSyxDQUFDLENBQUM7aUJBQy9CO3FCQUFNO29CQUNMLElBQUksQ0FBQyxvQkFBb0IsRUFBRSxDQUFDO2lCQUM3QjtnQkFDRCxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUM7YUFDZjtpQkFBTTtnQkFDTCxNQUFNLENBQUMsMkJBQTJCLENBQUMsQ0FBQztnQkFDcEMsSUFBSSxDQUFDLG9CQUFvQixDQUFDLGlCQUFpQixDQUN6QyxJQUFJLFFBQVEsQ0FBQztvQkFDWCxRQUFRLEVBQUUsY0FBYztvQkFDeEIsU0FBUyxFQUFFLDRCQUE0QjtpQkFDeEMsQ0FBQyxDQUNILENBQUM7YUFDSDtRQUNILENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxXQUFXO1FBQ2hCLE9BQU8sSUFBSSxPQUFPLENBQVUsS0FBSyxFQUFFLE9BQU8sRUFBRSxFQUFFO1lBQzVDLE1BQU0sZUFBZSxHQUFHLE1BQU0sSUFBSSxDQUFDLGVBQWUsRUFBRSxDQUFDO1lBQ3JELElBQUksZUFBZSxFQUFFO2dCQUNuQixPQUFPLENBQUMsSUFBSSxDQUFDLENBQUM7YUFDZjtRQUNILENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxXQUFXLENBQUMsS0FBYTtRQUM5QixPQUFPLElBQUksT0FBTyxDQUFVLEtBQUssRUFBRSxPQUFPLEVBQUUsTUFBTSxFQUFFLEVBQUU7WUFDcEQsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUUsQ0FBQztZQUMxQixNQUFNLElBQUksQ0FBQyxjQUFjLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQztZQUMzRCxNQUFNLFNBQVMsR0FBRyxXQUFXLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLEtBQUssS0FBSyxLQUFLLENBQUMsQ0FBQztZQUNuRSxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksRUFBRSxDQUFDO1lBQzFCLElBQUksQ0FBQyxTQUFTLEVBQUU7Z0JBQ2QsSUFBSSxJQUFJLEdBQUcsSUFBSSxJQUFJLEVBQUUsQ0FBQztnQkFDdEIsSUFBSSxDQUFDLElBQUksR0FBRyxLQUFLLENBQUM7Z0JBQ2xCLElBQUksQ0FBQyxFQUFFLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxPQUFPLEVBQUUsQ0FBQztnQkFDeEMsSUFBSSxDQUFDLEtBQUssR0FBRyxJQUFJLENBQUM7Z0JBQ2xCLElBQUksQ0FBQyxLQUFLLEdBQUcsS0FBSyxDQUFDO2dCQUNuQixXQUFXLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztnQkFDN0IsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDO2dCQUNkLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxtQkFBbUIsQ0FDM0MsSUFBSSxTQUFTLENBQUM7b0JBQ1osUUFBUSxFQUFFLHlCQUF5QjtvQkFDbkMsU0FBUyxFQUFFLGtEQUFrRDtpQkFDOUQsQ0FBQyxDQUNILENBQUM7YUFDSDtpQkFBTTtnQkFDTCxNQUFNLENBQUMsMEJBQTBCLENBQUMsQ0FBQztnQkFDbkMsSUFBSSxDQUFDLG9CQUFvQixDQUFDLGlCQUFpQixDQUN6QyxJQUFJLFFBQVEsQ0FBQztvQkFDWCxRQUFRLEVBQUUscUJBQXFCO29CQUMvQixTQUFTLEVBQUUsYUFBYSxLQUFLLDZCQUE2QjtpQkFDM0QsQ0FBQyxDQUNILENBQUM7YUFDSDtRQUNILENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0kseUJBQXlCLENBQUMsS0FBYTtRQUM1QyxPQUFPLElBQUksT0FBTyxDQUFVLEtBQUssRUFBRSxPQUFPLEVBQUUsTUFBTSxFQUFFLEVBQUU7WUFDcEQsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUUsQ0FBQztZQUMxQixNQUFNLElBQUksQ0FBQyxjQUFjLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQztZQUMzRCxNQUFNLFNBQVMsR0FBRyxXQUFXLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLEtBQUssS0FBSyxLQUFLLENBQUMsQ0FBQztZQUNuRSxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksRUFBRSxDQUFDO1lBQzFCLElBQUksU0FBUyxFQUFFO2dCQUNiLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQztnQkFDZCxJQUFJLENBQUMsb0JBQW9CLENBQUMsbUJBQW1CLENBQzNDLElBQUksU0FBUyxDQUFDO29CQUNaLFFBQVEsRUFBRSw2QkFBNkI7b0JBQ3ZDLFNBQVMsRUFBRSxzREFBc0Q7aUJBQ2xFLENBQUMsQ0FDSCxDQUFDO2FBQ0g7aUJBQU07Z0JBQ0wsTUFBTSxDQUFDLHlCQUF5QixDQUFDLENBQUM7Z0JBQ2xDLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxpQkFBaUIsQ0FDekMsSUFBSSxRQUFRLENBQUM7b0JBQ1gsUUFBUSxFQUFFLHlCQUF5QjtvQkFDbkMsU0FBUyxFQUFFLGFBQWEsS0FBSyw2QkFBNkI7aUJBQzNELENBQUMsQ0FDSCxDQUFDO2FBQ0g7UUFDSCxDQUFDLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLDBCQUEwQixDQUFDLFVBQWtCO1FBQ2xELE9BQU8sSUFBSSxPQUFPLENBQVUsQ0FBQyxPQUFPLEVBQUUsRUFBRTtZQUN0QyxNQUFNLFNBQVMsR0FBRyxXQUFXLENBQUMsS0FBSyxDQUFDLElBQUksQ0FDdEMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxrQkFBa0IsS0FBSyxVQUFVLENBQzNDLENBQUM7WUFDRixJQUFJLFNBQVMsRUFBRTtnQkFDYixPQUFPLENBQUMsSUFBSSxDQUFDLENBQUM7YUFDZjtpQkFBTTtnQkFDTCxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUM7YUFDaEI7UUFDSCxDQUFDLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0IsQ0FDdkIsV0FBbUIsRUFDbkIsVUFBa0I7UUFFbEIsT0FBTyxJQUFJLE9BQU8sQ0FBVSxLQUFLLEVBQUUsT0FBTyxFQUFFLE1BQU0sRUFBRSxFQUFFO1lBQ3BELElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLENBQUM7WUFDMUIsTUFBTSxJQUFJLENBQUMsY0FBYyxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLENBQUM7WUFDM0QsTUFBTSxTQUFTLEdBQUcsV0FBVyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQ3RDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsa0JBQWtCLEtBQUssVUFBVSxDQUMzQyxDQUFDO1lBQ0YsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUUsQ0FBQztZQUMxQixJQUFJLFNBQVMsRUFBRTtnQkFDYixTQUFTLENBQUMsUUFBUSxHQUFHLFdBQVcsQ0FBQztnQkFDakMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDO2dCQUNkLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxtQkFBbUIsQ0FDM0MsSUFBSSxTQUFTLENBQUM7b0JBQ1osUUFBUSxFQUFFLGNBQWM7b0JBQ3hCLFNBQVMsRUFBRSwrQkFBK0I7aUJBQzNDLENBQUMsQ0FDSCxDQUFDO2FBQ0g7aUJBQU07Z0JBQ0wsTUFBTSxDQUFDLDJDQUEyQyxDQUFDLENBQUM7Z0JBQ3BELElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxpQkFBaUIsQ0FDekMsSUFBSSxRQUFRLENBQUM7b0JBQ1gsUUFBUSxFQUFFLGlCQUFpQjtvQkFDM0IsU0FBUyxFQUFFLDJDQUEyQztpQkFDdkQsQ0FBQyxDQUNILENBQUM7YUFDSDtRQUNILENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQzs7NEdBek1VLGVBQWUsa0JBY2hCLGFBQWE7Z0hBZFosZUFBZSxjQUxkLE1BQU07MkZBS1AsZUFBZTtrQkFOM0IsVUFBVTttQkFBQztvQkFDVixVQUFVLEVBQUUsTUFBTTtpQkFDbkI7OzBCQWtCSSxNQUFNOzJCQUFDLGFBQWEiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBJbmplY3QsIEluamVjdGFibGUgfSBmcm9tICdAYW5ndWxhci9jb3JlJztcclxuaW1wb3J0IHsgUHJnRXJyb3IgfSBmcm9tICcuLi8uLi8uLi9jb3JlL21vZGVscy9lcnJvci5tb2RlbCc7XHJcbmltcG9ydCB7IFByZ1N1Y2VzcyB9IGZyb20gJy4uLy4uLy4uL2NvcmUvbW9kZWxzL3N1Y2Nlc3MubW9kZWwnO1xyXG5pbXBvcnQgeyBJc0xvYWRpbmdEYXRhU2VydmljZSB9IGZyb20gJy4uLy4uLy4uL2NvcmUvc2VydmljZXMvaXMtbG9hZGluZy1kYXRhL2lzbG9hZGluZ2RhdGEuc2VydmljZSc7XHJcbmltcG9ydCB7IE5vdGlmaWNhdGlvbnNTZXJ2aWNlIH0gZnJvbSAnLi4vLi4vLi4vY29yZS9zZXJ2aWNlcy9ub3RpZmljYXRpb25zL25vdGlmaWNhdGlvbnMuc2VydmljZSc7XHJcbmltcG9ydCB7IFV0aWxpdHlTZXJ2aWNlIH0gZnJvbSAnLi4vLi4vLi4vY29yZS9zZXJ2aWNlcy91dGlsaXR5L3V0aWxpdHkuc2VydmljZSc7XHJcbmltcG9ydCB7IEF1dGhTdG9yYWdlIH0gZnJvbSAnLi4vLi4vbW9kZWxzL3ByZy1hdXRoLWNvbmZpZyc7XHJcbmltcG9ydCB7IFVzZXIgfSBmcm9tICcuLi8uLi9tb2RlbHMvdXNlci5tb2RlbCc7XHJcbmltcG9ydCB7IEFic3RyYWN0QXV0aFNlcnZpY2UgfSBmcm9tICcuLi8uLi9zZXJ2aWNlcy9hdXRoL2Fic3RyYWN0LWF1dGguc2VydmljZSc7XHJcbmltcG9ydCB7IFVzZXJzQkRNT0NrIH0gZnJvbSAnLi4vZGF0YWJhc2UubW9jayc7XHJcblxyXG4vKipcclxuICogSW5qZWN0YWJsZVxyXG4gKi9cclxuQEluamVjdGFibGUoe1xyXG4gIHByb3ZpZGVkSW46ICdyb290JyxcclxufSlcclxuLyoqXHJcbiAqIE1PQ0sgQWJzdHJhY3RBdXRoU2VydmljZSBpbXBsZW1lbnRhdGlvblxyXG4gKi9cclxuZXhwb3J0IGNsYXNzIE1vY2tBdXRoU2VydmljZSBleHRlbmRzIEFic3RyYWN0QXV0aFNlcnZpY2Uge1xyXG4gIC8qKlxyXG4gICAqIGRlZmF1bHQgdGltZSBpbiBzZWNvbmRzXHJcbiAgICovXHJcbiAgcHJpdmF0ZSBzbGVlcFRpbWVTZWMgPSAyO1xyXG5cclxuICAvKipcclxuICAgKiBjb25zdHJ1Y3RvclxyXG4gICAqIEBwYXJhbSBhdXRoU3RvcmFnZVxyXG4gICAqIEBwYXJhbSBub3RpZmljYXRpb25zU2VydmljZVxyXG4gICAqIEBwYXJhbSB1dGlsaXR5U2VydmljZVxyXG4gICAqIEBwYXJhbSBpc0xvYWRpbmdEYXRhXHJcbiAgICovXHJcbiAgY29uc3RydWN0b3IoXHJcbiAgICBASW5qZWN0KCdBVVRIU1RPUkFHRScpIGF1dGhTdG9yYWdlOiBBdXRoU3RvcmFnZSxcclxuICAgIHByaXZhdGUgbm90aWZpY2F0aW9uc1NlcnZpY2U6IE5vdGlmaWNhdGlvbnNTZXJ2aWNlLFxyXG4gICAgcHJpdmF0ZSB1dGlsaXR5U2VydmljZTogVXRpbGl0eVNlcnZpY2UsXHJcbiAgICBwcml2YXRlIGlzTG9hZGluZ0RhdGE6IElzTG9hZGluZ0RhdGFTZXJ2aWNlXHJcbiAgKSB7XHJcbiAgICBzdXBlcihhdXRoU3RvcmFnZSk7XHJcbiAgfVxyXG5cclxuICAvKipcclxuICAgKiBVc2VyIGxvZ2luIChBc3luYylcclxuICAgKlxyXG4gICAqIEBwYXJhbSB7c3RyaW5nfSBlbWFpbCBUaGUgdXNlciBlLW1haWxcclxuICAgKiBAcGFyYW0ge3N0cmluZ30gcGFzc3dvcmQgVGhlIHVzZXIgcGFzc3dvcmRcclxuICAgKiBAcGFyYW0ge2Jvb2xlYW59IHN0b3JlVXNlck5hbWUgU3RvcmUgdXNlcm5hbWVcclxuICAgKiBAcmV0dXJucyBCb29sZWFuIHByb21pc2UsIHRydWUgd2hlbiBzdWNjZXNzZnVsXHJcbiAgICovXHJcbiAgcHVibGljIGxvZ2luQXN5bmMoXHJcbiAgICBlbWFpbDogc3RyaW5nLFxyXG4gICAgcGFzc3dvcmQ6IHN0cmluZyxcclxuICAgIHN0b3JlVXNlck5hbWU6IGJvb2xlYW5cclxuICApOiBQcm9taXNlPGJvb2xlYW4+IHtcclxuICAgIHJldHVybiBuZXcgUHJvbWlzZTxib29sZWFuPihhc3luYyAocmVzb2x2ZSwgcmVqZWN0KSA9PiB7XHJcbiAgICAgIHRoaXMuaXNMb2FkaW5nRGF0YS5zaG93KCk7XHJcbiAgICAgIGF3YWl0IHRoaXMudXRpbGl0eVNlcnZpY2Uuc2xlZXBTZWNBc3luYyh0aGlzLnNsZWVwVGltZVNlYyk7XHJcbiAgICAgIGNvbnN0IHVzZXIgPSBVc2Vyc0JETU9Day5Vc2Vycy5maW5kKFxyXG4gICAgICAgICh1KSA9PiB1LmVtYWlsID09PSBlbWFpbCB8fCB1Lm5hbWUgPT09IGVtYWlsXHJcbiAgICAgICk7XHJcbiAgICAgIHRoaXMuaXNMb2FkaW5nRGF0YS5oaWRlKCk7XHJcbiAgICAgIGlmICh1c2VyICYmIHBhc3N3b3JkID09PSB1c2VyLnBhc3N3b3JkKSB7XHJcbiAgICAgICAgdGhpcy5zZXRMb2dnZWRVc2VyKHVzZXIpO1xyXG4gICAgICAgIGlmIChzdG9yZVVzZXJOYW1lKSB7XHJcbiAgICAgICAgICB0aGlzLnNldFN0b3JlZFVzZXJOYW1lKGVtYWlsKTtcclxuICAgICAgICB9IGVsc2Uge1xyXG4gICAgICAgICAgdGhpcy5yZW1vdmVTdG9yZWRVc2VyTmFtZSgpO1xyXG4gICAgICAgIH1cclxuICAgICAgICByZXNvbHZlKHRydWUpO1xyXG4gICAgICB9IGVsc2Uge1xyXG4gICAgICAgIHJlamVjdCgnQXV0aGVudGljYXRpb24gaGFzIGZhaWxlZCcpO1xyXG4gICAgICAgIHRoaXMubm90aWZpY2F0aW9uc1NlcnZpY2UuZXJyb3JOb3RpZmljYXRpb24oXHJcbiAgICAgICAgICBuZXcgUHJnRXJyb3Ioe1xyXG4gICAgICAgICAgICB0aXRsZUtleTogJ0xvZ2luIEVycm9yLicsXHJcbiAgICAgICAgICAgIGRldGFpbEtleTogJ0ludmFsaWQgZS1tYWlsIG9yIHBhc3N3b3JkJyxcclxuICAgICAgICAgIH0pXHJcbiAgICAgICAgKTtcclxuICAgICAgfVxyXG4gICAgfSk7XHJcbiAgfVxyXG5cclxuICAvKipcclxuICAgKiBVc2VyIGxvZ291dCAoQXN5bmMpXHJcbiAgICpcclxuICAgKiBAcmV0dXJucyBCb29sZWFuIHByb21pc2UsIHRydWUgd2hlbiBzdWNjZXNzZnVsXHJcbiAgICovXHJcbiAgcHVibGljIGxvZ291dEFzeW5jKCk6IFByb21pc2U8Ym9vbGVhbj4ge1xyXG4gICAgcmV0dXJuIG5ldyBQcm9taXNlPGJvb2xlYW4+KGFzeW5jIChyZXNvbHZlKSA9PiB7XHJcbiAgICAgIGNvbnN0IGNoZWNrVXNlckxvZ091dCA9IGF3YWl0IHRoaXMuYmFzZUxvZ291dEFzeW5jKCk7XHJcbiAgICAgIGlmIChjaGVja1VzZXJMb2dPdXQpIHtcclxuICAgICAgICByZXNvbHZlKHRydWUpO1xyXG4gICAgICB9XHJcbiAgICB9KTtcclxuICB9XHJcblxyXG4gIC8qKlxyXG4gICAqIFVzZXIgc2lnbiB1cCAoQXN5bmMpXHJcbiAgICpcclxuICAgKiBAcmV0dXJucyBCb29sZWFuIHByb21pc2UsIHRydWUgd2hlbiBzdWNjZXNzZnVsXHJcbiAgICovXHJcbiAgcHVibGljIHNpZ251cEFzeW5jKGVtYWlsOiBzdHJpbmcpOiBQcm9taXNlPGJvb2xlYW4+IHtcclxuICAgIHJldHVybiBuZXcgUHJvbWlzZTxib29sZWFuPihhc3luYyAocmVzb2x2ZSwgcmVqZWN0KSA9PiB7XHJcbiAgICAgIHRoaXMuaXNMb2FkaW5nRGF0YS5zaG93KCk7XHJcbiAgICAgIGF3YWl0IHRoaXMudXRpbGl0eVNlcnZpY2Uuc2xlZXBTZWNBc3luYyh0aGlzLnNsZWVwVGltZVNlYyk7XHJcbiAgICAgIGNvbnN0IGNoZWNrVXNlciA9IFVzZXJzQkRNT0NrLlVzZXJzLmZpbmQoKHUpID0+IHUuZW1haWwgPT09IGVtYWlsKTtcclxuICAgICAgdGhpcy5pc0xvYWRpbmdEYXRhLmhpZGUoKTtcclxuICAgICAgaWYgKCFjaGVja1VzZXIpIHtcclxuICAgICAgICBsZXQgdXNlciA9IG5ldyBVc2VyKCk7XHJcbiAgICAgICAgdXNlci5uYW1lID0gZW1haWw7XHJcbiAgICAgICAgdXNlci5pZCA9IHRoaXMudXRpbGl0eVNlcnZpY2UubmV3R3VpZCgpO1xyXG4gICAgICAgIHVzZXIudG9rZW4gPSBudWxsO1xyXG4gICAgICAgIHVzZXIuZW1haWwgPSBlbWFpbDtcclxuICAgICAgICBVc2Vyc0JETU9Day5Vc2Vycy5wdXNoKHVzZXIpO1xyXG4gICAgICAgIHJlc29sdmUodHJ1ZSk7XHJcbiAgICAgICAgdGhpcy5ub3RpZmljYXRpb25zU2VydmljZS5zdWNjZXNzTm90aWZpY2F0aW9uKFxyXG4gICAgICAgICAgbmV3IFByZ1N1Y2Vzcyh7XHJcbiAgICAgICAgICAgIHRpdGxlS2V5OiAnUmVnaXN0cmF0aW9uIGNvbXBsZXRlZC4nLFxyXG4gICAgICAgICAgICBkZXRhaWxLZXk6ICdBbiBlbWFpbCBhcyBiZWVuIHNlbnQgd2l0aCBhIHRlbXBvcmFyeSBwYXNzd29yZC4nLFxyXG4gICAgICAgICAgfSlcclxuICAgICAgICApO1xyXG4gICAgICB9IGVsc2Uge1xyXG4gICAgICAgIHJlamVjdCgnUmVnaXN0cmF0aW9uIGhhcyBmYWlsZWQuJyk7XHJcbiAgICAgICAgdGhpcy5ub3RpZmljYXRpb25zU2VydmljZS5lcnJvck5vdGlmaWNhdGlvbihcclxuICAgICAgICAgIG5ldyBQcmdFcnJvcih7XHJcbiAgICAgICAgICAgIHRpdGxlS2V5OiAnUmVnaXN0cmF0aW9uIEVycm9yLicsXHJcbiAgICAgICAgICAgIGRldGFpbEtleTogYFRoZSBlbWFpbCAke2VtYWlsfSBhbHJlYWR5IGV4aXN0cyBpbiBkYXRhYmFzZWAsXHJcbiAgICAgICAgICB9KVxyXG4gICAgICAgICk7XHJcbiAgICAgIH1cclxuICAgIH0pO1xyXG4gIH1cclxuXHJcbiAgLyoqXHJcbiAgICogUmVxdWVzdCB0byByZXNldCB0aGUgcGFzc3dvcmQgKEFzeW5jKVxyXG4gICAqXHJcbiAgICogQHBhcmFtIHtzdHJpbmd9IGVtYWlsIFRoZSB1c2VyIGUtbWFpbFxyXG4gICAqIEByZXR1cm5zIEJvb2xlYW4gcHJvbWlzZSwgdHJ1ZSB3aGVuIHN1Y2Nlc3NmdWxcclxuICAgKi9cclxuICBwdWJsaWMgcmVxdWVzdFJlc2V0UGFzc3dvcmRBc3luYyhlbWFpbDogc3RyaW5nKTogUHJvbWlzZTxib29sZWFuPiB7XHJcbiAgICByZXR1cm4gbmV3IFByb21pc2U8Ym9vbGVhbj4oYXN5bmMgKHJlc29sdmUsIHJlamVjdCkgPT4ge1xyXG4gICAgICB0aGlzLmlzTG9hZGluZ0RhdGEuc2hvdygpO1xyXG4gICAgICBhd2FpdCB0aGlzLnV0aWxpdHlTZXJ2aWNlLnNsZWVwU2VjQXN5bmModGhpcy5zbGVlcFRpbWVTZWMpO1xyXG4gICAgICBjb25zdCBjaGVja1VzZXIgPSBVc2Vyc0JETU9Day5Vc2Vycy5maW5kKCh1KSA9PiB1LmVtYWlsID09PSBlbWFpbCk7XHJcbiAgICAgIHRoaXMuaXNMb2FkaW5nRGF0YS5oaWRlKCk7XHJcbiAgICAgIGlmIChjaGVja1VzZXIpIHtcclxuICAgICAgICByZXNvbHZlKHRydWUpO1xyXG4gICAgICAgIHRoaXMubm90aWZpY2F0aW9uc1NlcnZpY2Uuc3VjY2Vzc05vdGlmaWNhdGlvbihcclxuICAgICAgICAgIG5ldyBQcmdTdWNlc3Moe1xyXG4gICAgICAgICAgICB0aXRsZUtleTogJ1JlcXVlc3QgUGFzc3dvcmQgY29tcGxldGVkLicsXHJcbiAgICAgICAgICAgIGRldGFpbEtleTogJ0FuIGVtYWlsIGFzIGJlZW4gc2VudCB3aXRoIGEgbGluayB0byByZXNldCBwYXNzd29yZC4nLFxyXG4gICAgICAgICAgfSlcclxuICAgICAgICApO1xyXG4gICAgICB9IGVsc2Uge1xyXG4gICAgICAgIHJlamVjdCgnUmVxdWVzdCBQYXNzd29yZCBGYWlsZWQnKTtcclxuICAgICAgICB0aGlzLm5vdGlmaWNhdGlvbnNTZXJ2aWNlLmVycm9yTm90aWZpY2F0aW9uKFxyXG4gICAgICAgICAgbmV3IFByZ0Vycm9yKHtcclxuICAgICAgICAgICAgdGl0bGVLZXk6ICdSZXF1ZXN0IFBhc3N3b3JkIEZhaWxlZCcsXHJcbiAgICAgICAgICAgIGRldGFpbEtleTogYFRoZSBlbWFpbCAke2VtYWlsfSBkb2Vzbid0IGV4aXN0IGluIGRhdGFiYXNlLmAsXHJcbiAgICAgICAgICB9KVxyXG4gICAgICAgICk7XHJcbiAgICAgIH1cclxuICAgIH0pO1xyXG4gIH1cclxuXHJcbiAgLyoqXHJcbiAgICogR2V0dGVyIGZvciB0b2tlbiByZXNldCBwYXNzd29yZCAoQXN5bmMpXHJcbiAgICpcclxuICAgKiBAcGFyYW0ge3N0cmluZ30gdG9rZW5SZXNldCBUaGUgdG9rZW4gcmVzZXQgcGFzc3dvcmRcclxuICAgKiBAcmV0dXJucyBCb29sZWFuIHByb21pc2UsIHRydWUgd2hlbiBzdWNjZXNzZnVsXHJcbiAgICovXHJcbiAgcHVibGljIGdldFRva2VuUmVzZXRQYXNzd29yZEFzeW5jKHRva2VuUmVzZXQ6IHN0cmluZyk6IFByb21pc2U8Ym9vbGVhbj4ge1xyXG4gICAgcmV0dXJuIG5ldyBQcm9taXNlPGJvb2xlYW4+KChyZXNvbHZlKSA9PiB7XHJcbiAgICAgIGNvbnN0IGNoZWNrVXNlciA9IFVzZXJzQkRNT0NrLlVzZXJzLmZpbmQoXHJcbiAgICAgICAgKHUpID0+IHUudG9rZW5SZXNldFBhc3N3b3JkID09PSB0b2tlblJlc2V0XHJcbiAgICAgICk7XHJcbiAgICAgIGlmIChjaGVja1VzZXIpIHtcclxuICAgICAgICByZXNvbHZlKHRydWUpO1xyXG4gICAgICB9IGVsc2Uge1xyXG4gICAgICAgIHJlc29sdmUoZmFsc2UpO1xyXG4gICAgICB9XHJcbiAgICB9KTtcclxuICB9XHJcblxyXG4gIC8qKlxyXG4gICAqIE1ldGhvZCBmb3IgcmVzZXQgcGFzc3dvcmQgKEFzeW5jKVxyXG4gICAqXHJcbiAgICogQHBhcmFtIHtzdHJpbmd9IG5ld1Bhc3N3b3JkIFRoZSBuZXcgcGFzc3dvcmRcclxuICAgKiBAcGFyYW0ge3N0cmluZ30gdG9rZW5SZXNldCBUaGUgdG9rZW4gcmVzZXQgcGFzc3dvcmRcclxuICAgKiBAcmV0dXJucyBCb29sZWFuIHByb21pc2UsIHRydWUgd2hlbiBzdWNjZXNzZnVsXHJcbiAgICovXHJcbiAgcHVibGljIHJlc2V0UGFzc3dvcmRBc3luYyhcclxuICAgIG5ld1Bhc3N3b3JkOiBzdHJpbmcsXHJcbiAgICB0b2tlblJlc2V0OiBzdHJpbmdcclxuICApOiBQcm9taXNlPGJvb2xlYW4+IHtcclxuICAgIHJldHVybiBuZXcgUHJvbWlzZTxib29sZWFuPihhc3luYyAocmVzb2x2ZSwgcmVqZWN0KSA9PiB7XHJcbiAgICAgIHRoaXMuaXNMb2FkaW5nRGF0YS5zaG93KCk7XHJcbiAgICAgIGF3YWl0IHRoaXMudXRpbGl0eVNlcnZpY2Uuc2xlZXBTZWNBc3luYyh0aGlzLnNsZWVwVGltZVNlYyk7XHJcbiAgICAgIGNvbnN0IGNoZWNrVXNlciA9IFVzZXJzQkRNT0NrLlVzZXJzLmZpbmQoXHJcbiAgICAgICAgKHUpID0+IHUudG9rZW5SZXNldFBhc3N3b3JkID09PSB0b2tlblJlc2V0XHJcbiAgICAgICk7XHJcbiAgICAgIHRoaXMuaXNMb2FkaW5nRGF0YS5oaWRlKCk7XHJcbiAgICAgIGlmIChjaGVja1VzZXIpIHtcclxuICAgICAgICBjaGVja1VzZXIucGFzc3dvcmQgPSBuZXdQYXNzd29yZDtcclxuICAgICAgICByZXNvbHZlKHRydWUpO1xyXG4gICAgICAgIHRoaXMubm90aWZpY2F0aW9uc1NlcnZpY2Uuc3VjY2Vzc05vdGlmaWNhdGlvbihcclxuICAgICAgICAgIG5ldyBQcmdTdWNlc3Moe1xyXG4gICAgICAgICAgICB0aXRsZUtleTogJ05ldyBQYXNzd29yZCcsXHJcbiAgICAgICAgICAgIGRldGFpbEtleTogJ1Bhc3N3b3JkIENoYW5nZWQgU3VjY2Vzc2Z1bGx5JyxcclxuICAgICAgICAgIH0pXHJcbiAgICAgICAgKTtcclxuICAgICAgfSBlbHNlIHtcclxuICAgICAgICByZWplY3QoJ1RoZSBwYXNzd29yZCBjaGFuZ2Ugb3BlcmF0aW9uIGhhcyBmYWlsZWQuJyk7XHJcbiAgICAgICAgdGhpcy5ub3RpZmljYXRpb25zU2VydmljZS5lcnJvck5vdGlmaWNhdGlvbihcclxuICAgICAgICAgIG5ldyBQcmdFcnJvcih7XHJcbiAgICAgICAgICAgIHRpdGxlS2V5OiAnUGFzc3dvcmQgRXJyb3IuJyxcclxuICAgICAgICAgICAgZGV0YWlsS2V5OiBgVGhlIHBhc3N3b3JkIGNoYW5nZSBvcGVyYXRpb24gaGFzIGZhaWxlZC5gLFxyXG4gICAgICAgICAgfSlcclxuICAgICAgICApO1xyXG4gICAgICB9XHJcbiAgICB9KTtcclxuICB9XHJcbn1cclxuIl19